/*
 * Remote Laboratory Component Analyzer Part
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * (c) 2014 - 2015 Timothy Pearson
 * Raptor Engineering
 * http://www.raptorengineeringinc.com
 */

#ifndef REMOTELAB_COMPANALYZERPART_H
#define REMOTELAB_COMPANALYZERPART_H

#include <ntqthread.h>
#include <ntqeventloop.h>

#include <tqvariant.h>
#include <tqvaluevector.h>

#include <tdeparts/browserextension.h>
#include <tdeparts/statusbarextension.h>
#include <tdeparts/part.h>
#include <kurl.h>

#include <tqtrla.h>

#include <sevensegment.h>

#define MAXTRACES 255

class TDEAboutData;
using KParts::StatusBarExtension;
class TraceWidget;
class TQSocket;
class TQTimer;
class TQMutex;
class TQRectF;
class CompAnalyzerBase;

namespace RemoteLab
{
	typedef enum CompAnalyzerEventType {
		NoEvent = 			0,
		Initialize = 			1,
		TxRxSyncPoint = 		2,
		StateChanged = 			3,
		ExtendedErrorReceived = 	4,
		ConfigurationDataReceived = 	5,
		ChangeMeasurementSource =	6,
		GetMeasurement = 		7,
		GetMaximumFrequency = 		8,
		GetMinimumFrequency = 		9,
		SetFrequency = 			10,
		MeasurementsReceived = 		11,
		SweepMeasurementsReceived = 	12,
		StartSweep = 			13,
		AbortSweep = 			14,
		OtherEvent = 			15
	} CompAnalyzerEventType;

	typedef enum CompAnalyzerStartupState {
		StartSelectInstrument = 	0,
		StartGetMaximumFrequency =	1,
		StartGetMinimumFrequency =	2,
		StartDone = 			3
	} CompAnalyzerStartupState;

	typedef enum CompAnalyzerPartState {
		Idle = 			0,
		Initializing =		1,
		FreeRunning =		2,
		FrequencySweepWrite = 	3,
		FrequencySweepRead = 	4,
		CommunicationFailure = 	5
	} CompAnalyzerPartState;

	typedef struct CompAnalyzerMeasurement {
		TQ_UINT32 status;
		TQ_UINT32 parameter;
		TQ_UINT32 type;
		double value;
		double frequency;
	} CompAnalyzerMeasurement;

	typedef TQPair<CompAnalyzerEventType, TQVariant> CompAnalyzerEvent;
	typedef TQValueVector<CompAnalyzerEvent> CompAnalyzerEventQueue;
	typedef TQValueList<CompAnalyzerMeasurement> CompAnalyzerMeasurementList;

	typedef TQPair<unsigned int,TQString> AllowedMeasurementInfo;
	typedef TQValueList<AllowedMeasurementInfo> AllowedMeasurementInfoList;

	typedef struct CompAnalyzerInstrumentLimits {
		TQValueList<AllowedMeasurementInfoList> allowedMeasurements;
		double maxFrequency;
		double minFrequency;
	} CompAnalyzerInstrumentLimits;

#ifndef QT_NO_DATASTREAM
	Q_EXPORT TQDataStream &operator<<(TQDataStream &, const CompAnalyzerMeasurement &);
	Q_EXPORT TQDataStream &operator>>(TQDataStream &, CompAnalyzerMeasurement &);
#endif

	class CompAnalyzerWorker : public TQObject
	{
		TQ_OBJECT

		public:
			CompAnalyzerWorker();
			~CompAnalyzerWorker();

		signals:
			void outboundQueueUpdated();

		public slots:
			void run();
			void wake();
			void dataReceived();
	
		public:
			void resetInboundQueue();
			void appendItemToInboundQueue(CompAnalyzerEvent item, bool syncPoint=false);
			bool itemTypeInInboundQueue(CompAnalyzerEventType type);
			bool syncPointActive();
			void lockOutboundQueue();
			void unlockOutboundQueue();
			CompAnalyzerEventQueue* outboundQueue();
			CompAnalyzerPartState currentState();

			CompAnalyzerInstrumentLimits getInstrumentLimits();
			void setNewParameterSourceList(TQValueList<TQ_UINT32>);

			void setSweepStartFrequency(double hz);
			void setSweepEndFrequency(double hz);
			void setSweepStepFrequency(double hz);
			double sweepStartFrequency();
			double sweepEndFrequency();
			double sweepStepFrequency();
			unsigned int sweepStepNumber();

		private:
			CompAnalyzerEventType nextInboundQueueEvent();
			void clearInboundQueueSyncPoint();
			void eraseNextInboundQueueEvent(bool clearSyncPoint=false);
			void setCurrentState(CompAnalyzerPartState state);

		public:
			TDEKerberosClientSocket* m_socket;
			TQMutex* m_instrumentMutex;

		private:
			CompAnalyzerEventQueue m_outboundQueue;
			CompAnalyzerEventQueue m_inboundQueue;
			TQMutex* m_outboundQueueMutex;
			TQMutex* m_inboundQueueMutex;
			TQMutex* m_networkDataMutex;
			TQMutex* m_currentStateMutex;
			TQMutex* m_sweepStepMutex;
			CompAnalyzerPartState m_currentState;
			CompAnalyzerStartupState m_startupState;
			CompAnalyzerEventType m_lastNetworkTransmissionEvent;
			bool m_newData;
			TQValueList<TQ_UINT32> m_sourceList;
			CompAnalyzerInstrumentLimits m_instrumentLimits;

			double m_sweepStart;
			double m_sweepEnd;
			double m_sweepStep;
			double m_sweepCurrentFrequency;
			TQ_UINT32 m_sweepStepNumber;
	};

	class CompAnalyzerPart : public KParts::RemoteInstrumentPart
	{
		Q_OBJECT
	
		public:
			CompAnalyzerPart( QWidget *, const char *, TQObject *, const char *, const TQStringList&);
			~CompAnalyzerPart();
	
			virtual bool openFile() { return false; }	// pure virtual in the base class
			virtual bool closeURL();
			static TDEAboutData *createAboutData();

		signals:
			void wakeWorkerThread();

		public slots:
			virtual bool openURL(const KURL &url);
			void processOutboundQueue();
			void updateZoomWidgetLimits(const TQRectF& zoomRect);
		
		private slots:
			void postInit();
			void processLockouts();
			void connectionFinishedCallback();
			void disconnectFromServerCallback();
			void connectionStatusChangedCallback();
			void setTickerMessage(TQString message);
			void networkTick();
			void networkTimeout();
			void parameterASourceChanged(int);
			void parameterBSourceChanged(int);
			void frequencyInputChanged(double);
			void startSweepClicked();
			void stopSweepClicked();

			void processAutosave();
			void saveWaveforms();
			void saveWaveforms(TQString fileName);
			void recallWaveforms();

			void updateGraticule();

		private:
			TQString parameterMeasurementUnits(TQ_UINT32 parameter);
			TQString parameterNameToMeasurementUnits(TQString name, unsigned int parameter_index);
			void requestNetworkOperation(CompAnalyzerEvent item, bool syncPoint=false);
			void parameterSourceChanged();
			void patWatchDog();
	
		private:
			int m_commHandlerState;
			int m_commHandlerMode;
			int m_commHandlerCommandState;
			TQTimer* m_updateTimeoutTimer;
			bool m_connectionActiveAndValid;
			bool m_instrumentSettingsValid;
			unsigned char m_tickerState;
			CompAnalyzerBase* m_base;
			TQMutex* m_instrumentMutex;
			TQString m_TextToSend;
			TQValueList<AllowedMeasurementInfoList> m_parameterSourceValues;

			TraceWidget* m_traceWidget;
			TQGridLayout* m_traceControlWidgetGrid;

			SensorList m_sensorList;
			TQ_INT16 m_maxNumberOfTraces;
			TQ_INT16 m_hdivs;
			TQ_INT16 m_vdivs;
			TQ_INT32 m_samplesInTrace[MAXTRACES+1];
			bool m_channelActive[MAXTRACES+1];
			TQString m_traceUnits[MAXTRACES+1];

			CompAnalyzerWorker* m_worker;
			TQEventLoopThread* m_workerThread;
	};
}

#endif
