/***************************************************************************
    smb4ksambaoptionshandler  -  This class handles the Samba options.
                             -------------------
    begin                : So Mai 14 2006
    copyright            : (C) 2006-2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KSAMBAOPTIONSHANDLER_H
#define SMB4KSAMBAOPTIONSHANDLER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// TQt includes
#include <tqobject.h>
#include <tqmap.h>

// KDE includes
#include <tdemacros.h>
#include <tdeconfig.h>

// forward declarations
class Smb4KSambaOptionsInfo;
class Smb4KShare;


/**
 * This class belongs to the core classes of Smb4K and handles the global
 * and the custom Samba options.
 *
 * @author Alexander Reinholdt <dustpuppy@mail.berlios.de>
 */

class TDE_EXPORT Smb4KSambaOptionsHandler : public TQObject
{
  TQ_OBJECT
  

  public:
    /**
     * The constructor
     *
     * @param parent            The parent object
     *
     * @param name              The name of this object
     */
    Smb4KSambaOptionsHandler( TQObject *parent = 0, const char *name = 0 );

    /**
     * The destructor
     */
    ~Smb4KSambaOptionsHandler();

    /**
     * Retrieve the list of shares that have custom options defined.
     * You will not only get those options that have different ports etc.
     * defined but also all shares that are to be remounted.
     *
     * @returns the list of all shares that have custom options defined.
     */
    const TQValueList<Smb4KSambaOptionsInfo *> &customOptionsList();

    /**
     * This functions sets the remount flag of the share @p share to true or false.
     * In case the share is not yet in the list of shares that are to be remounted,
     * it will be added. If you set the remount flag to false on an existing entry,
     * it will stay in the list even if no other custom options were defined.
     *
     * @param share             The Smb4KShare object that represents the share
     *
     * @param yes               true if you want the share to be remounted and false
     *                          otherwise
     */
    void remount( Smb4KShare *share, bool yes );

    /**
     * Commit the whole list of shares with custom options to the configuration
     * file. You should call this if you exit the application.
     */
    void sync();

    /**
     * This function returns the line of arguments for the 'smbclient' program.
     * The arguments are spcific to the share that is defined by @p share. You have
     * to provide the name of the shares - as always - in the form //HOST/SHARE.
     *
     * @param share             The name of the share.
     *
     * @returns a list of arguments for use with the 'smbclient' program.
     */
    const TQString smbclientOptions( const TQString &share = TQString() );

    /**
     * This function returns the "global" options for nmblookup, i.e. the domain
     * the client is in, if Kerberos should be used, etc.
     *
     * @param with_broadcast    Return the global broadcast address if defined.
     *
     * @returns a string with the "global" options for nmblookup
     */
    const TQString nmblookupOptions( bool with_broadcast = true );

    /**
     * This function returns the options defined in the global section of the smb.conf
     * file. All option names have been converted to lower case and you can find each
     * entry by providing the option name in lowercase (!) as key.
     *
     * @returns a list of the options defined in smb.conf.
     */
    const TQMap<TQString,TQString> &globalSambaOptions();

    /**
     * This function returns the WINS server the system is using.
     *
     * @returns the name or IP of the WINS server
     */
    const TQString &winsServer();

    /**
     * This enumeration is for use with the netOptions() function. It tells which
     * command to use.
     */
    enum NetCommand { Share, ServerDomain, LookupHost, LookupMaster, Domain };

    /**
     * This function returns the options for the net command.
     *
     * @param command           One of the entries of the NetCommand enumeration.
     *
     * @param networkItem       The name of the network Item. May be empty.
     *
     * @param protocol          Force a certain protocol (rap/rpc/ads) to be used.
     *
     * @returns the list of arguments for the net command or TQString() if an error occurred.
     */
    const TQString netOptions( int command, const TQString &networkItem, const TQString &protocol = TQString() );

    /**
     * This function returns the options for smbmount/mount.cifs under Linux
     * and similar operating systems or for mount_smbfs under FreeBSD.
     *
     * Note: Under Linux etc. this is a comma-separated list which ends with
     * a comma, so remember this when you build up the command line.
     *
     * @param share             The share that is to be mounted.
     */
    const TQString mountOptions( const TQString &share );

    /**
     * Find a network item in the list.
     *
     * Please note that if the host where a share you are probing for
     * is located, a pointer to this *host* item will be returned unless
     * you set @p exactMatch to true in which case NULL is returned! If
     * neither the host nor the share is found, NULL is returned.
     *
     * @param item              The name of the network item to find.
     *
     * @param exactMatch        The name has to match exactly the result that's returned.
     *
     * @returns                 the network item.
     */
    Smb4KSambaOptionsInfo *findItem( const TQString &item, bool exactMatch = false ) { return find_item( item, exactMatch ); }

    /**
     * Add a new Smb4KSambaOptionsInfo object to the list of custom options. If the item already exists,
     * the old options will be replaced by the new ones.
     *
     * @param info              The Smb4KSambaOptionsInfo object
     *
     * @param sync              If true, the list is sync'ed with the config file.
     */
    void addItem( Smb4KSambaOptionsInfo *info, bool sync );

    /**
     * Remove an item from the list.
     *
     * @param name              The name of the item.
     *
     * @param sync              If true, the list is sync'ed with the config file.
     */
    void removeItem( const TQString &name, bool sync );

  private:
    /**
     * The list of network items that have custom options defined.
     */
    TQValueList<Smb4KSambaOptionsInfo *> m_list;

    /**
     * This function reads the options from the config file.
     */
    void read_options();

    /**
     * Write the list of custom shares to the file.
     */
    void write_options();

    /**
     * This function searches a particular network item in the list. If this item is a share
     * and it is not found, @p exactMatch determines if NULL is returned or if the values of
     * the item that matches @p item closest (i.e. the host, or another share that's located
     * on the host). In most cases you want @p exactMatch to be false.
     * Please note: Do not delete the pointer that's returned by this function or you will
     * remove an item from the list!
     *
     * @param item              The name of the network item.
     *
     * @param exactMatch        The name has to match exactly the result that's returned.
     *
     * @returns                 The Smb4KSambaOptionsInfo object associated with the network item.
     */
    Smb4KSambaOptionsInfo *find_item( const TQString &item, bool exactMatch = false );

    /**
     * This function reads the entries of the global section of Samba's configuration
     * file smb.conf and puts them into a map.
     */
    void read_smb_conf();

    /**
     * This map carries the options defined in the [global] section of Samba's configuration
     * file smb.conf. You can access a certain value by providing the lower case option name
     * as key.
     */
    TQMap<TQString,TQString> m_samba_options;

    /**
     * The WINS server
     */
    TQString m_wins_server;
};


#endif
