/*
 *  Copyright (c) 2004 Cyrille Berger <cberger@cberger.net>
 *                2004 Sven Langkamp <longamp@reallygood.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kis_autogradient.h"

#include <tqpainter.h>
#include <tqcombobox.h>

#include <kcolorbutton.h>
#include <knuminput.h>
#include "kis_int_spinbox.h"
#include "kis_gradient.h"
#include "kis_autogradient_resource.h"

#include "kis_gradient_slider_widget.h"

/****************************** KisAutogradient ******************************/

KisAutogradient::KisAutogradient(TQWidget *parent, const char* name, const TQString& caption) : KisWdgAutogradient(parent, name)
{
    setCaption(caption);
    m_autogradientResource = new KisAutogradientResource();
    m_autogradientResource->createSegment( INTERP_LINEAR, COLOR_INTERP_RGB, 0.0, 1.0, 0.5, TQt::black, TQt::white );
    connect(gradientSlider, TQ_SIGNAL( sigSelectedSegment( KisGradientSegment* ) ), TQ_SLOT( slotSelectedSegment(KisGradientSegment*) ));
    connect(gradientSlider, TQ_SIGNAL( sigChangedSegment(KisGradientSegment*) ), TQ_SLOT( slotChangedSegment(KisGradientSegment*) ));
    gradientSlider->setGradientResource( m_autogradientResource );
    connect(comboBoxColorInterpolationType, TQ_SIGNAL( activated(int) ), TQ_SLOT( slotChangedColorInterpolation(int) ));
    connect(comboBoxInterpolationType, TQ_SIGNAL( activated(int) ), TQ_SLOT( slotChangedInterpolation(int) ));
    connect(leftColorButton, TQ_SIGNAL( changed(const TQColor&) ), TQ_SLOT( slotChangedLeftColor(const TQColor&) ));
    connect(rightColorButton, TQ_SIGNAL( changed(const TQColor&) ), TQ_SLOT( slotChangedRightColor(const TQColor&) ));

//     intNumInputLeftOpacity->setRange( 0, 100, false);
    connect(intNumInputLeftOpacity, TQ_SIGNAL( valueChanged(int) ), TQ_SLOT( slotChangedLeftOpacity(int) ));
//     intNumInputRightOpacity->setRange( 0, 100, false);
    connect(intNumInputRightOpacity, TQ_SIGNAL( valueChanged(int) ), TQ_SLOT( slotChangedRightOpacity(int) ));

}

void KisAutogradient::activate()
{
    paramChanged();
}

void KisAutogradient::slotSelectedSegment(KisGradientSegment* segment)
{
    leftColorButton->setColor( segment->startColor().color() );
    rightColorButton->setColor( segment->endColor().color() );
    comboBoxColorInterpolationType->setCurrentItem( segment->colorInterpolation() );
    comboBoxInterpolationType->setCurrentItem( segment->interpolation() );

    int leftOpacity = tqRound(segment->startColor().alpha() * 100);
    intNumInputLeftOpacity->setValue( leftOpacity );

    int rightOpacity = tqRound(segment->endColor().alpha() * 100);
    intNumInputRightOpacity->setValue( rightOpacity );

    paramChanged();
}

void KisAutogradient::slotChangedSegment(KisGradientSegment*)
{
    paramChanged();
}

void KisAutogradient::slotChangedInterpolation(int type)
{
    KisGradientSegment* segment = gradientSlider->selectedSegment();
    if(segment)
        segment->setInterpolation( type );
    gradientSlider->update();

    paramChanged();
}

void KisAutogradient::slotChangedColorInterpolation(int type)
{
    KisGradientSegment* segment = gradientSlider->selectedSegment();
    if(segment)
        segment->setColorInterpolation( type );
    gradientSlider->update();

    paramChanged();
}

void KisAutogradient::slotChangedLeftColor( const TQColor& color)
{
    KisGradientSegment* segment = gradientSlider->selectedSegment();
    if(segment)
        segment->setStartColor( Color( color, segment->startColor().alpha() ) );
    gradientSlider->update();

    paramChanged();
}

void KisAutogradient::slotChangedRightColor( const TQColor& color)
{
    KisGradientSegment* segment = gradientSlider->selectedSegment();
    if(segment)
        segment->setEndColor( Color( color, segment->endColor().alpha() ) );
    gradientSlider->repaint();

    paramChanged();
}

void KisAutogradient::slotChangedLeftOpacity( int value )
{
    KisGradientSegment* segment = gradientSlider->selectedSegment();
    if(segment)
        segment->setStartColor( Color( segment->startColor().color(), (double)value / 100 ) );
    gradientSlider->repaint(false);

    paramChanged();
}

void KisAutogradient::slotChangedRightOpacity( int value )
{
    KisGradientSegment* segment = gradientSlider->selectedSegment();
    if(segment)
        segment->setEndColor( Color( segment->endColor().color(), (double)value / 100 ) );
    gradientSlider->repaint(false);

    paramChanged();
}

void KisAutogradient::paramChanged()
{
    m_autogradientResource->updatePreview ();
    emit activatedResource( m_autogradientResource );
}

#include "kis_autogradient.moc"
