/**********************************************************************
** Copyright (C) 2000 Trolltech AS.  All rights reserved.
**
** This file is part of TQt Designer.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#ifndef PROPERTYEDITOR_H
#define PROPERTYEDITOR_H

#include <tqfeatures.h>
#include <tqvariant.h>
#include <tqlistview.h>
#include <tqptrlist.h>
#include <tqguardedptr.h>
#include <tqtabwidget.h>
#include <tqmodules.h>
#include "hierarchyview.h"

class PropertyList;
class PropertyEditor;
class TQPainter;
class TQColorGroup;
class TQComboBox;
class TQLineEdit;
class TQPushButton;
class TQHBox;
class TQSpinBox;
class TQLabel;
class FormWindow;
class TQCloseEvent;
class TQResizeEvent;
class PropertyWhatsThis;
class TQDateEdit;
class TQTimeEdit;
class TQDateTimeEdit;

class PropertyItem : public TQListViewItem
{
public:
    PropertyItem( PropertyList *l, PropertyItem *after, PropertyItem *prop, const TQString &propName );
    ~PropertyItem();

    void paintCell( TQPainter *p, const TQColorGroup &cg, int column, int width, int align );
    void paintBranches( TQPainter * p, const TQColorGroup & cg,
			int w, int y, int h );
    void paintFocus( TQPainter *p, const TQColorGroup &cg, const TQRect &r );

    virtual bool hasSubItems() const;
    virtual void createChildren();
    virtual void initChildren();

    bool isOpen() const;
    void setOpen( bool b );

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );
    TQVariant value() const;
    TQString name() const;
    virtual void notifyValueChange();

    virtual void setChanged( bool b, bool updateDb = true );
    bool isChanged() const;

    virtual void placeEditor( TQWidget *w );

    virtual PropertyItem *propertyParent() const;
    virtual void childValueChanged( PropertyItem *child );

    void addChild( PropertyItem *i );
    int childCount() const;
    PropertyItem *child( int i ) const;

    virtual bool hasCustomContents() const;
    virtual void drawCustomContents( TQPainter *p, const TQRect &r );

    void updateBackColor();

    void setup() { TQListViewItem::setup(); setHeight( TQListViewItem::height() + 2 ); }

    virtual TQString currentItem() const;
    virtual int currentIntItem() const;
    virtual void setCurrentItem( const TQString &s );
    virtual void setCurrentItem( int i );
    virtual int currentIntItemFromObject() const;
    virtual TQString currentItemFromObject() const;

    void setFocus( TQWidget *w );

    virtual void toggle();
    void setText( int col, const TQString &txt );

protected:
    PropertyList *listview;
    TQVariant val;

private:
    TQColor backgroundColor();
    void createResetButton();
    void updateResetButtonState();

private:
    bool open, changed;
    PropertyItem *property;
    TQString propertyName;
    TQPtrList<PropertyItem> children;
    TQColor backColor;
    TQPushButton *resetButton;

};

class PropertyTextItem : public TQObject,
			 public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyTextItem( PropertyList *l, PropertyItem *after, PropertyItem *prop,
		      const TQString &propName, bool comment, bool multiLine, bool ascii = false, bool a = false );
    ~PropertyTextItem();

    virtual void createChildren();
    virtual void initChildren();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );
    virtual bool hasSubItems() const;
    virtual void childValueChanged( PropertyItem *child );

    virtual void setChanged( bool b, bool updateDb = true );

private slots:
    void setValue();
    void getText();

private:
    TQLineEdit *lined();
    TQGuardedPtr<TQLineEdit> lin;
    TQGuardedPtr<TQHBox> box;
    TQPushButton *button;
    bool withComment, hasMultiLines, asciiOnly, accel;

};


class PropertyBoolItem : public TQObject,
			 public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyBoolItem( PropertyList *l, PropertyItem *after, PropertyItem *prop, const TQString &propName );
    ~PropertyBoolItem();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );
    virtual void toggle();

private slots:
    void setValue();

private:
    TQComboBox *combo();
    TQGuardedPtr<TQComboBox> comb;

};

class PropertyIntItem : public TQObject,
			public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyIntItem( PropertyList *l, PropertyItem *after, PropertyItem *prop,
		     const TQString &propName, bool s );
    ~PropertyIntItem();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );

private slots:
    void setValue();

private:
    TQSpinBox *spinBox();
    TQGuardedPtr<TQSpinBox> spinBx;
    bool signedValue;

};

class PropertyDoubleItem : public TQObject,
			public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyDoubleItem( PropertyList *l, PropertyItem *after, PropertyItem *prop,
		     const TQString &propName );
    ~PropertyDoubleItem();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );

private slots:
    void setValue();

private:
    TQLineEdit *lined();
    TQGuardedPtr<TQLineEdit> lin;
};

class PropertyListItem : public TQObject,
			 public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyListItem( PropertyList *l, PropertyItem *after, PropertyItem *prop,
		      const TQString &propName, bool editable );
    ~PropertyListItem();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );

    TQString currentItem() const;
    int currentIntItem() const;
    void setCurrentItem( const TQString &s );
    void setCurrentItem( int i );
    int currentIntItemFromObject() const;
    TQString currentItemFromObject() const;
    void addItem( const TQString &s );

private slots:
    void setValue();

private:
    TQComboBox *combo();
    TQGuardedPtr<TQComboBox> comb;
    int oldInt;
    bool editable;
    TQString oldString;

};

class PropertyFontItem : public TQObject,
			 public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyFontItem( PropertyList *l, PropertyItem *after, PropertyItem *prop, const TQString &propName );
    ~PropertyFontItem();

    virtual void createChildren();
    virtual void initChildren();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );
    virtual bool hasSubItems() const;
    virtual void childValueChanged( PropertyItem *child );

private slots:
    void getFont();

private:
    TQGuardedPtr<TQLineEdit> lined;
    TQGuardedPtr<TQPushButton> button;
    TQGuardedPtr<TQHBox> box;

};

class PropertyCoordItem : public TQObject,
			  public PropertyItem
{
    TQ_OBJECT
  

public:
    enum Type { Rect, Size, Point };

    PropertyCoordItem( PropertyList *l, PropertyItem *after, PropertyItem *prop,
		       const TQString &propName, Type t );
    ~PropertyCoordItem();

    virtual void createChildren();
    virtual void initChildren();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );
    virtual bool hasSubItems() const;
    virtual void childValueChanged( PropertyItem *child );

private:
    TQLineEdit *lined();
    TQGuardedPtr<TQLineEdit> lin;
    Type typ;

};

class PropertyColorItem : public TQObject,
			  public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyColorItem( PropertyList *l, PropertyItem *after, PropertyItem *prop,
		       const TQString &propName, bool children );
    ~PropertyColorItem();

    virtual void createChildren();
    virtual void initChildren();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );
    virtual bool hasSubItems() const;
    virtual void childValueChanged( PropertyItem *child );

    virtual bool hasCustomContents() const;
    virtual void drawCustomContents( TQPainter *p, const TQRect &r );

private slots:
    void getColor();

private:
    TQGuardedPtr<TQHBox> box;
    TQGuardedPtr<TQFrame> colorPrev;
    TQGuardedPtr<TQPushButton> button;
    bool withChildren;

};

class PropertyPixmapItem : public TQObject,
			   public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyPixmapItem( PropertyList *l, PropertyItem *after, PropertyItem *prop,
			const TQString &propName, bool isIconSet = false );
    ~PropertyPixmapItem();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );

    virtual bool hasCustomContents() const;
    virtual void drawCustomContents( TQPainter *p, const TQRect &r );

private slots:
    void getPixmap();

private:
    TQGuardedPtr<TQHBox> box;
    TQGuardedPtr<TQLabel> pixPrev;
    TQPushButton *button;
    bool iconSet;

};


class PropertySizePolicyItem : public TQObject,
			  public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertySizePolicyItem( PropertyList *l, PropertyItem *after, PropertyItem *prop,
			    const TQString &propName );
    ~PropertySizePolicyItem();

    virtual void createChildren();
    virtual void initChildren();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );
    virtual bool hasSubItems() const;
    virtual void childValueChanged( PropertyItem *child );

private:
    TQLineEdit *lined();
    TQGuardedPtr<TQLineEdit> lin;

};

class PropertyPaletteItem : public TQObject,
			    public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyPaletteItem( PropertyList *l, PropertyItem *after, PropertyItem *prop,
			const TQString &propName );
    ~PropertyPaletteItem();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );

    virtual bool hasCustomContents() const;
    virtual void drawCustomContents( TQPainter *p, const TQRect &r );

private slots:
    void getPalette();

private:
    TQGuardedPtr<TQHBox> box;
    TQGuardedPtr<TQLabel> palettePrev;
    TQGuardedPtr<TQPushButton> button;

};

class PropertyCursorItem : public TQObject,
			   public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyCursorItem( PropertyList *l, PropertyItem *after, PropertyItem *prop,
			const TQString &propName );
    ~PropertyCursorItem();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );

private slots:
    void setValue();

private:
    TQComboBox *combo();
    TQGuardedPtr<TQComboBox> comb;

};


class PropertyDatabaseItem : public TQObject,
			     public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyDatabaseItem( PropertyList *l, PropertyItem *after, PropertyItem *prop, const TQString &propName, bool wField );
    ~PropertyDatabaseItem();

    virtual void createChildren();
    virtual void initChildren();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );
    virtual bool hasSubItems() const;
    virtual void childValueChanged( PropertyItem *child );

private:
    TQGuardedPtr<TQLineEdit> lined;
    TQGuardedPtr<TQPushButton> button;
    TQGuardedPtr<TQHBox> box;
    bool withField;

};

class PropertyList : public TQListView
{
    TQ_OBJECT
  

public:
    PropertyList( PropertyEditor *e );

    virtual void setupProperties();

    virtual void setCurrentItem( TQListViewItem *i );
    virtual void valueChanged( PropertyItem *i );
    virtual void refetchData();
    virtual void setPropertyValue( PropertyItem *i );
    virtual void setCurrentProperty( const TQString &n );

    PropertyEditor *propertyEditor() const;
    TQString whatsThisAt( const TQPoint &p );
    void showCurrentWhatsThis();

public slots:
    void updateEditorSize();
    void resetProperty();
    void toggleSort();

private slots:
    void itemPressed( TQListViewItem *i, const TQPoint &p, int c );
    void toggleOpen( TQListViewItem *i );
    bool eventFilter( TQObject *o, TQEvent *e );

protected:
    void resizeEvent( TQResizeEvent *e );
    void paintEmptyArea( TQPainter *p, const TQRect &r );
    bool addPropertyItem( PropertyItem *&item, const TQCString &name, TQVariant::Type t );

    void viewportDragEnterEvent( TQDragEnterEvent * );
    void viewportDragMoveEvent ( TQDragMoveEvent * );
    void viewportDropEvent ( TQDropEvent * );

protected:
    PropertyEditor *editor;

private:
    void readPropertyDocs();
    TQString whatsThisText( TQListViewItem *i );

private:
    PropertyListItem* pressItem;
    TQPoint pressPos;
    bool mousePressed;
    bool showSorted;
    TQMap<TQString, TQString> propertyDocs;
    PropertyWhatsThis *whatsThis;

};

class PropertyEditor : public TQTabWidget
{
    TQ_OBJECT
  

public:
    PropertyEditor( TQWidget *parent );

    TQObject *widget() const;

    void clear();
    void setup();

    void emitWidgetChanged();
    void refetchData();

    void closed( FormWindow *w );

    PropertyList *propertyList() const;
    FormWindow *formWindow() const;
#ifndef KOMMANDER
//    EventList *eventList() const;
#endif

    TQString currentProperty() const;
    TQString classOfCurrentProperty() const;
    TQMetaObject* metaObjectOfCurrentProperty() const;

    void resetFocus();

signals:
    void hidden();

public slots:
    void setWidget( TQObject *w, FormWindow *fw );

protected:
    void closeEvent( TQCloseEvent *e );

private:
    TQObject *wid;
    PropertyList *listview;
#ifndef KOMMANDER
    EventList *eList;
#endif
    FormWindow *formwindow;

};

class PropertyDateItem : public TQObject,
			 public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyDateItem( PropertyList *l, PropertyItem *after, PropertyItem *prop, const TQString &propName );
    ~PropertyDateItem();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );

private slots:
    void setValue();

private:
    TQDateEdit *lined();
    TQGuardedPtr<TQDateEdit> lin;

};

class PropertyTimeItem : public TQObject,
			 public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyTimeItem( PropertyList *l, PropertyItem *after, PropertyItem *prop, const TQString &propName );
    ~PropertyTimeItem();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );

private slots:
    void setValue();

private:
    TQTimeEdit *lined();
    TQGuardedPtr<TQTimeEdit> lin;

};

class PropertyDateTimeItem : public TQObject,
			 public PropertyItem
{
    TQ_OBJECT
  

public:
    PropertyDateTimeItem( PropertyList *l, PropertyItem *after, PropertyItem *prop, const TQString &propName );
    ~PropertyDateTimeItem();

    virtual void showEditor();
    virtual void hideEditor();

    virtual void setValue( const TQVariant &v );

private slots:
    void setValue();

private:
    TQDateTimeEdit *lined();
    TQGuardedPtr<TQDateTimeEdit> lin;

};

#endif
