/***************************************************************************
*   Copyright (C) 2004-2009 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <tqlineedit.h>
#include <tqlayout.h>
#include <tqlistbox.h>
#include <tqtimer.h>
#include <tqtooltip.h>
#include <tqlabel.h>

#include <kcombobox.h>
#include <kpushbutton.h>
#include <kcombobox.h>
#include <tdelocale.h>
#include <kdialog.h>
#include <kiconloader.h>
#include <tdepopupmenu.h>

#include "settings.h"
#include "searchbar.h"

namespace KBibTeX
{

    SearchBar::SearchBar( TQWidget *parent, const char *name )
            : TQWidget( parent, name )
    {
        m_timerInput = new TQTimer( this );
        connect( m_timerInput, TQ_SIGNAL( timeout() ), TQ_SLOT( slotTimeout() ) );

        setupGUI();
    }

    SearchBar::~SearchBar()
    {
        delete m_timerInput;
    }

    void SearchBar::setFactory( KXMLGUIFactory *factory, KXMLGUIClient *client )
    {
        TDEPopupMenu * menu = static_cast<TDEPopupMenu*>( factory -> container( "popup_newelements", client ) );
        Settings * settings = Settings::self();
        m_pushButtonAddElement->setPopup( menu );
        m_comboboxFilter->setHistoryItems( settings->editing_FilterHistory );
    }

    void SearchBar::restoreState()
    {
        Settings * settings = Settings::self();
        if ( settings->editing_UseSpecialFont )
            m_comboboxFilter->setFont( settings->editing_SpecialFont );
        else
            m_comboboxFilter->setFont( TDEGlobalSettings::generalFont() );
    }

    void SearchBar::setSearch( const TQString&text, BibTeX::Element::FilterType filterType, BibTeX::EntryField::FieldType fieldType )
    {
        m_comboboxFilter->setCurrentText( text );
        switch ( filterType )
        {
        case BibTeX::Element::ftExact: m_comboboxFilterType->setCurrentItem( 0 ); break;
        case BibTeX::Element::ftEveryWord: m_comboboxFilterType->setCurrentItem( 1 ); break;
        case BibTeX::Element::ftAnyWord: m_comboboxFilterType->setCurrentItem( 2 ); break;
        }
        m_comboboxRestrictTo->setCurrentItem(( int ) fieldType + 1 );
    }

    void SearchBar::setupGUI()
    {
        TQBoxLayout * layout = new TQHBoxLayout( this, 3 /* KDialog::marginHint()*/, KDialog::spacingHint() );
        TDEIconLoader iconLoader = TDEIconLoader( "kbibtex" );

        m_pushButtonAddElement = new KPushButton( this );
        m_pushButtonAddElement->setIconSet( TQIconSet( BarIcon( "add" ) ) );
        layout->addWidget( m_pushButtonAddElement );
        TQToolTip::add( m_pushButtonAddElement, i18n( "Add a new BibTeX entry, comment or macro to this file" ) );

        m_pushButtonSearchOnlineDatabases = new KPushButton( this );
        m_pushButtonSearchOnlineDatabases->setIconSet( TQIconSet( BarIcon( "network" ) ) );
        layout->addWidget( m_pushButtonSearchOnlineDatabases );
        TQToolTip::add( m_pushButtonSearchOnlineDatabases, i18n( "Add a new BibTeX entry from an online database" ) );
        connect( m_pushButtonSearchOnlineDatabases, TQ_SIGNAL( clicked() ), this, TQ_SIGNAL( onlineSearch() ) );

        layout->insertSpacing( 2, KDialog::spacingHint() );

        m_pushButtonClearSearchText = new KPushButton( this );
        m_pushButtonClearSearchText->setIconSet( TQIconSet( BarIcon( "locationbar_erase" ) ) );
        layout->addWidget( m_pushButtonClearSearchText );
        TQToolTip::add( m_pushButtonClearSearchText, i18n( "Erase current search pattern" ) );
        m_pushButtonClearSearchText->setSizePolicy( TQSizePolicy::Preferred, TQSizePolicy::Preferred );

        TQLabel *label = new TQLabel( i18n( "&Search:" ), this );
        layout->addWidget( label );

        m_comboboxFilter = new KHistoryCombo( TRUE, this, "search_combobox" );
        layout->addWidget( m_comboboxFilter );
        label->setBuddy( m_comboboxFilter );
        m_comboboxFilter->setSizePolicy( TQSizePolicy::MinimumExpanding, TQSizePolicy::Preferred );
        m_comboboxFilter->setMaxCount( 256 );

        m_comboboxFilterType = new KComboBox( FALSE, this );
        m_comboboxFilterType->setSizePolicy( TQSizePolicy::Minimum, TQSizePolicy::Preferred );
        layout->addWidget( m_comboboxFilterType );

        m_comboboxFilterType->insertItem( i18n( "Exact" ) );
        m_comboboxFilterType->insertItem( i18n( "Every word" ) );
        m_comboboxFilterType->insertItem( i18n( "Any word" ) );
        m_comboboxFilterType->setCurrentItem( 1 );

        label = new TQLabel( i18n( "Restrict to:" ), this );
        layout->addWidget( label );
        m_comboboxRestrictTo = new KComboBox( FALSE, this );
        m_comboboxRestrictTo->setSizePolicy( TQSizePolicy::Minimum, TQSizePolicy::Preferred );
        layout->addWidget( m_comboboxRestrictTo );
        label->setBuddy( m_comboboxRestrictTo );

        m_comboboxRestrictTo->insertItem( i18n( "All fields" ) );
        for ( int i = ( int ) BibTeX::EntryField::ftAbstract; i <= ( int ) BibTeX::EntryField::ftYear; i++ )
        {
            BibTeX::EntryField::FieldType fieldType = ( BibTeX::EntryField::FieldType ) i;
            m_comboboxRestrictTo->insertItem( Settings::fieldTypeToI18NString( fieldType ) );
        }

        connect( m_comboboxFilter->lineEdit(), TQ_SIGNAL( textChanged( const TQString & ) ), this, TQ_SLOT( slotKeyPressed() ) );
        connect( m_comboboxFilter, TQ_SIGNAL( activated( const TQString& ) ), m_comboboxFilter, TQ_SLOT( addToHistory( const TQString& ) ) );
        connect( m_pushButtonClearSearchText, TQ_SIGNAL( clicked() ), this, TQ_SLOT( slotClear() ) );
        connect( m_comboboxFilterType, TQ_SIGNAL( textChanged( const TQString& ) ), this, TQ_SLOT( slotAnnounceDoSearch() ) );
        connect( m_comboboxFilter, TQ_SIGNAL( textChanged( const TQString& ) ), this, TQ_SLOT( slotAnnounceDoSearch() ) );
        connect( m_comboboxFilter->lineEdit(), TQ_SIGNAL( returnPressed() ), this, TQ_SLOT( slotAnnounceDoSearch() ) );
        connect( m_comboboxFilterType, TQ_SIGNAL( activated( int ) ), this, TQ_SLOT( slotTimeout() ) );
        connect( m_comboboxRestrictTo, TQ_SIGNAL( activated( int ) ), this, TQ_SLOT( slotTimeout() ) );

        setSizePolicy( TQSizePolicy::MinimumExpanding, TQSizePolicy::Preferred );
    }

    void SearchBar::slotClear()
    {
        m_comboboxFilter->lineEdit() ->clear();
        m_comboboxRestrictTo->setCurrentItem( 0 );
        m_comboboxFilterType->setCurrentItem( 1 );
        m_comboboxFilter->setFocus();
    }

    void SearchBar::slotTimeout()
    {
        BibTeX::EntryField::FieldType fieldType = BibTeX::EntryField::ftUnknown;
        if ( m_comboboxRestrictTo->currentItem() > 0 )
            fieldType = ( BibTeX::EntryField::FieldType )( m_comboboxRestrictTo->currentItem() - 1 + ( int ) BibTeX::EntryField::ftAbstract );

        Settings * settings = Settings::self();
        settings->editing_FilterHistory = m_comboboxFilter->historyItems();

        BibTeX::Element::FilterType filterType = BibTeX::Element::ftExact;
        if ( m_comboboxFilterType->currentItem() == 1 ) filterType = BibTeX::Element::ftEveryWord;
        else if ( m_comboboxFilterType->currentItem() == 2 ) filterType = BibTeX::Element::ftAnyWord;

        emit doSearch( m_comboboxFilter->currentText(), filterType, fieldType );
    }

    void SearchBar::slotAnnounceDoSearch()
    {
        if ( m_timerInput->isActive() )
            m_timerInput->stop();

        m_timerInput->start( 0, true );
    }

    void SearchBar::slotKeyPressed()
    {
        Settings * settings = Settings::self();
        if ( settings->editing_SearchBarClearField )
            m_comboboxRestrictTo->setCurrentItem( 0 );
    }

}
#include "searchbar.moc"
