/***************************************************************************
*   Copyright (C) 2003 by                                                 *
*   Unai Garro (ugarro@users.sourceforge.net)                             *
*   Cyril Bosselut (bosselut@b1project.com)                               *
*   Jason Kivlighn (jkivlighn@gmail.com)                                  *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "resizerecipedialog.h"

#include <cmath>

#include <tqvbox.h>
#include <tqvariant.h>
#include <tqbuttongroup.h>
#include <tqframe.h>
#include <tqlabel.h>
#include <knuminput.h>
#include <klineedit.h>
#include <tqradiobutton.h>
#include <tqlayout.h>
#include <tqtooltip.h>
#include <tqwhatsthis.h>
#include <tqimage.h>
#include <tqpixmap.h>

#include <tdelocale.h>
#include <tdemessagebox.h>
#include <kdebug.h>

#include "datablocks/recipe.h"
#include "widgets/fractioninput.h"

#define FACTOR_RADIO_BUTTON 0
#define SERVINGS_RADIO_BUTTON 1

ResizeRecipeDialog::ResizeRecipeDialog( TQWidget *parent, Recipe *recipe )
		: KDialogBase( parent, "ResizeRecipeDialog", true, i18n( "Resize Recipe" ),
		    KDialogBase::Ok | KDialogBase::Cancel, KDialogBase::Ok ),
		m_recipe( recipe )
{
	TQVBox *page = makeVBoxMainWidget();

	buttonGroup = new TQButtonGroup( page );
	buttonGroup->setSizePolicy( TQSizePolicy( ( TQSizePolicy::SizeType ) 5, ( TQSizePolicy::SizeType ) 7, 0, 1, buttonGroup->sizePolicy().hasHeightForWidth() ) );
	buttonGroup->setLineWidth( 0 );
	buttonGroup->setColumnLayout( 0, TQt::Vertical );
	buttonGroup->layout() ->setSpacing( 6 );
	buttonGroup->layout() ->setMargin( 11 );
	buttonGroupLayout = new TQVBoxLayout( buttonGroup->layout() );
	buttonGroupLayout->setAlignment( TQt::AlignTop );

	yieldRadioButton = new TQRadioButton( buttonGroup );
	buttonGroup->insert( yieldRadioButton, SERVINGS_RADIO_BUTTON );
	buttonGroupLayout->addWidget( yieldRadioButton );

	yieldFrame = new TQFrame( buttonGroup );
	yieldFrame->setFrameShape( TQFrame::Box );
	yieldFrame->setFrameShadow( TQFrame::Sunken );
	yieldFrame->setLineWidth( 1 );
	yieldFrameLayout = new TQGridLayout( yieldFrame, 1, 1, 11, 6 );

	currentYieldLabel = new TQLabel( yieldFrame );

	yieldFrameLayout->addWidget( currentYieldLabel, 0, 0 );

	newYieldLabel = new TQLabel( yieldFrame );

	yieldFrameLayout->addMultiCellWidget( newYieldLabel, 1, 1, 0, 1 );

	currentYieldInput = new KLineEdit( yieldFrame );
	currentYieldInput->setReadOnly( TRUE );
	currentYieldInput->setAlignment( TQt::AlignRight );
	yieldFrameLayout->addMultiCellWidget( currentYieldInput, 0, 0, 1, 2 );

	newYieldInput = new FractionInput( yieldFrame );
	yieldFrameLayout->addWidget( newYieldInput, 1, 2 );

	buttonGroupLayout->addWidget( yieldFrame );

	factorRadioButton = new TQRadioButton( buttonGroup );
	buttonGroup->insert( factorRadioButton, FACTOR_RADIO_BUTTON );
	buttonGroupLayout->addWidget( factorRadioButton );

	factorFrame = new TQFrame( buttonGroup );
	factorFrame->setSizePolicy( TQSizePolicy( ( TQSizePolicy::SizeType ) 7, ( TQSizePolicy::SizeType ) 5, 1, 0, factorFrame->sizePolicy().hasHeightForWidth() ) );
	factorFrame->setFrameShape( TQFrame::Box );
	factorFrame->setFrameShadow( TQFrame::Sunken );
	factorFrame->setLineWidth( 1 );
	factorFrameLayout = new TQHBoxLayout( factorFrame, 11, 6 );

	factorLabel = new TQLabel( factorFrame );
	factorFrameLayout->addWidget( factorLabel );

	factorInput = new FractionInput( factorFrame );
	factorInput->setSizePolicy( TQSizePolicy( ( TQSizePolicy::SizeType ) 7, ( TQSizePolicy::SizeType ) 5, 0, 0, factorInput->sizePolicy().hasHeightForWidth() ) );
	factorFrameLayout->addWidget( factorInput );
	buttonGroupLayout->addWidget( factorFrame );

	languageChange();


	newYieldInput->setValue( m_recipe->yield.amount, 0 ); //Ignore the range info, it doesn't work in this context
	currentYieldInput->setText( m_recipe->yield.toString() );

	if ( recipe->yield.amount_offset > 0 ) {
		yieldRadioButton->setEnabled(false);
		buttonGroup->setButton( FACTOR_RADIO_BUTTON );
		activateCurrentOption( FACTOR_RADIO_BUTTON );
	}
	else {
		buttonGroup->setButton( SERVINGS_RADIO_BUTTON );
		activateCurrentOption( SERVINGS_RADIO_BUTTON );
	}

	// signals and slots connections
	connect( buttonGroup, TQ_SIGNAL( clicked( int ) ), this, TQ_SLOT( activateCurrentOption( int ) ) );
}

void ResizeRecipeDialog::languageChange()
{
	buttonGroup->setTitle( TQString::null );
	yieldRadioButton->setText( i18n( "Scale by yield" ) );
	newYieldLabel->setText( i18n( "New yield:" ) );
	currentYieldLabel->setText( i18n( "Current yield:" ) );
	factorRadioButton->setText( i18n( "Scale by factor" ) );
	factorLabel->setText( i18n( "Factor (i.e. 1/2 to half, 3 to triple):" ) );
}

void ResizeRecipeDialog::activateCurrentOption( int button_id )
{
	switch ( button_id ) {
	case SERVINGS_RADIO_BUTTON:
		factorFrame->setEnabled( false );
		yieldFrame->setEnabled( true );
		break;
	case FACTOR_RADIO_BUTTON:
		factorFrame->setEnabled( true );
		yieldFrame->setEnabled( false );
		break;
	default:
		break;
	}
}

void ResizeRecipeDialog::accept()
{
	if ( currentYieldInput->text().toInt() == 0 )
		KMessageBox::error( this, i18n( "Unable to scale a recipe with zero yield" ) );
	else if ( buttonGroup->selected() == yieldRadioButton ) {
		if ( newYieldInput->isInputValid() ) {
			double new_yield = newYieldInput->value().toDouble();
			double current_yield = MixedNumber::fromString(currentYieldInput->text()).toDouble();

			resizeRecipe( new_yield / current_yield );
		}
		else {
			KMessageBox::error( this, i18n( "Invalid input" ) );
			newYieldInput->selectAll();
			return;
		}
	}
	else {
		if ( factorInput->isInputValid() && factorInput->value() > 0 )
			resizeRecipe( factorInput->value().toDouble() );
		else {
			KMessageBox::error( this, i18n( "Invalid input" ) );
			factorInput->selectAll();
			return ;
		}
	}

	TQDialog::accept();
}

//TODO YIELD: handle ranges
void ResizeRecipeDialog::resizeRecipe( double factor )
{
	m_recipe->yield.amount = MixedNumber::fromString(currentYieldInput->text()).toDouble() * factor;

	for ( IngredientList::iterator ing_it = m_recipe->ingList.begin(); ing_it != m_recipe->ingList.end(); ++ing_it ) {
		( *ing_it ).amount = ( *ing_it ).amount * factor;
		( *ing_it ).amount_offset = ( *ing_it ).amount_offset * factor;
	}
}

#include "resizerecipedialog.moc"
