/***************************************************************************
 *   Copyright (C) 2012 by Timothy Pearson                                 *
 *   kb9vqf@pearsoncomputing.net                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <unistd.h>

#include <tqpushbutton.h>
#include <tqlabel.h>
#include <tqstring.h>
#include <tqstringlist.h>
#include <tqfile.h>
#include <tqtimer.h>
#include <tqcursor.h>
#include <tqspinbox.h>
#include <tqcheckbox.h>
#include <tqradiobutton.h>

#include <ksimpleconfig.h>
#include <tdeglobal.h>
#include <tdeglobalsettings.h>
#include <kstandarddirs.h>
#include <tdelocale.h>
#include <tdeapplication.h>
#include <tdelistview.h>
#include <krun.h>
#include <tdemessagebox.h>
#include <tdeconfig.h>
#include <knuminput.h>
#include <klineedit.h>
#include <ktextedit.h>
#include <kpassdlg.h>
#include <kurlrequester.h>
#include <ksslcertificate.h>

#include <stdlib.h>

#include <kdebug.h>

#include "realmintropage.h"
#include "realmconfigpage.h"
#include "certconfigpage.h"
#include "realmfinishpage.h"

#include "primaryrealmwizard.h"
#include "primaryrealmwizard.moc"

PrimaryRealmWizard::PrimaryRealmWizard(LDAPController* controller, TQString fqdn, LDAPCertConfig certinfo, TQWidget *parent, const char *name)
	: KWizard(parent, name, true), m_controller(controller), m_fqdn(fqdn), m_certconfig(certinfo) {

	setCaption(i18n("LDAP Realm Wizard"));

	intropage = new PrimaryRealmIntroPage(this);
	addPage (intropage, i18n( "Step 1: Introduction" ) );
	setHelpEnabled(TQWizard::page(0), false);

	realmpage = new PrimaryRealmConfigPage(this);
	addPage (realmpage, i18n( "Step 2: Set Up New Realm" ) );
	setHelpEnabled(TQWizard::page(1), false);

	certpage = new PrimaryCertConfigPage(this);
	addPage (certpage, i18n( "Step 3: Set Up Certificates" ) );
	setHelpEnabled(TQWizard::page(2), false);

	finishpage = new PrimaryRealmFinishPage(this);
	addPage (finishpage, i18n( "Step 4: Initialize New Realm" ) );
	setHelpEnabled(TQWizard::page(3), false);

	// Set up some defaults
	realmpage->txtKDCPort->setValue(88);
	realmpage->txtAdminServerPort->setValue(749);
	realmpage->txtUIDOffset->setValue(5000);
	realmpage->txtGIDOffset->setValue(5000);
	realmpage->txtGIDOffset->setValue(5000);
	TQString domainGuess = m_fqdn;
	int firstDot = domainGuess.find(".");
	if (firstDot >= 0) {
		domainGuess.remove(0, firstDot+1);
	}
	realmpage->txtRealmName->setText(domainGuess);
	realmpage->txtKDC->setText(m_fqdn);
	realmpage->txtAdminServer->setText(m_fqdn);
	realmpage->realmNameChanged();
	certpage->generateKeysEnabled->setChecked(true);
	finishpage->ldapAdminGroupname->setText("realmadmins");
	finishpage->ldapMachineAdminGroupname->setText("machineadmins");
	finishpage->ldapStandardUserGroupname->setText("standardusers");

	// Load certificate info
	certpage->organizationName->setText(m_certconfig.organizationName);
	certpage->orgUnitName->setText(m_certconfig.orgUnitName);
	certpage->commonName->setText(m_certconfig.commonName);
	certpage->localityName->setText(m_certconfig.localityName);
	certpage->stateOrProvinceName->setText(m_certconfig.stateOrProvinceName);
	certpage->countryName->setText(m_certconfig.countryName);
	certpage->emailAddress->setText(m_certconfig.emailAddress);

	// Other setup
	finishpage->ldapAdminRealm->setEnabled(false);

	// Kerberos won't work unless the DNS suffix matches the realm name
	realmpage->txtRealmName->setEnabled(false);

	setFinishEnabled(TQWizard::page(3), true);

	setPosition();
}

PrimaryRealmWizard::~PrimaryRealmWizard() {
}


void PrimaryRealmWizard::next() {
	if (currentPage()==intropage) {
		TQWizard::next();
		realmpage->validateEntries();

		// Focus the first entry field on the new wizard page
		realmpage->txtKDC->setFocus();
		realmpage->txtKDC->selectAll();
	}
	else if (currentPage()==realmpage) {
		// Save realm information
		m_realmconfig.name = realmpage->txtRealmName->text();
		m_realmconfig.bonded = false;
		m_realmconfig.uid_offset = realmpage->txtUIDOffset->value();
		m_realmconfig.gid_offset = realmpage->txtGIDOffset->value();
		m_realmconfig.domain_mappings = TQStringList::split("\n", realmpage->txtDomains->text(), FALSE);
		m_realmconfig.kdc = realmpage->txtKDC->text();
		m_realmconfig.kdc_port = realmpage->txtKDCPort->value();
		m_realmconfig.admin_server = realmpage->txtAdminServer->text();
		m_realmconfig.admin_server_port = realmpage->txtAdminServerPort->value();
		m_realmconfig.pkinit_require_eku = realmpage->checkRequireEKU->isChecked();
		m_realmconfig.pkinit_require_krbtgt_otherName = realmpage->checkRequireKrbtgtOtherName->isChecked();
		m_realmconfig.win2k_pkinit = realmpage->checkWin2k->isChecked();
		m_realmconfig.win2k_pkinit_require_binding = realmpage->checkWin2kPkinitRequireBinding->isChecked();

		finishpage->ldapAdminRealm->setText(realmpage->txtRealmName->text());
		TQWizard::next();
		certpage->processLockouts();
		certpage->validateEntries();

		// Focus the first entry field on the new wizard page
		certpage->organizationName->setFocus();
		certpage->organizationName->selectAll();
	}
	else if (currentPage()==certpage) {
		// Save certificate information
		m_certconfig.generate_certs = certpage->generateKeysEnabled->isOn();
		m_certconfig.provided_kerberos_pem = certpage->kerberosPEM->url();
		m_certconfig.provided_kerberos_pemkey = certpage->kerberosPEMKEY->url();
		m_certconfig.provided_kerberos_crt = certpage->kerberosCRT->url();
		m_certconfig.provided_kerberos_key = certpage->kerberosKEY->url();
		m_certconfig.provided_ldap_crt = certpage->ldapCRT->url();
		m_certconfig.provided_ldap_key = certpage->ldapKEY->url();
		if (m_certconfig.generate_certs) {
			m_certconfig.organizationName = certpage->organizationName->text();
			m_certconfig.orgUnitName = certpage->orgUnitName->text();
			m_certconfig.commonName = certpage->commonName->text();
			m_certconfig.localityName = certpage->localityName->text();
			m_certconfig.stateOrProvinceName = certpage->stateOrProvinceName->text();
			m_certconfig.countryName = certpage->countryName->text();
			m_certconfig.emailAddress = certpage->emailAddress->text();
		}
		else {
			// If generate_certs == false, we need to load m_certconfig structure with data from the provided certificate
			// If this is not done, the automatic certificate updater will fail!
			TQFile file(m_certconfig.provided_kerberos_pem);
			if (file.open(IO_ReadOnly)) {
				TQByteArray ba = file.readAll();
				file.close();
		
				TQCString ssldata(ba);
				ssldata.replace("-----BEGIN CERTIFICATE-----", "");
				ssldata.replace("-----END CERTIFICATE-----", "");
				ssldata.replace("\n", "");
				KSSLCertificate* cert = KSSLCertificate::fromString(ssldata);
				if (cert) {
					TQString subj = cert->getSubject();
					TQStringList subjList = TQStringList::split("/", subj, false);
					for (TQStringList::Iterator it = subjList.begin(); it != subjList.end(); ++it) {
						TQStringList kvPair = TQStringList::split("=", *it, false);
						if (kvPair[0] == "O") {
							m_certconfig.organizationName = kvPair[1];
						}
						else if (kvPair[0] == "OU") {
							m_certconfig.orgUnitName = kvPair[1];
						}
						else if (kvPair[0] == "CN") {
							m_certconfig.commonName = kvPair[1];
						}
						else if (kvPair[0] == "L") {
							m_certconfig.localityName = kvPair[1];
						}
						else if (kvPair[0] == "ST") {
							m_certconfig.stateOrProvinceName = kvPair[1];
						}
						else if (kvPair[0] == "C") {
							m_certconfig.countryName = kvPair[1];
						}
						else if (kvPair[0] == "emailAddress") {
							m_certconfig.emailAddress = kvPair[1];
						}
					}
					delete cert;
				}
			}
		}

		TQWizard::next();
		finishpage->validateEntries();

		// Focus the first entry field on the new wizard page
		finishpage->ldapAdminUsername->setFocus();
		finishpage->ldapAdminUsername->selectAll();
	}
	if (currentPage()==finishpage) {
		//
	}
}

void PrimaryRealmWizard::slotNext() {
	TQWizard::next();
}

void PrimaryRealmWizard::back() {
	TQWizard::back();
}

bool PrimaryRealmWizard::askClose(){
	TQString text;
	if (currentPage()==intropage) {
		return true;
	}
	else {
		if ((currentPage()==certpage) || (currentPage()==finishpage)) {
			text = i18n("<p>Are you sure you want to quit the LDAP Realm Wizard?</p>"
				"<p>If yes, click <b>Quit</b> and all changes will be lost."
				"<br>If not, click <b>Cancel</b> to return and finish your setup.</p>");
		}
		else {
			text = i18n("<p>Are you sure you want to quit the LDAP Realm Wizard?</p>"
				"<p>If not, click <b>Cancel</b> to return and finish setup.</p>");
		}
		int status = KMessageBox::warningContinueCancel(this,  text, i18n("All Changes Will Be Lost"), KStdGuiItem::quit());
		if(status==KMessageBox::Continue){
			setDefaults();
			return true;
		} else {
			return false;
		}
	}
}

/** the cancel button is connected to the reject() slot of TQDialog,
 *  so we have to reimplement this here to add a dialogbox to ask if we
 *  really want to quit the wizard.
 */
void PrimaryRealmWizard::reject() {
	if (askClose()){
		done(-1);
	}
}

void PrimaryRealmWizard::closeEvent(TQCloseEvent* e){
	if ( askClose() )
		done(0);
	else
		e->ignore();
}

/** maybe call a dialog that the wizard has finished. */
void PrimaryRealmWizard::accept() {
	// Validate entries
	if (finishpage->ldapAdminPassword->password() != finishpage->ldapConfirmAdminPassword->password()) {
		KMessageBox::error(this, i18n("<qt><b>Passwords do not match!</b><p>Please re-enter the new administration account password</qt>"), i18n("Input Error"));
		return;
	}
	if (finishpage->ldapAdminPassword->password() == "") {
		KMessageBox::error(this, i18n("<qt><b>Password required!</b><p>Please enter the new administration account password</qt>"), i18n("Input Error"));
		return;
	}

	// Try to create realm
	TQString errorString;
	// FIXME
	// root account should not be locked to "admin"!
	// when fixing, please fix the other instance of locked "admin" in ldapcontroller.cpp ::load()
	backButton()->setEnabled(false);
	nextButton()->setEnabled(false);
	finishButton()->setEnabled(false);
	cancelButton()->setEnabled(false);
	finishpage->setEnabled(false);

	if (m_controller->createNewLDAPRealm(this, m_realmconfig, finishpage->ldapAdminUsername->text(), finishpage->ldapAdminGroupname->text(), finishpage->ldapMachineAdminGroupname->text(), finishpage->ldapStandardUserGroupname->text(), finishpage->ldapAdminPassword->password().utf8(), "admin", finishpage->ldapAdminPassword->password().utf8(), finishpage->ldapAdminRealm->text(), m_certconfig, &errorString) == 0) {
		done(0);
	}
	else {
		KMessageBox::error(this, i18n("<qt><b>Unable to create new realm!</b><p>Details: %1</qt>").arg(errorString), i18n("Unable to create new realm"));
	}

	finishpage->setEnabled(true);
	backButton()->setEnabled(true);
	finishButton()->setEnabled(true);
	cancelButton()->setEnabled(true);
}

/** calls all save functions after resetting all features/ OS/ theme selections to Trinity default */
void PrimaryRealmWizard::setDefaults() {
// 	if(realm_dirty)
// 		realmpage->save(false);
}

/** there seems to be a bug in TQWizard, that makes this evil hack necessary */
void PrimaryRealmWizard::setPosition() {
	TQSize hint = intropage->sizeHint();
	TQSize realm_size = realmpage->sizeHint();
	TQSize finish_size = finishpage->sizeHint();

	// get the width of the broadest child-widget
	if ( hint.width() < realm_size.width() )
		hint.setWidth(realm_size.width());
	if ( hint.width() < finish_size.width() )
		hint.setWidth(finish_size.width());

	// get the height of the highest child-widget
	if ( hint.height() < realm_size.height() )
		hint.setHeight(realm_size.height());
	if ( hint.height() < finish_size.height() )
		hint.setHeight(finish_size.height());

	// set the position
	TQRect rect = TDEGlobalSettings::desktopGeometry(TQCursor::pos());
	int w = rect.x() + (rect.width() - hint.width())/2 - 9;
	int h = rect.y() + (rect.height() - hint.height())/2;
	move(w, h);
}
