/*
 * Copyright (C) 2004, Mart Kelder (mart.kde@hccnet.nl)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "tdeio_delete.h"

#include "mailid.h"
#include "stringid.h"
#include "tdeio.h"
#include "tdeio_proto.h"

#include <kdebug.h>
#include <tdelocale.h>
#include <kurl.h>
#include <tdeio/global.h>
#include <tdeio/jobclasses.h>
#include <tdeio/scheduler.h>

#include <tqptrlist.h>

TDEIO_Delete::TDEIO_Delete( TQObject * parent, const char * name ) : TQObject( parent, name ),
	_tdeio( 0 ),
	_total( 0 ),
	_jobs( 0 ),
	_slave( 0 ),
	_valid( true )
{
	_jobs = new TQPtrList< TDEIO::Job >;
}

TDEIO_Delete::~TDEIO_Delete( )
{
	disConnect( );
	delete _jobs;
}

bool TDEIO_Delete::deleteMails( TQPtrList< const KornMailId > * ids, KKioDrop *drop )
{
	KURL kurl = *drop->_kurl;
	TDEIO::MetaData metadata = *drop->_metadata;
		
	_tdeio = drop;
	_valid = true;
	
	//disConnect earlier operations
	disConnect( );
	if( _tdeio->_protocol->connectionBased( ) )
	{
		if( ! setupSlave( kurl, metadata, _tdeio->_protocol ) )
		{
			_valid = false;
			return false;
		}
	}
	
	_total = ids->count( );
	
	for( const KornMailId * item = ids->first(); item; item = ids->next() )
		deleteItem( item, kurl, metadata, _tdeio->_protocol );
	
	if( _jobs->count() == 0 )
	{
		_tdeio->emitDeleteMailsReady( true );
		disConnect( );
		return true;
	}
	
	if( _tdeio->_protocol->commitDelete() )
		commitDelete( kurl, metadata, _tdeio->_protocol );
			
	_tdeio->emitDeleteMailsTotalSteps( _total );
		
	return true;
}

void TDEIO_Delete::disConnect( )
{
	_jobs->clear( );

	if( _slave )
	{
		TDEIO::Scheduler::disconnectSlave( _slave );
		_slave = 0;
	}
}

bool TDEIO_Delete::setupSlave( KURL kurl, TDEIO::MetaData metadata, const TDEIO_Protocol *& protocol )
{
	protocol->deleteMailConnectKURL( kurl, metadata );
	
	if( kurl.port() == 0 )
		kurl.setPort( protocol->defaultPort( _tdeio->_ssl ) );
		
	if( ! ( _slave = TDEIO::Scheduler::getConnectedSlave( kurl, metadata ) ) )
	{
		kdWarning() << i18n( "Could not get a connected slave; I cannot delete this way..." ) << endl;
		_valid = false;
		return false;
	}

	return true;
}

void TDEIO_Delete::deleteItem( const KornMailId *item, KURL kurl, TDEIO::MetaData metadata, const TDEIO_Protocol *& protocol )
{
	TDEIO::Job* job = 0;

	kurl = dynamic_cast<const KornStringId*>( item )->getId();
	
	protocol->deleteMailKURL( kurl, metadata );
	
	if( kurl.port() == 0 )
		kurl.setPort( protocol->defaultPort( _tdeio->_ssl ) );
		
	if( protocol->deleteFunction() == TDEIO_Protocol::get )
	{
		job = TDEIO::get( kurl, true, false );
		
		if( protocol->connectionBased() )
			TDEIO::Scheduler::assignJobToSlave( _slave, dynamic_cast< TDEIO::SimpleJob* >( job ) );
		else
			TDEIO::Scheduler::scheduleJob( dynamic_cast< TDEIO::SimpleJob* >( job ) );
	}
	else if( protocol->deleteFunction() == TDEIO_Protocol::del )
	{
		job = TDEIO::del( kurl, false, false );
	}
	else
		return; //Unknown deleteFunction
		
	connect( job, TQ_SIGNAL( result( TDEIO::Job* ) ), this, TQ_SLOT( slotResult( TDEIO::Job* ) ) );
		
	job->addMetaData( metadata );
	
	_jobs->append( dynamic_cast< TDEIO::Job* >( job ) );
}

/*
 * Some protocols needs to a command to commit protocols.
 */
void TDEIO_Delete::commitDelete( KURL kurl, TDEIO::MetaData metadata, const TDEIO_Protocol *& protocol )
{
	protocol->deleteCommitKURL( kurl, metadata );
	
	if( kurl.port() == 0 )
		kurl.setPort( protocol->defaultPort( _tdeio->_ssl ) );
	
	TDEIO::TransferJob *job = TDEIO::get( kurl, true, false );
	job->addMetaData( metadata );
	connect( job, TQ_SIGNAL( result( TDEIO::Job* ) ), this, TQ_SLOT( slotResult( TDEIO::Job* ) ) );
	
	_jobs->append( dynamic_cast< TDEIO::Job* >( job ) );
	
	if( protocol->connectionBased() )
		TDEIO::Scheduler::assignJobToSlave( _slave, job );
	else
		TDEIO::Scheduler::scheduleJob( job );

	_total++;
}

void TDEIO_Delete::canceled( )
{
	disConnect( );
}

void TDEIO_Delete::slotResult( TDEIO::Job* job )
{
	if( job->error() )
	{
		kdWarning() << i18n( "An error occurred when deleting email: %1." ).arg( job->errorString() ) << endl;
		_valid = false;
	}
	
	_jobs->remove( job );
	
	_tdeio->emitDeleteMailsProgress( _total - _jobs->count() );
	
	if( _jobs->isEmpty() )
	{
		_tdeio->emitDeleteMailsReady( _valid );
		disConnect();
	}
}


#include "tdeio_delete.moc"
