/*
 *  This file is part of the KDE project
 *  Copyright (C) 2002 Ian Reinhart Geiser <geiseri@kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License version 2.0 as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  $Id$
 */

#include "weatherbar.h"
#include "dockwidget.h"
#include "sidebarwidget.h"

#include <tdeapplication.h>
#include <tqlabel.h>
#include <tqfont.h>
#include <tqlayout.h>
#include <tqscrollview.h>
#include <tqgroupbox.h>
#include <tdelocale.h>
#include <kdebug.h>
#include <tdeconfig.h>
#include <kstandarddirs.h>
#include <tdeglobalsettings.h>
#include <tdemessagebox.h>
#include <dcopclient.h>
#include <dcopref.h>


KonqSidebarWeather::KonqSidebarWeather(TDEInstance* inst, TQObject* parent,
                                         TQWidget* widgetParent,
                                         TQString& desktopName, const char* name)

    : KonqSidebarPlugin(inst, parent, widgetParent, desktopName, name),
    DCOPObject(name)
{

	m_container = new sidebarwidget(widgetParent,"sidebarwidget");

	kdDebug() << "Calling Get Widget" << endl;

	kdDebug() << "Get weatherstation list... " << endl;

	if (!(
		connectDCOPSignal(0, 0, "fileUpdate(TQString)",     "refresh(TQString)", false) &&
		connectDCOPSignal(0, 0, "stationAdded(TQString)",   "updateWidgets()",   false) &&
		connectDCOPSignal(0, 0, "stationRemoved(TQString)", "updateWidgets()",   false)
	))
		kdDebug() << "Could not attach signal..." << endl;
	else
		kdDebug() << "attached dcop signals..." << endl;

	timeOut = new TQTimer(this, "timeOut" );
	connect(timeOut, TQ_SIGNAL(timeout()), this, TQ_SLOT(updateWidgets()));

	updateWidgets();
}


KonqSidebarWeather::~KonqSidebarWeather()
{
}

void* KonqSidebarWeather::provides(const TQString&)
{
    return 0;
}

void KonqSidebarWeather::emitStatusBarText(const TQString& s)
{
}

TQWidget* KonqSidebarWeather::getWidget()
{
	return m_container;
}

void KonqSidebarWeather::refresh(TQString stationID)
{
	kdDebug() << "refresh " << stationID << endl;
	if(m_widgets.find(stationID))
	{
		/* These updates are seemingly now handled by dockwidget itself
		DCOPRef dcopCall( "KWeatherService", "WeatherService" );
		m_widgets[stationID]->setWeatherIcon(dcopCall.call("currentIcon(TQString)", stationID ,true ));
		m_widgets[stationID]->setTemperature(dcopCall.call("temperature(TQString)", stationID,true ));
		m_widgets[stationID]->setPressure(dcopCall.call("pressure(TQString)", stationID,true ));
		m_widgets[stationID]->setWind(dcopCall.call("wind(TQString)", stationID,true ));
		*/

		m_widgets[stationID]->showWeather();
	}
}

void KonqSidebarWeather::handleURL(const KURL& url)
{
}

void KonqSidebarWeather::handlePreview(const KFileItemList& items)
{
}

void KonqSidebarWeather::handlePreviewOnMouseOver(const KFileItem& item)
{
}

void KonqSidebarWeather::updateWidgets()
{
	kdDebug() << "updating station widgets" << endl;

	timeOut->stop();

	DCOPRef dcopCall( "KWeatherService", "WeatherService" );
	DCOPReply reply = dcopCall.call("listStations()", true );

	// KWeatherService might not be active at this point
	if( !reply.isValid() ) {
		kdDebug() << "Starting KWeatherService" << endl;
		if( !startWeatherService() ) {
			KMessageBox::sorry(0, i18n("Could not start the weather service!"));
			return;
		}

		reply = dcopCall.call("listStations()", true );
	}

	if ( reply.isValid() ) {
		TQStringList replyList = reply;

		// Check for new cities
		for(int i = 0; i < replyList.size(); i++)
		{
			if(!m_widgets.find(replyList[i]))
			{
				TQLabel *city = new TQLabel(
					dcopCall.call("stationName(TQString)", replyList[i], true),
					m_container->viewport()
				);
				city->setPaletteBackgroundColor(TDEGlobalSettings::highlightColor());
				city->setPaletteForegroundColor(TDEGlobalSettings::highlightedTextColor());
				dockwidget *d = new dockwidget(replyList[i], m_container->viewport());
				d->resizeView(TQSize(d->width(), 48));
				city->show();
				d->show();
				m_labels.insert(replyList[i], city);
				m_widgets.insert(replyList[i], d);
			}
			dcopCall.send("update(TQString)", replyList[i]);
		}

		// Check for removed cities
		TQDictIterator<dockwidget> it(m_widgets);
		for(; it.current(); ++it)
		{
			TQString current(it.currentKey());
			if(!replyList.contains(current))
			{
				m_widgets[current]->~dockwidget();
				m_labels[current]->~TQLabel();
				m_widgets.remove(current);
				m_labels.remove(current);
			}
		}
	} else {
		KMessageBox::sorry(0, i18n("The weather service is unreachable!"));
	}

	timeOut->start(15*60000);
}

bool KonqSidebarWeather::startWeatherService()
{
	TQByteArray data, replyData;
	TQCString replyType;
	TQDataStream arg(data, IO_WriteOnly);
	arg << TQString("KWeatherService") << TQStringList();

	if ( !kapp->dcopClient()->call(
			"tdelauncher", "tdelauncher",
			"start_service_by_name(TQString,TQStringList)",
			data, replyType, replyData) ) {
		kdDebug() << "Cannot start weather service: tdelauncher call failed." << endl;
		return false;
	}

	TQDataStream reply(replyData, IO_ReadOnly);
	if ( replyType != "serviceResult" )
	{
		kdDebug() << "Cannot start weather service: unknown reply type by tdelauncher." << endl;
		return false;
	}

	int result;
	TQCString dcopName;
	TQString error;
	reply >> result >> dcopName >> error;
	if (result != 0)
	{
		kdDebug() << "Cannot start weather service: " << error.local8Bit().data() << endl;
		return false;
	}

	return true;
}

extern "C"
{
    KDE_EXPORT void* create_weather_sidebar(TDEInstance* inst, TQObject* par, TQWidget*widp,
                                      TQString& desktopname, const char* name)
    {
        return new KonqSidebarWeather(inst, par, widp, desktopname, name);
    }

    KDE_EXPORT bool add_weather_sidebar(TQString* fn, TQString* /*param*/,
                                  TQMap<TQString, TQString>* map)
    {
                    map->insert("Type","Link");
                        map->insert("Icon","kweather");
                        map->insert("Name",i18n("Weather"));
                        map->insert("Open","false");
                          map->insert("X-TDE-KonqSidebarModule","weather_sidebar");
                        fn->setLatin1("weatherbar%1.desktop");
                        return true;
    }
}

#include "weatherbar.moc"

