/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2003-10-22
 * Description : preview raw file widget used in single convert
 *
 * Copyright (C) 2003-2005 by Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Copyright (C) 2006-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt includes.

#include <tqpainter.h>
#include <tqimage.h>
#include <tqstring.h>
#include <tqevent.h>
#include <tqtimer.h>
#include <tqfile.h>

// KDE includes.

#include <tdelocale.h>

// Local includes.

#include "previewwidget.h"
#include "previewwidget.moc"

namespace KIPIRawConverterPlugin
{

class PreviewWidgetPriv
{
public:

    PreviewWidgetPriv()
    {
        pix   = 0;
        timer = 0;
    }

    TQPixmap *pix;
    TQPixmap  preview;

    TQTimer  *timer;

    TQString  text;

    TQImage   image;
};

PreviewWidget::PreviewWidget(TQWidget *parent)
             : TQFrame(parent, 0, TQt::WRepaintNoErase)
{
    d = new PreviewWidgetPriv;
    setFrameStyle(TQFrame::GroupBoxPanel|TQFrame::Plain); 
    setMargin(0); 
    setLineWidth(1); 

    setMinimumSize(TQSize(400, 300));
    setSizePolicy(TQSizePolicy::Expanding, TQSizePolicy::Expanding);

    d->pix = new TQPixmap(400, 300);
    d->pix->fill(TQt::black);

    d->timer = new TQTimer(this);

    connect(d->timer, TQT_SIGNAL(timeout()),
            this, TQT_SLOT(slotResize()));
}

PreviewWidget::~PreviewWidget()
{
    delete d;
}

void PreviewWidget::load(const TQString& file)
{
    d->text = "";
    d->pix->fill(TQt::black);
    d->image.load(file);

    if (!d->image.isNull()) 
    {
        TQImage img = d->image.scale(width(),height(),TQImage::ScaleMin);
        int x = d->pix->width()/2 - img.width()/2;
        int y = d->pix->height()/2 - img.height()/2;

        TQPainter p(d->pix);
        p.drawImage(x, y, img);
        p.setPen(TQPen(TQt::white));
        p.drawRect(x,y,img.width(),img.height());
        p.end();
    }
    else 
    {
        setInfo(i18n( "Failed to load image after processing" ));
        return;
    }

    update();
}

void PreviewWidget::setInfo(const TQString& text, const TQColor& color, const TQPixmap& preview)
{
    d->text    = text;
    d->preview = preview;
    d->pix->fill(TQt::black);
    TQPainter p(d->pix);
    p.setPen(TQPen(color));

    if (!d->preview.isNull())
    {
        p.drawPixmap(d->pix->width()/2-d->preview.width()/2, d->pix->height()/4-d->preview.height()/2, 
                     d->preview, 0, 0, d->preview.width(), d->preview.height());
        p.drawText(0, d->pix->height()/2, d->pix->width(), d->pix->height()/2,
                   TQt::AlignCenter|TQt::WordBreak, d->text);
    }
    else
    {
        p.drawText(0, 0, d->pix->width(), d->pix->height(),
                   TQt::AlignCenter|TQt::WordBreak, d->text);
    }
    p.end();
    update();
}

void PreviewWidget::paintEvent(TQPaintEvent *e)
{
    TQRect r(e->rect());
    bitBlt(this, r.topLeft(), d->pix, r, TQt::CopyROP);
}

void PreviewWidget::resizeEvent(TQResizeEvent*)
{
    d->timer->start(10,true);
}

void PreviewWidget::slotResize()
{
    if (d->timer->isActive()) return;

    d->pix->resize(width(),height());
    d->pix->fill(TQt::black);
    if (!d->text.isEmpty()) 
    {
        TQPainter p(d->pix);
        p.setPen(TQPen(TQt::white));

        if (!d->preview.isNull())
        {
            p.drawPixmap(d->pix->width()/2-d->preview.width()/2, d->pix->height()/4-d->preview.height()/2, 
                        d->preview, 0, 0, d->preview.width(), d->preview.height());
            p.drawText(0, d->pix->height()/2, d->pix->width(), d->pix->height()/2,
                    TQt::AlignCenter|TQt::WordBreak, d->text);
        }
        else
        {
            p.drawText(0, 0, d->pix->width(), d->pix->height(),
                    TQt::AlignCenter|TQt::WordBreak, d->text);
        }

        p.end();
    }
    else 
    {
        if (!d->image.isNull()) 
        {
            TQImage img = d->image.scale(width(),height(), TQImage::ScaleMin);
            int x = d->pix->width()/2 - img.width()/2;
            int y = d->pix->height()/2 - img.height()/2;

            TQPainter p(d->pix);
            p.drawImage(x, y, img);
            p.setPen(TQPen(TQt::white));
            p.drawRect(x,y,img.width(),img.height());
            p.end();
        }
    }

    update();
}

} // NameSpace KIPIRawConverterPlugin

