/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2006-10-12
 * Description : IPTC status settings page.
 *
 * Copyright (C) 2006-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// QT includes.

#include <tqlayout.h>
#include <tqcombobox.h>
#include <tqlabel.h>
#include <tqwhatsthis.h>
#include <tqvalidator.h>

// KDE includes.

#include <tdelocale.h>
#include <kdialog.h>
#include <klineedit.h>
#include <tdeapplication.h>
#include <tdeaboutdata.h>
#include <kactivelabel.h>

// LibKExiv2 includes. 

#include <libkexiv2/kexiv2.h>

// Local includes.

#include "metadatacheckbox.h"
#include "pluginsversion.h"
#include "iptcstatus.h"
#include "iptcstatus.moc"

namespace KIPIMetadataEditPlugin
{

class IPTCStatusPriv
{
public:

    IPTCStatusPriv()
    {
        priorityCB          = 0;
        objectCycleCB       = 0;
        objectTypeCB        = 0;
        statusEdit          = 0;
        JobIDEdit           = 0;
        priorityCheck       = 0;
        objectCycleCheck    = 0;
        objectTypeCheck     = 0;
        statusCheck         = 0;
        JobIDCheck          = 0;
    }

    TQCheckBox        *statusCheck;
    TQCheckBox        *JobIDCheck;

    TQComboBox        *priorityCB;
    TQComboBox        *objectCycleCB;
    TQComboBox        *objectTypeCB;
    TQComboBox        *objectAttributeCB;

    KLineEdit        *statusEdit;
    KLineEdit        *objectTypeDescEdit;
    KLineEdit        *objectAttributeDescEdit;
    KLineEdit        *JobIDEdit;

    MetadataCheckBox *priorityCheck;
    MetadataCheckBox *objectCycleCheck;
    MetadataCheckBox *objectTypeCheck;
    MetadataCheckBox *objectAttributeCheck;
};

IPTCStatus::IPTCStatus(TQWidget* parent)
          : TQWidget(parent)
{
    d = new IPTCStatusPriv;

    TQGridLayout* grid = new TQGridLayout(parent, 11, 2, KDialog::spacingHint());

    // IPTC only accept printable Ascii char.
    TQRegExp asciiRx("[\x20-\x7F]+$");
    TQValidator *asciiValidator = new TQRegExpValidator(asciiRx, TQT_TQOBJECT(this));

    // --------------------------------------------------------

    d->statusCheck = new TQCheckBox(i18n("Edit Status:"), parent);
    d->statusEdit  = new KLineEdit(parent);
    d->statusEdit->setValidator(asciiValidator);
    d->statusEdit->setMaxLength(64);
    grid->addMultiCellWidget(d->statusCheck, 0, 0, 0, 2);
    grid->addMultiCellWidget(d->statusEdit, 1, 1, 0, 2);
    TQWhatsThis::add(d->statusEdit, i18n("<p>Set here the content status. This field is limited "
                                        "to 64 ASCII characters."));

    // --------------------------------------------------------

    d->priorityCheck = new MetadataCheckBox(i18n("Priority:"), parent);
    d->priorityCB    = new TQComboBox(false, parent);
    d->priorityCB->insertItem(i18n("0: None"), 0);
    d->priorityCB->insertItem(i18n("1: High"), 1);
    d->priorityCB->insertItem("2", 2);
    d->priorityCB->insertItem("3", 3);
    d->priorityCB->insertItem("4", 4);
    d->priorityCB->insertItem(i18n("5: Normal"), 5);
    d->priorityCB->insertItem("6", 6);
    d->priorityCB->insertItem("7", 7);
    d->priorityCB->insertItem(i18n("8: Low"), 8);
    grid->addMultiCellWidget(d->priorityCheck, 2, 2, 0, 0);
    grid->addMultiCellWidget(d->priorityCB, 2, 2, 1, 1);
    TQWhatsThis::add(d->priorityCB, i18n("<p>Select here the editorial urgency of content."));

    // --------------------------------------------------------

    d->objectCycleCheck = new MetadataCheckBox(i18n("Object Cycle:"), parent);
    d->objectCycleCB    = new TQComboBox(false, parent);
    d->objectCycleCB->insertItem(i18n("Morning"),   0);
    d->objectCycleCB->insertItem(i18n("Afternoon"), 1);
    d->objectCycleCB->insertItem(i18n("Evening"),   2);
    grid->addMultiCellWidget(d->objectCycleCheck, 3, 3, 0, 0);
    grid->addMultiCellWidget(d->objectCycleCB, 3, 3, 1, 1);
    TQWhatsThis::add(d->objectCycleCB, i18n("<p>Select here the editorial cycle of content."));
      
    // --------------------------------------------------------

    d->objectTypeCheck    = new MetadataCheckBox(i18n("Object Type:"), parent);
    d->objectTypeCB       = new TQComboBox(false, parent);
    d->objectTypeDescEdit = new KLineEdit(parent);
    d->objectTypeDescEdit->setValidator(asciiValidator);
    d->objectTypeDescEdit->setMaxLength(64);
    d->objectTypeCB->insertItem(i18n("News"),     0);
    d->objectTypeCB->insertItem(i18n("Data"),     1);
    d->objectTypeCB->insertItem(i18n("Advisory"), 2);
    grid->addMultiCellWidget(d->objectTypeCheck, 4, 4, 0, 0);
    grid->addMultiCellWidget(d->objectTypeCB, 4, 4, 1, 1);
    grid->addMultiCellWidget(d->objectTypeDescEdit, 5, 5, 0, 2);
    TQWhatsThis::add(d->objectTypeCB, i18n("<p>Select here the editorial type of content."));
    TQWhatsThis::add(d->objectTypeDescEdit, i18n("<p>Set here the editorial type description of content. "
                                                "This field is limited to 64 ASCII characters."));

    // --------------------------------------------------------

    d->objectAttributeCheck    = new MetadataCheckBox(i18n("Object Attribute:"), parent);
    d->objectAttributeCB       = new TQComboBox(false, parent);
    d->objectAttributeDescEdit = new KLineEdit(parent);
    d->objectAttributeDescEdit->setValidator(asciiValidator);
    d->objectAttributeDescEdit->setMaxLength(64);
    d->objectAttributeCB->insertItem(i18n("Current"),                           0);
    d->objectAttributeCB->insertItem(i18n("Analysis"),                          1);
    d->objectAttributeCB->insertItem(i18n("Archive material"),                  2);
    d->objectAttributeCB->insertItem(i18n("Background"),                        3);
    d->objectAttributeCB->insertItem(i18n("Feature"),                           4);
    d->objectAttributeCB->insertItem(i18n("Forecast"),                          5);
    d->objectAttributeCB->insertItem(i18n("History"),                           6);
    d->objectAttributeCB->insertItem(i18n("Obituary"),                          7);
    d->objectAttributeCB->insertItem(i18n("Opinion"),                           8);
    d->objectAttributeCB->insertItem(i18n("Polls & Surveys"),                   9);
    d->objectAttributeCB->insertItem(i18n("Profile"),                           10);
    d->objectAttributeCB->insertItem(i18n("Results Listings & Table"),          11);
    d->objectAttributeCB->insertItem(i18n("Side bar & Supporting information"), 12);
    d->objectAttributeCB->insertItem(i18n("Summary"),                           13);
    d->objectAttributeCB->insertItem(i18n("Transcript & Verbatim"),             14);
    d->objectAttributeCB->insertItem(i18n("Interview"),                         15);
    d->objectAttributeCB->insertItem(i18n("From the Scene"),                    16);
    d->objectAttributeCB->insertItem(i18n("Retrospective"),                     17);
    d->objectAttributeCB->insertItem(i18n("Statistics"),                        18);
    d->objectAttributeCB->insertItem(i18n("Update"),                            19);
    d->objectAttributeCB->insertItem(i18n("Wrap-up"),                           20);
    d->objectAttributeCB->insertItem(i18n("Press Release"),                     21);
    grid->addMultiCellWidget(d->objectAttributeCheck, 6, 6, 0, 0);
    grid->addMultiCellWidget(d->objectAttributeCB, 6, 6, 1, 2);
    grid->addMultiCellWidget(d->objectAttributeDescEdit, 7, 7, 0, 2);
    TQWhatsThis::add(d->objectAttributeCB, i18n("<p>Select here the editorial attribute of content."));
    TQWhatsThis::add(d->objectAttributeDescEdit, i18n("<p>Set here the editorial attribute description of "
                                                     "content. This field is limited to 64 ASCII characters."));

    // --------------------------------------------------------

    d->JobIDCheck = new TQCheckBox(i18n("Job Identification:"), parent);
    d->JobIDEdit  = new KLineEdit(parent);
    d->JobIDEdit->setValidator(asciiValidator);
    d->JobIDEdit->setMaxLength(32);
    grid->addMultiCellWidget(d->JobIDCheck, 8, 8, 0, 2);
    grid->addMultiCellWidget(d->JobIDEdit, 9, 9, 0, 2);
    TQWhatsThis::add(d->JobIDEdit, i18n("<p>Set here the string that identifies content that recurs. "
                                       "This field is limited to 32 ASCII characters."));

    // --------------------------------------------------------

    KActiveLabel *note = new KActiveLabel(i18n("<b>Note: "
                 "<b><a href='http://en.wikipedia.org/wiki/IPTC'>IPTC</a></b> "
                 "text tags only support the printable "
                 "<b><a href='http://en.wikipedia.org/wiki/Ascii'>ASCII</a></b> "
                 "characters set and limit strings size. "
                 "Use contextual help for details.</b>"), parent);

    grid->addMultiCellWidget(note, 10, 10, 0, 2);
    grid->setColStretch(2, 10);                     
    grid->setRowStretch(11, 10);                     

    // --------------------------------------------------------

    connect(d->priorityCheck, TQT_SIGNAL(toggled(bool)),
            d->priorityCB, TQT_SLOT(setEnabled(bool)));

    connect(d->objectCycleCheck, TQT_SIGNAL(toggled(bool)),
            d->objectCycleCB, TQT_SLOT(setEnabled(bool)));

    connect(d->objectTypeCheck, TQT_SIGNAL(toggled(bool)),
            d->objectTypeCB, TQT_SLOT(setEnabled(bool)));

    connect(d->objectTypeCheck, TQT_SIGNAL(toggled(bool)),
            d->objectTypeDescEdit, TQT_SLOT(setEnabled(bool)));

    connect(d->objectAttributeCheck, TQT_SIGNAL(toggled(bool)),
            d->objectAttributeCB, TQT_SLOT(setEnabled(bool)));

    connect(d->objectAttributeCheck, TQT_SIGNAL(toggled(bool)),
            d->objectAttributeDescEdit, TQT_SLOT(setEnabled(bool)));

    connect(d->statusCheck, TQT_SIGNAL(toggled(bool)),
            d->statusEdit, TQT_SLOT(setEnabled(bool)));

    connect(d->JobIDCheck, TQT_SIGNAL(toggled(bool)),
            d->JobIDEdit, TQT_SLOT(setEnabled(bool)));

    // --------------------------------------------------------

    connect(d->priorityCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->objectCycleCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->objectTypeCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->objectTypeCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->objectAttributeCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->objectAttributeCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->statusCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->JobIDCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));

    // --------------------------------------------------------

    connect(d->priorityCB, TQT_SIGNAL(activated(int)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->objectCycleCB, TQT_SIGNAL(activated(int)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->objectTypeCB, TQT_SIGNAL(activated(int)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->objectAttributeCB, TQT_SIGNAL(activated(int)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->statusEdit, TQT_SIGNAL(textChanged(const TQString &)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->objectTypeDescEdit, TQT_SIGNAL(textChanged(const TQString &)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->objectAttributeDescEdit, TQT_SIGNAL(textChanged(const TQString &)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->JobIDEdit, TQT_SIGNAL(textChanged(const TQString &)),
            this, TQT_SIGNAL(signalModified()));
}

IPTCStatus::~IPTCStatus()
{
    delete d;
}

void IPTCStatus::readMetadata(TQByteArray& iptcData)
{
    blockSignals(true);
    KExiv2Iface::KExiv2 exiv2Iface;
    exiv2Iface.setIptc(iptcData);
    TQString data;
    int     val;

    d->statusEdit->clear();
    d->statusCheck->setChecked(false);
    data = exiv2Iface.getIptcTagString("Iptc.Application2.EditStatus", false);    
    if (!data.isNull())
    {
        d->statusEdit->setText(data);
        d->statusCheck->setChecked(true);
    }
    d->statusEdit->setEnabled(d->statusCheck->isChecked());

    d->priorityCB->setCurrentItem(0);
    d->priorityCheck->setChecked(false);
    data = exiv2Iface.getIptcTagString("Iptc.Application2.Urgency", false);    
    if (!data.isNull())
    {
        val = data.toInt(); 
        if (val >= 0 && val <= 8)
        {
            d->priorityCB->setCurrentItem(val);
            d->priorityCheck->setChecked(true);
        }
        else
            d->priorityCheck->setValid(false);
    }
    d->priorityCB->setEnabled(d->priorityCheck->isChecked());

    d->objectCycleCB->setCurrentItem(0);
    d->objectCycleCheck->setChecked(false);
    data = exiv2Iface.getIptcTagString("Iptc.Application2.ObjectCycle", false);    
    if (!data.isNull())
    {
        if (data == TQString("a"))
        {
            d->objectCycleCB->setCurrentItem(0);
            d->objectCycleCheck->setChecked(true);
        }
        else if (data == TQString("b"))
        {
            d->objectCycleCB->setCurrentItem(1);
            d->objectCycleCheck->setChecked(true);
        }
        else if (data == TQString("c"))
        {
            d->objectCycleCB->setCurrentItem(2);
            d->objectCycleCheck->setChecked(true);
        }
        else 
            d->objectCycleCheck->setValid(false);
    }
    d->objectCycleCB->setEnabled(d->objectCycleCheck->isChecked());

    d->objectTypeCB->setCurrentItem(0);
    d->objectTypeDescEdit->clear();
    d->objectTypeCheck->setChecked(false);
    data = exiv2Iface.getIptcTagString("Iptc.Application2.ObjectType", false);    
    if (!data.isNull())
    {
        TQString typeSec = data.section(":", 0, 0);
        if (!typeSec.isEmpty())
        {
            int type = typeSec.toInt()-1;
            if (type >= 0 && type < 3)
            {
                d->objectTypeCB->setCurrentItem(type);
                d->objectTypeDescEdit->setText(data.section(":", -1));
                d->objectTypeCheck->setChecked(true);
            }
            else
                d->objectTypeCheck->setValid(false);
        }
    }
    d->objectTypeCB->setEnabled(d->objectTypeCheck->isChecked());
    d->objectTypeDescEdit->setEnabled(d->objectTypeCheck->isChecked());

    d->objectAttributeCB->setCurrentItem(0);
    d->objectAttributeDescEdit->clear();
    d->objectAttributeCheck->setChecked(false);
    data = exiv2Iface.getIptcTagString("Iptc.Application2.ObjectAttribute", false);    
    if (!data.isNull())
    {
        TQString attSec = data.section(":", 0, 0);
        if (!attSec.isEmpty())
        {
            int att = attSec.toInt()-1;
            if (att >= 0 && att < 21)
            {
                d->objectAttributeCB->setCurrentItem(att);
                d->objectAttributeDescEdit->setText(data.section(":", -1));
                d->objectAttributeCheck->setChecked(true);
            }
            else 
                d->objectAttributeCheck->setValid(false);
        }
    }
    d->objectAttributeCB->setEnabled(d->objectAttributeCheck->isChecked());
    d->objectAttributeDescEdit->setEnabled(d->objectAttributeCheck->isChecked());

    d->JobIDEdit->clear();
    d->JobIDCheck->setChecked(false);
    data = exiv2Iface.getIptcTagString("Iptc.Application2.FixtureId", false);    
    if (!data.isNull())
    {
        d->JobIDEdit->setText(data);
        d->JobIDCheck->setChecked(true);
    }
    d->JobIDEdit->setEnabled(d->JobIDCheck->isChecked());

    blockSignals(false);
}

void IPTCStatus::applyMetadata(TQByteArray& iptcData)
{
    KExiv2Iface::KExiv2 exiv2Iface;
    exiv2Iface.setIptc(iptcData);

    if (d->statusCheck->isChecked())
        exiv2Iface.setIptcTagString("Iptc.Application2.EditStatus", d->statusEdit->text());
    else
        exiv2Iface.removeIptcTag("Iptc.Application2.EditStatus");

    if (d->priorityCheck->isChecked())
        exiv2Iface.setIptcTagString("Iptc.Application2.Urgency", TQString::number(d->priorityCB->currentItem()));
    else if (d->priorityCheck->isValid())
        exiv2Iface.removeIptcTag("Iptc.Application2.Urgency");

    if (d->objectCycleCheck->isChecked())
    {
        switch (d->objectCycleCB->currentItem())
        {
            case(0):
                exiv2Iface.setIptcTagString("Iptc.Application2.ObjectCycle", TQString("a"));
                break;
                
            case(1):
                exiv2Iface.setIptcTagString("Iptc.Application2.ObjectCycle", TQString("b"));
                break;
        
            case(2):
                exiv2Iface.setIptcTagString("Iptc.Application2.ObjectCycle", TQString("c"));
                break;
        }
    }
    else if (d->objectCycleCheck->isValid())
        exiv2Iface.removeIptcTag("Iptc.Application2.ObjectCycle");

    if (d->objectTypeCheck->isChecked())
    {
        TQString objectType;
        objectType.sprintf("%2d", d->objectTypeCB->currentItem()+1);
        objectType.append(TQString(":%1").arg(d->objectTypeDescEdit->text()));
        exiv2Iface.setIptcTagString("Iptc.Application2.ObjectType", objectType);
    }
    else if (d->objectTypeCheck->isValid())
        exiv2Iface.removeIptcTag("Iptc.Application2.ObjectType");

    if (d->objectAttributeCheck->isChecked())
    {
        TQString objectAttribute;
        objectAttribute.sprintf("%3d", d->objectAttributeCB->currentItem()+1);
        objectAttribute.append(TQString(":%1").arg(d->objectAttributeDescEdit->text()));
        exiv2Iface.setIptcTagString("Iptc.Application2.ObjectAttribute", objectAttribute);
    }
    else if (d->objectAttributeCheck->isValid())
        exiv2Iface.removeIptcTag("Iptc.Application2.ObjectAttribute");

    if (d->JobIDCheck->isChecked())
        exiv2Iface.setIptcTagString("Iptc.Application2.FixtureId", d->JobIDEdit->text());
    else
        exiv2Iface.removeIptcTag("Iptc.Application2.FixtureId");

    exiv2Iface.setImageProgramId(TQString("Kipi-plugins"), TQString(kipiplugins_version));

    iptcData = exiv2Iface.getIptc();
}

}  // namespace KIPIMetadataEditPlugin
