/*
    This file is part of libtdepim.

    Copyright (c) 2004 Tobias Koenig <tokoe@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef KPIM_ADDRESSEESELECTOR_H
#define KPIM_ADDRESSEESELECTOR_H

#include <tdeabc/addressee.h>
#include <tdeabc/distributionlist.h>
#include <tdeabc/resource.h>
#include <kdialogbase.h>
#include <kdemacros.h>

#include <tqbitarray.h>
#include <tqpixmap.h>
#include <tqwidget.h>

class KComboBox;
class KLineEdit;
class TDEListView;
class TQSignalMapper;

namespace KPIM {

class AddresseeSelector;

class KDE_EXPORT Selection
{
  friend class AddresseeSelector;

  public:
    virtual ~Selection() {}

    /**
      Returns the number of fields the selection offers.
     */
    virtual uint fieldCount() const = 0;

    /**
      Returns the title for the field specified by index.
     */
    virtual TQString fieldTitle( uint index ) const = 0;

    /**
      Returns the number of items for the given addressee.
     */
    virtual uint itemCount( const TDEABC::Addressee &addresse ) const = 0;

    /**
      Returns the text that's used for the item specified by index.
     */
    virtual TQString itemText( const TDEABC::Addressee &addresse, uint index ) const = 0;

    /**
      Returns the icon that's used for the item specified by index.
     */
    virtual TQPixmap itemIcon( const TDEABC::Addressee &addresse, uint index ) const = 0;

    /**
      Returns whether the item specified by index is enabled.
     */
    virtual bool itemEnabled( const TDEABC::Addressee &addresse, uint index ) const = 0;

    /**
      Returns whether the item specified by index matches the passed pattern.
     */
    virtual bool itemMatches( const TDEABC::Addressee &addresse, uint index, const TQString &pattern ) const = 0;

    /**
      Returns whether the item specified by index equals the passed pattern.
     */
    virtual bool itemEquals( const TDEABC::Addressee &addresse, uint index, const TQString &pattern ) const = 0;

    /**
      Returns the text that's used for the given distribution list.
     */
    virtual TQString distributionListText( const TDEABC::DistributionList *distributionList ) const = 0;

    /**
      Returns the icon that's used for the given distribution list.
     */
    virtual TQPixmap distributionListIcon( const TDEABC::DistributionList *distributionList ) const = 0;

    /**
      Returns whether the given distribution list is enabled.
     */
    virtual bool distributionListEnabled( const TDEABC::DistributionList *distributionList ) const = 0;

    /**
      Returns whether the given distribution list matches the passed pattern.
     */
    virtual bool distributionListMatches(  const TDEABC::DistributionList *distributionList,
                                           const TQString &pattern ) const = 0;

    /**
      Returns the number of additional address books.
     */
    virtual uint addressBookCount() const = 0;

    /**
      Returns the title for an additional address book.
     */
    virtual TQString addressBookTitle( uint index ) const = 0;

    /**
      Returns the content for an additional address book.
     */
    virtual TDEABC::Addressee::List addressBookContent( uint index ) const = 0;

  protected:
    AddresseeSelector* selector() { return mSelector; }

  private:
    virtual void addSelectedAddressees( uint fieldIndex, const TDEABC::Addressee&, uint itemIndex ) = 0;
    virtual void addSelectedDistributionList( uint fieldIndex, const TDEABC::DistributionList* ) = 0;

    void setSelector( AddresseeSelector *selector ) { mSelector = selector; }

    AddresseeSelector *mSelector;
};

/**
  Internal helper class
 */
class SelectionItem
{
  public:
    typedef TQValueList<SelectionItem> List;

    SelectionItem( const TDEABC::Addressee &addressee, uint index );
    SelectionItem( TDEABC::DistributionList *list, uint index );
    SelectionItem();

    void addToField( int index );
    void removeFromField( int index );
    bool isInField( int index );

    TDEABC::Addressee addressee() const;
    TDEABC::DistributionList* distributionList() const;
    uint index() const;

  private:
    TDEABC::Addressee mAddressee;
    TDEABC::DistributionList *mDistributionList;
    uint mIndex;
    TQBitArray mField;
};

class KDE_EXPORT AddresseeSelector : public TQWidget
{
  TQ_OBJECT
  

  public:
    AddresseeSelector( Selection *selection,
                       TQWidget *parent, const char *name = 0 );
    ~AddresseeSelector();

    /**
      Writes back the selected items to the selection.
     */
    void finish();

    void setItemSelected( uint fieldIndex, const TDEABC::Addressee&, uint itemIndex );
    void setItemSelected( uint fieldIndex, const TDEABC::Addressee&,
                          uint itemIndex, const TQString& );

  private slots:
    void move( int index );
    void remove( int index );

    void updateAddresseeView();
    void reloadAddressBook();

  private:
    void init();
    void initGUI();

    void updateSelectionView( int index );
    void updateSelectionViews();

    Selection *mSelection;

    KComboBox *mAddressBookCombo;
    KLineEdit *mAddresseeFilter;
    TDEListView *mAddresseeView;
    SelectionItem::List mSelectionItems;

    TQValueList<TDEListView*> mSelectionViews;
    TQSignalMapper *mMoveMapper;
    TQSignalMapper *mRemoveMapper;

    TDEABC::DistributionListManager *mManager;

    class AddressBookManager;
    AddressBookManager *mAddressBookManager;
};

class KDE_EXPORT AddresseeSelectorDialog : public KDialogBase
{
  TQ_OBJECT
  

  public:
    AddresseeSelectorDialog( Selection *selection,
                             TQWidget *parent = 0, const char *name = 0 );

  protected slots:
    void accept();

  private:
    AddresseeSelector *mSelector;
};

}

#endif
