/* This file is part of the KDE project
   Copyright (C) 2004 Cedric Pasteur <cedric.pasteur@free.fr>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include <tqworkspace.h>
#include <tqdockarea.h>
#include <tqdockwindow.h>
#include <tqhbox.h>
#include <tqpainter.h>
#include <tqevent.h>
#include <tqobjectlist.h>

#include <tdeversion.h>
#include <tdeaction.h>
#include <kinstance.h>
#include <tdelocale.h>
#include <tdeaboutdata.h>
#include <kdebug.h>
#include <kstdaction.h>
#include <tdeapplication.h>
#include <kiconloader.h>
#include <tdefiledialog.h>
#include <klibloader.h>
#include <tdemessagebox.h>

#include "form.h"
#include "formIO.h"
#include "objecttree.h"
#include "container.h"
#include "formmanager.h"
#include "objecttreeview.h"

#include "kfd_kdev_part.h"

#define ENABLE_ACTION(name, enable) \
	if(actionCollection()->action( name )) \
		actionCollection()->action( name )->setEnabled( enable )

TDEInstance *KFDFactory::m_instance = 0L;

KFDFactory::KFDFactory()
: KParts::Factory(0, "libkformdesigner_kdev_part")
{}

KFDFactory::~KFDFactory()
{
	if (m_instance)
	{
		delete m_instance->aboutData();
		delete m_instance;
	}

	m_instance = 0;
}

KParts::Part*
KFDFactory::createPartObject( TQWidget *parentWidget, const char *, TQObject *, const char *name,
  const char *classname, const TQStringList &args)
{
	bool readOnly = (classname == "KParts::ReadOnlyPart");
	KFormDesignerKDevPart *part = new KFormDesignerKDevPart(parentWidget, name, readOnly, args);
	return part;
}

TDEInstance*
KFDFactory::instance()
{
	if (!m_instance)
		m_instance = new TDEInstance(aboutData());
	return m_instance;
}

TDEAboutData*
KFDFactory::aboutData()
{
	TDEAboutData *about = new TDEAboutData("kformdesigner_kdev_part", I18N_NOOP("Form Designer Part"), "0.3");
	return about;
}

// copied from kfd_part.cpp
class KFDPart_FormManager : public KFormDesigner::FormManager
{
	public:
		/*! Constructs FormManager object.
		 See WidgetLibrary's constructor documentation for information about
		 \a supportedFactoryGroups parameter.
		 Using \a options you can control manager's behaviour, see \ref Options. */
		KFDPart_FormManager(KFormDesignerPart *part, int options = 0, const char *name = 0)
		 : KFormDesigner::FormManager(part, options, name)
		 , m_part(part)
		{
		}

		virtual TDEAction* action( const char* name)
		{
			return m_part->actionCollection()->action( name );
		}

		virtual void enableAction( const char* name, bool enable ) {
			if(m_part->actionCollection()->action( name ))
				m_part->actionCollection()->action( name )->setEnabled( enable );
		}

		KFormDesignerPart *m_part;
};

//////////////

KFormDesigner::WidgetLibrary* KFormDesignerKDevPart::static_formsLibrary = 0L;

KFormDesignerKDevPart::KFormDesignerKDevPart(TQWidget *parent, const char *name, bool readOnly, const TQStringList &args)
: Designer(parent, name), m_count(0)
{
	setInstance(KFDFactory::instance());
	instance()->iconLoader()->addAppDir("kexi");
	instance()->iconLoader()->addAppDir("kformdesigner");

	setReadWrite(!readOnly);
	m_uniqueFormMode = true;
	m_openingFile = false;

	if(!args.grep("multipleMode").isEmpty())
		setUniqueFormMode(false);
	m_inShell = (!args.grep("shell").isEmpty());

	TQHBox *container = new TQHBox(parent, "kfd_container_widget");
	container->setFocusPolicy(TQWidget::ClickFocus);

	m_workspace = new TQWorkspace(container, "kfd_workspace");
	m_workspace->show();
	TQStringList supportedFactoryGroups;
/* @todo add configuration for supported factory groups */
	static_formsLibrary = KFormDesigner::FormManager::createWidgetLibrary( 
		new KFDPart_FormManager(this, 0, "kfd_manager"), supportedFactoryGroups );

	if(!readOnly)
	{
		TQDockArea *dockArea = new TQDockAreaQt::Vertical, TQDockArea::Reverse, container, "kfd_part_dockarea");

		TQDockWindow *dockTree = new TQDockWindow(dockArea);
		KFormDesigner::ObjectTreeView *view = new KFormDesigner::ObjectTreeView(dockTree);
		dockTree->setWidget(view);
		dockTree->setCaption(i18n("Objects"));
		dockTree->setResizeEnabled(true);
		dockTree->setFixedExtentWidth(256);

		TQDockWindow *dockEditor = new TQDockWindow(dockArea);
		KoProperty::Editor *editor = new KoProperty::Editor(dockEditor);
		dockEditor->setWidget(editor);
		dockEditor->setCaption(i18n("Properties"));
		dockEditor->setResizeEnabled(true);

		KFormDesigner::FormManager::self()->setEditor(editor);
		KFormDesigner::FormManager::self()->setObjectTreeView(view);

		setupActions();
		setModified(false);

		// action stuff
		connect(KFormDesigner::FormManager::self(), TQT_SIGNAL(widgetSelected(KFormDesigner::Form*, bool)), TQT_SLOT(slotWidgetSelected(KFormDesigner::Form*, bool)));
		connect(KFormDesigner::FormManager::self(), TQT_SIGNAL(formWidgetSelected(KFormDesigner::Form*)), TQT_SLOT(slotFormWidgetSelected(KFormDesigner::Form*)));
		connect(KFormDesigner::FormManager::self(), TQT_SIGNAL(noFormSelected()), TQT_SLOT(slotNoFormSelected()));
		connect(KFormDesigner::FormManager::self(), TQT_SIGNAL(undoEnabled(bool, const TQString&)), TQT_SLOT(setUndoEnabled(bool, const TQString&)));
		connect(KFormDesigner::FormManager::self(), TQT_SIGNAL(redoEnabled(bool, const TQString&)), TQT_SLOT(setRedoEnabled(bool, const TQString&)));

		connect(KFormDesigner::FormManager::self(), TQT_SIGNAL(dirty(KFormDesigner::Form*, bool)), this, TQT_SLOT(slotFormModified(KFormDesigner::Form*, bool)));

		connect(KFormDesigner::FormManager::self(), TQT_SIGNAL(createFormSlot(KFormDesigner::Form*, const TQString&, const TQString&)),
			this, TQT_SLOT(slotCreateFormSlot(KFormDesigner::Form*, const TQString&, const TQString &)));
	}

	container->show();
	setWidget(container);
	connect(m_workspace, TQT_SIGNAL(windowActivated(TQWidget*)), KFormDesigner::FormManager::self(), TQT_SLOT(windowChanged(TQWidget*)));
	slotNoFormSelected();
}

KFormDesigner::WidgetLibrary* KFormDesignerKDevPart::formsLibrary()
{
	return static_formsLibrary;
}

void
KFormDesignerKDevPart::setupActions()
{
	KStdAction::open(this, TQT_SLOT(open()), actionCollection());
	KStdAction::openNew(this, TQT_SLOT(createBlankForm()), actionCollection());
	KStdAction::save(this, TQT_SLOT(save()), actionCollection());
	KStdAction::saveAs(this, TQT_SLOT(saveAs()), actionCollection());
	KStdAction::cut(KFormDesigner::FormManager::self(), TQT_SLOT(cutWidget()), actionCollection());
	KStdAction::copy(KFormDesigner::FormManager::self(), TQT_SLOT(copyWidget()), actionCollection());
	KStdAction::paste(KFormDesigner::FormManager::self(), TQT_SLOT(pasteWidget()), actionCollection());
	KStdAction::undo(KFormDesigner::FormManager::self(), TQT_SLOT(undo()), actionCollection());
	KStdAction::redo(KFormDesigner::FormManager::self(), TQT_SLOT(redo()), actionCollection());
	KStdAction::selectAll(KFormDesigner::FormManager::self(), TQT_SLOT(selectAll()), actionCollection());
	new TDEAction(i18n("Clear Widget Contents"), "edit-clear", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(clearWidgetContent()), actionCollection(), "clear_contents");
	new TDEAction(i18n("Delete Widget"), "edit-delete", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(deleteWidget()), actionCollection(), "edit_delete");
	new TDEAction(i18n("Preview Form"), "document-print-preview", "Ctrl+T", this, TQT_SLOT(slotPreviewForm()), actionCollection(), "preview_form");
	new TDEAction(i18n("Edit Tab Order"), "tab_order", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(editTabOrder()), actionCollection(), "taborder");
	new TDEAction(i18n("Edit Pixmap Collection"), "icons", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(editFormPixmapCollection()), actionCollection(), "pixmap_collection");
	new TDEAction(i18n("Edit Form Connections"), "connections", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(editConnections()), actionCollection(), "form_connections");

	new TDEAction(i18n("Lay Out Widgets &Horizontally"), TQString(), TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(layoutHBox()), actionCollection(), "layout_hbox");
	new TDEAction(i18n("Lay Out Widgets &Vertically"), TQString(), TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(layoutVBox()), actionCollection(), "layout_vbox");
	new TDEAction(i18n("Lay Out Widgets in &Grid"), TQString(), TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(layoutGrid()), actionCollection(), "layout_grid");
	new TDEAction(i18n("&Break Layout"), TQString(), TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(breakLayout()), actionCollection(), "break_layout");

	new TDEAction(i18n("Bring Widget to Front"), "raise", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(bringWidgetToFront()), actionCollection(), "format_raise");
	new TDEAction(i18n("Send Widget to Back"), "lower", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(sendWidgetToBack()), actionCollection(), "format_lower");

	TDEActionMenu *alignMenu = new TDEActionMenu(i18n("Align Widgets' Positions"), "aopos2grid", actionCollection(), "align_menu");
	alignMenu->insert( new TDEAction(i18n("To Left"), "aoleft", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(alignWidgetsToLeft()), actionCollection(), "align_to_left") );
	alignMenu->insert( new TDEAction(i18n("To Right"), "aoright", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(alignWidgetsToRight()), actionCollection(), "align_to_right") );
	alignMenu->insert( new TDEAction(i18n("To Top"), "aotop", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(alignWidgetsToTop()), actionCollection(), "align_to_top") );
	alignMenu->insert( new TDEAction(i18n("To Bottom"), "aobottom", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(alignWidgetsToBottom()), actionCollection(), "align_to_bottom") );
	alignMenu->insert( new TDEAction(i18n("To Grid"), "aopos2grid", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(alignWidgetsToGrid()), actionCollection(), "align_to_grid") );

	TDEActionMenu *sizeMenu = new TDEActionMenu(i18n("Adjust Widgets' Sizes"), "aogrid", actionCollection(), "adjust_size_menu");
	sizeMenu->insert( new TDEAction(i18n("To Fit"), "aofit", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(adjustWidgetSize()), actionCollection(), "adjust_to_fit") );
	sizeMenu->insert( new TDEAction(i18n("To Grid"), "aogrid", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(adjustSizeToGrid()), actionCollection(), "adjust_size_grid") );
	sizeMenu->insert( new TDEAction(i18n("To Shortest"), "aoshortest", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(adjustHeightToSmall()), actionCollection(), "adjust_height_small") );
	sizeMenu->insert( new TDEAction(i18n("To Tallest"), "aotallest", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(adjustHeightToBig()), actionCollection(), "adjust_height_big") );
	sizeMenu->insert( new TDEAction(i18n("To Narrowest"), "aonarrowest", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(adjustWidthToSmall()), actionCollection(), "adjust_width_small") );
	sizeMenu->insert( new TDEAction(i18n("To Widest"), "aowidest", TDEShortcut(0), KFormDesigner::FormManager::self(), TQT_SLOT(adjustWidthToBig()), actionCollection(), "adjust_width_big") );

	if(m_inShell)
		setXMLFile("kformdesigner_part_shell.rc");
	else
		setXMLFile("kformdesigner_part.rc");
	KFormDesigner::FormManager::self()->createActions(formsLibrary(), actionCollection(), this);
}

void
KFormDesignerKDevPart::createBlankForm()
{
	if(KFormDesigner::FormManager::self()->activeForm() && m_uniqueFormMode)
	{
		m_openingFile = true;
		closeURL();
		m_openingFile = false;
	}

	if(m_uniqueFormMode && KFormDesigner::FormManager::self()->activeForm() && !KFormDesigner::FormManager::self()->activeForm()->isModified() && KFormDesigner::FormManager::self()->activeForm()->filename().isNull())
		return;  // active form is already a blank one

	TQString n = i18n("Form") + TQString::number(++m_count);
	Form *form = new Form(formsLibrary(), n.latin1());
	FormWidgetBase *w = new FormWidgetBase(this, m_workspace, n.latin1());

	w->setCaption(n);
	w->setIcon(SmallIcon("form"));
	w->resize(350, 300);
	w->show();
	w->setFocus();

	form->createToplevel(w, w);
	KFormDesigner::FormManager::self()->importForm(form);
}

void
KFormDesignerKDevPart::open()
{
	m_openingFile = true;
	KURL url = KFileDialog::getOpenURL("::kformdesigner", i18n("*.ui|TQt Designer UI Files"), m_workspace->topLevelWidget());
	if(!url.isEmpty())
		ReadWritePart::openURL(url);
	m_openingFile = false;
}

bool
KFormDesignerKDevPart::openFile()
{
	Form *form = new Form(formsLibrary());
	FormWidgetBase *w = new FormWidgetBase(this, m_workspace);
	form->createToplevel(w, w);

	if(!KFormDesigner::FormIO::loadForm(form, w, m_file))
	{
		delete form;
		delete w;
		return false;
	}

	w->show();
	KFormDesigner::FormManager::self()->importForm(form, !isReadWrite());
	return true;
}

bool
KFormDesignerKDevPart::saveFile()
{
	KFormDesigner::FormIO::saveForm(KFormDesigner::FormManager::self()->activeForm(), m_file);
	return true;
}

void
KFormDesignerKDevPart::saveAs()
{
	KURL url = KFileDialog::getSaveURL("::kformdesigner", i18n("*.ui|TQt Designer UI Files"), m_workspace);
	if(url.isEmpty())
		return;
	else
		ReadWritePart::saveAs(url);
}

bool
KFormDesignerKDevPart::closeForm(Form *form)
{
	int res = KMessageBox::warningYesNoCancel( m_workspace->topLevelWidget(),
		i18n( "The form \"%1\" has been modified.\n"
		"Do you want to save your changes or discard them?" ).arg( form->objectTree()->name() ),
		i18n( "Close Form" ), KStdGuiItem::save(), KStdGuiItem::discard() );

	if(res == KMessageBox::Yes)
		save();

	return (res != KMessageBox::Cancel);
}

bool
KFormDesignerKDevPart::closeForms()
{
	TQWidgetList list = m_workspace->windowList(TQWorkspace::CreationOrder);
	for(TQWidget *w = list.first(); w; w = list.next())
		if(w->close() == false)
			return false;

	return true;
}

bool
KFormDesignerKDevPart::closeURL()
{
	if(!KFormDesigner::FormManager::self()->activeForm())
		return true;

	if(m_uniqueFormMode || !m_openingFile)
		return closeForms();

	return true;
}

void
KFormDesignerKDevPart::slotFormModified(Form *, bool isDirty)
{
	setModified(isDirty);
}

void
KFormDesignerKDevPart::slotPreviewForm()
{
	if(!KFormDesigner::FormManager::self()->activeForm())
		return;

	FormWidgetBase *w = new FormWidgetBase(this, m_workspace);
	KFormDesigner::FormManager::self()->previewForm(KFormDesigner::FormManager::self()->activeForm(), w);
}

void
KFormDesignerKDevPart::slotWidgetSelected(Form *form, bool multiple)
{
	enableFormActions();
	// Enable edit actions
	ENABLE_ACTION("edit_copy", true);
	ENABLE_ACTION("edit_cut", true);
	ENABLE_ACTION("edit_delete", true);
	ENABLE_ACTION("clear_contents", true);

	// 'Align Widgets' menu
	ENABLE_ACTION("align_menu", multiple);
	ENABLE_ACTION("align_to_left", multiple);
	ENABLE_ACTION("align_to_right", multiple);
	ENABLE_ACTION("align_to_top", multiple);
	ENABLE_ACTION("align_to_bottom", multiple);

	ENABLE_ACTION("adjust_size_menu", true);
	ENABLE_ACTION("adjust_width_small", multiple);
	ENABLE_ACTION("adjust_width_big", multiple);
	ENABLE_ACTION("adjust_height_small", multiple);
	ENABLE_ACTION("adjust_height_big", multiple);

	ENABLE_ACTION("format_raise", true);
	ENABLE_ACTION("format_lower", true);

	// If the widgets selected is a container, we enable layout actions
	if(!multiple)
	{
		KFormDesigner::ObjectTreeItem *item = form->objectTree()->lookup( form->selectedWidgets()->first()->name() );
		if(item && item->container())
			multiple = true;
	}
	// Layout actions
	ENABLE_ACTION("layout_hbox", multiple);
	ENABLE_ACTION("layout_vbox", multiple);
	ENABLE_ACTION("layout_grid", multiple);

	KFormDesigner::Container *container = KFormDesigner::FormManager::self()->activeForm()->activeContainer();
	ENABLE_ACTION("break_layout", (container->layoutType() != KFormDesigner::Container::NoLayout));
}

void
KFormDesignerKDevPart::slotFormWidgetSelected(Form *form)
{
	disableWidgetActions();
	enableFormActions();

	// Layout actions
	ENABLE_ACTION("layout_hbox", true);
	ENABLE_ACTION("layout_vbox", true);
	ENABLE_ACTION("layout_grid", true);
	ENABLE_ACTION("break_layout", (form->toplevelContainer()->layoutType() != KFormDesigner::Container::NoLayout));
}

void
KFormDesignerKDevPart::slotNoFormSelected()
{
	disableWidgetActions();

	// Disable paste action
	ENABLE_ACTION("edit_paste", false);

	ENABLE_ACTION("edit_undo", false);
	ENABLE_ACTION("edit_redo", false);

	// Disable 'Tools' actions
	ENABLE_ACTION("pixmap_collection", false);
	ENABLE_ACTION("form_connections", false);
	ENABLE_ACTION("taborder", false);
	ENABLE_ACTION("change_style", false);

	// Disable items in 'File'
	ENABLE_ACTION("file_save", false);
	ENABLE_ACTION("file_save_as", false);
	ENABLE_ACTION("preview_form", false);
}

void
KFormDesignerKDevPart::enableFormActions()
{
	// Enable 'Tools' actions
	ENABLE_ACTION("pixmap_collection", true);
	ENABLE_ACTION("form_connections", true);
	ENABLE_ACTION("taborder", true);
	ENABLE_ACTION("change_style", true);

	// Enable items in 'File'
	ENABLE_ACTION("file_save", true);
	ENABLE_ACTION("file_save_as", true);
	ENABLE_ACTION("preview_form", true);

	ENABLE_ACTION("edit_paste", KFormDesigner::FormManager::self()->isPasteEnabled());
	ENABLE_ACTION("edit_select_all", true);
}

void
KFormDesignerKDevPart::disableWidgetActions()
{
	// Disable edit actions
	ENABLE_ACTION("edit_copy", false);
	ENABLE_ACTION("edit_cut", false);
	ENABLE_ACTION("edit_delete", false);
	ENABLE_ACTION("clear_contents", false);

	// Disable format functions
	ENABLE_ACTION("align_menu", false);
	ENABLE_ACTION("align_to_left", false);
	ENABLE_ACTION("align_to_right", false);
	ENABLE_ACTION("align_to_top", false);
	ENABLE_ACTION("align_to_bottom", false);
	ENABLE_ACTION("adjust_size_menu", false);
	ENABLE_ACTION("format_raise", false);
	ENABLE_ACTION("format_lower", false);

	ENABLE_ACTION("layout_hbox", false);
	ENABLE_ACTION("layout_vbox", false);
	ENABLE_ACTION("layout_grid", false);
	ENABLE_ACTION("break_layout", false);
}

void
KFormDesignerKDevPart::setUndoEnabled(bool enabled, const TQString &text)
{
	TDEAction *undoAction = actionCollection()->action("edit_undo");
	if(undoAction)
	{
		undoAction->setEnabled(enabled);
		if(!text.isNull())
			undoAction->setText(text);
	}
}

void
KFormDesignerKDevPart::setRedoEnabled(bool enabled, const TQString &text)
{
	TDEAction *redoAction = actionCollection()->action("edit_redo");
	if(redoAction)
	{
		redoAction->setEnabled(enabled);
		if(!text.isNull())
			redoAction->setText(text);
	}
}

void
KFormDesignerKDevPart::slotCreateFormSlot(Form *form, const TQString &widget, const TQString &signal)
{
	Function f;
	f.returnType = "void";
	f.function = widget + "_" + signal;
	f.specifier = "non virtual";
	f.access = "public";
	f.type = ftTQtSlot;
	emit addedFunction(designerType(), form->objectTree()->name(), f);
}

KFormDesignerKDevPart::~KFormDesignerKDevPart()
{
}


//////  FormWidgetBase : helper widget to draw rects on top of widgets

//repaint all children widgets
static void repaintAll(TQWidget *w)
{
	TQObjectList *list = w->queryList("TQWidget");
	TQObjectListIt it(*list);
	for (TQObject *obj; (obj=it.current()); ++it ) {
		static_cast<TQWidget*>(obj)->repaint();
	}
	delete list;
}

void
FormWidgetBase::drawRects(const TQValueList<TQRect> &list, int type)
{
	TQPainter p;
	p.begin(this, true);
	bool unclipped = testWFlags( WPaintUnclipped );
	setWFlags( WPaintUnclipped );

	if (prev_rect.isValid()) {
		//redraw prev. selection's rectangle
		p.drawPixmap( TQPoint(prev_rect.x()-2, prev_rect.y()-2), buffer, TQRect(prev_rect.x()-2, prev_rect.y()-2, prev_rect.width()+4, prev_rect.height()+4));
	}
	p.setBrush(TQBrush::NoBrush);
	if(type == 1) // selection rect
		p.setPen(TQPen(white, 1, TQt::DotLine));
	else if(type == 2) // insert rect
		p.setPen(TQPen(white, 2));
	p.setRasterOp(XorROP);

	prev_rect = TQRect();
	TQValueList<TQRect>::ConstIterator endIt = list.constEnd();
	for(TQValueList<TQRect>::ConstIterator it = list.constBegin(); it != endIt; ++it) {
		p.drawRect(*it);
		prev_rect = prev_rect.unite(*it);
	}

	if (!unclipped)
		clearWFlags( WPaintUnclipped );
	p.end();
}

void
FormWidgetBase::drawRect(const TQRect& r, int type)
{
	TQValueList<TQRect> l;
	l.append(r);
	drawRects(l, type);
}

void
FormWidgetBase::initRect()
{
	repaintAll(this);
	buffer.resize( width(), height() );
	buffer = TQPixmap::grabWindow( winId() );
	prev_rect = TQRect();
}

void
FormWidgetBase::clearRect()
{
	TQPainter p;
	p.begin(this, true);
	bool unclipped = testWFlags( WPaintUnclipped );
	setWFlags( WPaintUnclipped );

	//redraw entire form surface
	p.drawPixmap( TQPoint(0,0), buffer, TQRect(0,0,buffer.width(), buffer.height()) );

	if (!unclipped)
		clearWFlags( WPaintUnclipped );
	p.end();

	repaintAll(this);
}

void
FormWidgetBase::highlightWidgets(TQWidget *from, TQWidget *to)//, const TQPoint &point)
{
	TQPoint fromPoint, toPoint;
	if(from && from->parentWidget() && (from != this))
		fromPoint = from->parentWidget()->mapTo(this, from->pos());
	if(to && to->parentWidget() && (to != this))
		toPoint = to->parentWidget()->mapTo(this, to->pos());

	TQPainter p;
	p.begin(this, true);
	bool unclipped = testWFlags( WPaintUnclipped );
	setWFlags( WPaintUnclipped );

	if (prev_rect.isValid()) {
		//redraw prev. selection's rectangle
		p.drawPixmap( TQPoint(prev_rect.x(), prev_rect.y()), buffer, TQRect(prev_rect.x(), prev_rect.y(), prev_rect.width(), prev_rect.height()));
	}

	p.setPen( TQPen(TQt::red, 2) );

	if(to)
	{
		TQPixmap pix1 = TQPixmap::grabWidget(from);
		TQPixmap pix2 = TQPixmap::grabWidget(to);

		if((from != this) && (to != this))
			p.drawLine( from->parentWidget()->mapTo(this, from->geometry().center()), to->parentWidget()->mapTo(this, to->geometry().center()) );

		p.drawPixmap(fromPoint.x(), fromPoint.y(), pix1);
		p.drawPixmap(toPoint.x(), toPoint.y(), pix2);

		if(to == this)
			p.drawRoundRect(2, 2, width()-4, height()-4, 4, 4);
		else
			p.drawRoundRect(toPoint.x(), toPoint.y(), to->width(), to->height(), 5, 5);
	}

	if(from == this)
		p.drawRoundRect(2, 2, width()-4, height()-4, 4, 4);
	else
		p.drawRoundRect(fromPoint.x(),  fromPoint.y(), from->width(), from->height(), 5, 5);

	if((to == this) || (from == this))
		prev_rect = TQRect(0, 0, buffer.width(), buffer.height());
	else if(to)
	{
		prev_rect.setX( (fromPoint.x() < toPoint.x()) ? (fromPoint.x() - 5) : (toPoint.x() - 5) );
		prev_rect.setY( (fromPoint.y() < toPoint.y()) ? (fromPoint.y() - 5) : (toPoint.y() - 5) );
		prev_rect.setRight( (fromPoint.x() < toPoint.x()) ? (toPoint.x() + to->width() + 10) : (fromPoint.x() + from->width() + 10) );
		prev_rect.setBottom( (fromPoint.y() < toPoint.y()) ? (toPoint.y() + to->height() + 10) : (fromPoint.y() + from->height() + 10) ) ;
	}
	else
		prev_rect = TQRect(fromPoint.x()- 5,  fromPoint.y() -5, from->width() + 10, from->height() + 10);

	if (!unclipped)
		clearWFlags( WPaintUnclipped );
	p.end();
}

void
FormWidgetBase::closeEvent(TQCloseEvent *ev)
{
	Form *form = KFormDesigner::FormManager::self()->formForWidget(this);
	if(!form || !form->isModified() || !form->objectTree()) // == preview form
		ev->accept();
	else
	{
		bool close = m_part->closeForm(form);
		if(close)
			ev->accept();
		else
			ev->ignore();
	}
}

K_EXPORT_COMPONENT_FACTORY(libkformdesigner_kdev_part, KFDFactory)

#include "kfd_kdev_part.moc"

