/*
 * krecord.cpp
 *
 * Copyright (C) 2004-2005 Christophe Thommeret <hftom@free.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <tqlayout.h>
#include <tqlabel.h>
#include <tqfile.h>
#include <tqdir.h>
#include <tqframe.h>

#include <tdelocale.h>
#include <kiconloader.h>
#include <tdemessagebox.h>

#include "krecord.h"



KRecord::KRecord( TQStringList chanList, TQPtrList<RecTimer> *t, TQWidget *parent, TQSize size ) : TQDialog( parent )
{
	timers = t;
	channelsList = chanList;

	TQVBoxLayout *vb = new TQVBoxLayout( 0, 0, 6 );
	TQLabel *lab = new TQLabel( i18n("Timers list:"), this );
	vb->addWidget( lab );

	list = new TDEListView( this );
	list->addColumn( "" );
	list->addColumn( i18n( "Name" ) );
	list->addColumn( i18n( "Channel" ) );
	list->addColumn( i18n( "Begin" ) );
	list->addColumn( i18n( "Duration" ) );
	list->setResizePolicy( TDEListView::AutoOneFit );
	list->setAllColumnsShowFocus( true );
	list->setFullWidth( true );
	TQListViewItem * item = new TQListViewItem( list, 0 );
	item->setText( 0, "00" );
	item->setText( 1, "Un nom assez long mais pas trop" );
	item->setText( 2, "Une chaine du meme calibre" );
	item->setText( 3, "00:00 00/00/00mm" );
	item->setText( 4, "00:00mm" );
	vb->addWidget( list );

	TQVBoxLayout *vb1 = new TQVBoxLayout( 0, 0, 6 );
	newBtn = new KPushButton(i18n("New"), this);
	vb1->addWidget( newBtn );
	editBtn = new KPushButton(i18n("Edit"), this);
	vb1->addWidget( editBtn );
	deleteBtn = new KPushButton(i18n("Delete"), this);
	vb1->addWidget( deleteBtn );
	vb1->addItem( new TQSpacerItem( 20, 20, TQSizePolicy::Minimum, TQSizePolicy::MinimumExpanding ) );

	TQHBoxLayout *hb = new TQHBoxLayout( 0, 0, 6 );
	hb->addLayout( vb );
	hb->addLayout( vb1 );

	TQHBoxLayout *hb1 = new TQHBoxLayout( 0 );
	hb1->addItem( new TQSpacerItem( 20, 20, TQSizePolicy::MinimumExpanding, TQSizePolicy::Minimum ) );
	okBtn = new KPushButton(i18n("Close"), this);
	hb1->addWidget( okBtn );

	TQVBoxLayout *vb2 = new TQVBoxLayout( this, 6, 6 );
	vb2->addLayout( hb );
	TQFrame *line = new TQFrame( this, "line1" );
	line->setFrameStyle( TQFrame::HLine );
	line->setFrameShadow( TQFrame::Sunken );
	line->setFrameShape( TQFrame::HLine );
	vb2->addWidget( line );
	vb2->addLayout( hb1 );

	TDEIconLoader *icon = new TDEIconLoader();
	newBtn->setGuiItem( KGuiItem(i18n("New"), icon->loadIconSet("document-new", TDEIcon::Small) ) );
	editBtn->setGuiItem( KGuiItem(i18n("Edit"), icon->loadIconSet("edit", TDEIcon::Small) ) );
	//editBtn->setEnabled( false );
	deleteBtn->setGuiItem( KGuiItem(i18n("Stop/Delete"), icon->loadIconSet("edit-cut", TDEIcon::Small) ) );
	okBtn->setGuiItem( KGuiItem(i18n("Close"), icon->loadIconSet("system-log-out", TDEIcon::Small) ) );

	isRecording = icon->loadIcon("player_record", TDEIcon::Small);
	yesRepeat = icon->loadIcon("reload", TDEIcon::Small);

	setCaption( i18n("Timers") );

	connect( okBtn, TQT_SIGNAL(clicked()), this, TQT_SLOT(accept()) );
	connect( newBtn, TQT_SIGNAL(clicked()), this, TQT_SLOT(newTimer()) );
	connect( deleteBtn, TQT_SIGNAL(clicked()), this, TQT_SLOT(deleteTimer()) );
	connect( editBtn, TQT_SIGNAL(clicked()), this, TQT_SLOT(editTimer()) );

	refresh();
	resize( size );
        delete icon;
}



void KRecord::refresh()
{
	TQListViewItem *after, *itt;
	RecTimer *ti;
	int i;
	TQString s;

	list->clear();
	list->setSorting( -1 );
	for ( i=0; i<(int)timers->count(); i++ ) {
		ti = timers->at(i);
		after = where( ti );
		/*if ( ti->running ) s = ">";
		else s = "";
		s = s+ti->name;*/
		s = ti->name;
		itt = new TDEListViewItem( (TQListView*)list, "", s, ti->channel, TDEGlobal::locale()->formatDateTime( ti->begin ), ti->duration.toString("hh:mm") );
		if ( ti->mode )
			itt->setPixmap( 0, yesRepeat );
		if ( ti->running )
			itt->setPixmap( 1, isRecording );
		if ( after )
			itt->moveItem( after );
	}
}



void KRecord::newTimer()
{
	TQListViewItem *itt, *after;
	RecTimer *t = new RecTimer();

	t->running = 0;
	t->mode = 0;

	KTimerEditor dlg( true, channelsList, *t, this );
	int ret=dlg.exec();
	if ( ret==KTimerEditor::Accepted ) {
		*t = dlg.timer;
		after = where( t, true );
		itt = new TDEListViewItem( (TQListView*)list, "", t->name, t->channel, TDEGlobal::locale()->formatDateTime( t->begin ), t->duration.toString("hh:mm") );
		if ( t->mode )
			itt->setPixmap( 0, yesRepeat );
		if ( t->running )
			itt->setPixmap( 1, isRecording );
		if ( after )
			itt->moveItem( after );
	}
	else delete t;
}



void KRecord::editTimer()
{
	TQListViewItem *it, *after;
	int i=0;
	RecTimer *t, *tn=0;

	it = list->firstChild();
	while ( it!=0 ) {
		if ( it->isSelected() )
			break;
		it = it->nextSibling();
	}
	if ( !it )
		return;

	for ( i=0; i<(int)timers->count(); i++ ) {
		t = timers->at(i);
		if ( t->name==it->text(1) && t->channel==it->text(2) ) {
			tn = t;
			break;
		}
	}

	if ( !tn )
		return;

	KTimerEditor dlg( false, channelsList, *tn, this );
	int ret=dlg.exec();
	if ( ret==KTimerEditor::Accepted ) {
		for ( i=0; i<(int)timers->count(); i++ ) {
			if ( timers->at(i)==tn ) {
				if ( tn->running ) {
					int ms = tn->begin.time().secsTo( TQTime::currentTime() );
					ms = ((dlg.timer.duration.hour()*3600+dlg.timer.duration.minute()*60)-ms)*1000;
					if ( ms<2000 )
						ms = 0;
					tn->duration = dlg.timer.duration;
					emit updateTimer( tn, ms );
					if ( ms )
						refresh();
				}
				else {
					timers->remove( tn );
					t = new RecTimer();
					*t = dlg.timer;
					after = where( t, true );
					refresh();
				}
				break;
			}
		}
	}
}



void KRecord::deleteTimer()
{
	TQListViewItem *it;
	int i=0, ret;
	TDEIconLoader *icon = new TDEIconLoader();

	it = list->firstChild();
       while ( it!=0 ) {
		if ( it->isSelected() )  {
			if ( timers->at(i)->mode )
				ret = KMessageBox::questionYesNoCancel( this, i18n("This timer is repeated. Do you want to skip the current job or delete the timer?"), i18n("Warning"), KGuiItem(i18n("Skip Current"), icon->loadIconSet("next", TDEIcon::Small) ), KGuiItem(i18n("Delete"), icon->loadIconSet("edit-cut", TDEIcon::Small) ) );
			else
				ret = KMessageBox::questionYesNo( this, i18n("Delete the selected timer?") );
			if ( ret==KMessageBox::Yes ) {
				emit updateTimer( timers->at(i), 0 );
			}
			else if ( ret==KMessageBox::No && timers->at(i)->mode ) {
				timers->at(i)->mode=0;
				emit updateTimer( timers->at(i), 0 );
			}
			break;
		}
		i++;
		it = it->nextSibling();
	}

	delete icon;
}



TQListViewItem* KRecord::where( RecTimer *rt, bool add )
{
	TQListViewItem *it, *ret=0;
	int i, r=0;
	RecTimer *t;

	for ( i=0; i<(int)timers->count(); i++ ) {
		t = timers->at(i);
		if ( rt->begin>t->begin )
			r=i+1;
	}
	if ( add )
		timers->insert( r, rt );
	if ( !r )
		return ret;

	it = list->firstChild();
	i=0;
	while ( it!=0 ) {
		if ( i==(r-1) ) {
			ret = it;
			break;
		}
		i++;
		it = it->nextSibling();
	}
	return ret;
}



void KRecord::accept()
{
	done( Accepted );
}



KRecord::~KRecord()
{
}

#include "krecord.moc"
