/***************************************************************************
    smb4k.cpp  -  The main class of Smb4K.
                             -------------------
    begin                : Sam Mär  1 14:57:21 CET 2003
    copyright            : (C) 2003-2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

// TQt includes
#include <tqvaluelist.h>
#include <tqpopupmenu.h>

// KDE includes
#include <kstatusbar.h>
#include <kstdaction.h>
#include <klibloader.h>
#include <tdemessagebox.h>
#include <kuniqueapplication.h>
#include <kiconloader.h>
#include <tdeactionclasses.h>
#include <tdelocale.h>
#include <kprogress.h>
#include <tdeconfigdialog.h>
#include <tdeaccel.h>

// application specific includes
#include "smb4k.h"
#include "smb4ksystemtray.h"
#include "dialogs/smb4kbookmarkeditor.h"
#include "core/smb4ksettings.h"
#include "core/smb4kcore.h"
#include "core/smb4tdeglobal.h"
#include "core/smb4kbookmark.h"
#include "core/smb4kshare.h"
#include "core/smb4kdefs.h"

using namespace Smb4TDEGlobal;


Smb4KApp::Smb4KApp( TQWidget *parent, const char *name )
: KParts::DockMainWindow( parent, name )
{
  m_system_tray = NULL;

  m_shares_part = NULL;
  m_browser_part = NULL;
  m_search_part = NULL;

  // Set XML file:
  setXMLFile( "smb4k_shell.rc" );

  // Set up the actions.
  setupActions();

  // Set up the status bar:
  setupStatusBar();

  // Set up the view:
  setupView();

  // Setup all things that need to be done *after* createGUI()
  // was run:
  TQPopupMenu *settings = static_cast<TQPopupMenu *>( TQT_TQWIDGET(child( "settings", "TQPopupMenu", true )) );

  if ( settings )
  {
    settings->insertItem( i18n( "&Dock Widgets" ), dockHideShowMenu(), 3, 3 );
  }

  slotSetupBookmarksMenu();

  // Set up the system tray window:
  setupSystemTray();

  // Resize main window (will be overwritten by configuration):
  resize( 800, 600 );

  // Apply main window settings:
  applyMainWindowSettings( Smb4KSettings::self()->config(), "MainWindow" );
  readDockConfig( Smb4KSettings::self()->config(), "MainWindow" );

  // We have to set the tab labels here. The writeDockConfig()
  // readDockConfig() combo will ignore changes of the localization,
  // because the first labels will be saved and then loaded again
  // and again.
  KDockWidget *dock = NULL;
  KDockTabGroup *parent_group = NULL;

  if ( (dock = manager()->getDockWidgetFromName( "NetworkBrowser" )) != NULL )
  {
    dock->setTabPageLabel( i18n( "Network Browser" ) );

    if ( (parent_group = dock->parentDockTabGroup()) != NULL )
    {
      parent_group->changeTab( dock, i18n( "Network Browser" ) );
    }
  }

  if ( (dock = manager()->getDockWidgetFromName( "SearchDialog" )) != NULL )
  {
    dock->setTabPageLabel( i18n( "Search Dialog" ) );

    if ( (parent_group = dock->parentDockTabGroup()) != NULL )
    {
      parent_group->changeTab( dock,  i18n( "Search Dialog" ) );
    }
  }

  // Connections
  connect( actionCollection(),           TQT_SIGNAL( actionHighlighted( TDEAction * ) ),
           this,                         TQT_SLOT( slotActionHighlighted( TDEAction * ) ) );

  connect( Smb4KCore::self(),            TQT_SIGNAL( runStateChanged() ),
           this,                         TQT_SLOT( slotRunStateChanged() ) );

  connect( Smb4KCore::bookmarkHandler(), TQT_SIGNAL( bookmarksUpdated() ),
           this,                         TQT_SLOT( slotSetupBookmarksMenu() ) );

  connect( Smb4KCore::mounter(),         TQT_SIGNAL( updated() ),
           this,                         TQT_SLOT( slotShareListUpdated() ) );
}


Smb4KApp::~Smb4KApp()
{
}


void Smb4KApp::setupActions()
{
  actionCollection()->setHighlightingEnabled( true );

  (void) KStdAction::quit( TQT_TQOBJECT(this), TQT_SLOT( slotQuit() ), actionCollection(), "quit_action" );

  // Set up the "Settings" menu:
  setStandardToolBarMenuEnabled( true );
  createStandardStatusBarAction();

  TDEActionMenu *view_modes  = new TDEActionMenu( i18n( "Shares Vie&w" ), "view_choose",
                             actionCollection(), "view_modes_menu" );

  TDERadioAction *icon_view  = new TDERadioAction( i18n( "&Icon View" ), "view_icon", CTRL+Key_I,
                             TQT_TQOBJECT(this), TQT_SLOT( slotChangeSharesView() ), actionCollection(), "icon_view_action" );
  icon_view->setExclusiveGroup( "SharesViewActions" );

  TDERadioAction *list_view  = new TDERadioAction( i18n( "List Vie&w" ), "view_detailed", CTRL+Key_W,
                             TQT_TQOBJECT(this), TQT_SLOT( slotChangeSharesView() ), actionCollection(), "list_view_action" );
  list_view->setExclusiveGroup( "SharesViewActions" );

  switch ( Smb4KSettings::sharesView() )
  {
    case Smb4KSettings::EnumSharesView::IconView:
    {
      icon_view->setChecked( true );

      break;
    }
    case Smb4KSettings::EnumSharesView::ListView:
    {
      list_view->setChecked( true );

      break;
    }
    default:
    {
      break;
    }
  }

  view_modes->insert( icon_view, -1 );
  view_modes->insert( list_view, -1 );

  (void) KStdAction::preferences( TQT_TQOBJECT(this), TQT_SLOT( slotConfigDialog() ), actionCollection(), "configure_action" );

  // Notes:
  // (1) Actions from the parts will be included by setupView().
  // (2) The list of dock widgets will be plugged into the menu
  //     by setupView().
  // (3) The bookmark menu is managed by slotSetupBookmarksMenu().
}


void Smb4KApp::setupStatusBar()
{
  // Insert the items:
  statusBar()->insertItem( i18n( "Ready." ), Message, 1, false );

  KProgress *progress = new KProgress( statusBar(), "StatusBarProgressBar", 0 );
  progress->setFixedWidth( 75 );
  progress->setFixedHeight( fontMetrics().height() );
  progress->setPercentageVisible( false );
  statusBar()->addWidget( progress, 0, true );

  statusBar()->insertFixedItem( TQString( "Smb4K %1" ).arg( VERSION ), Version, true );

  // Align the items:
  statusBar()->setItemAlignment( Message, AlignAuto );
  statusBar()->setItemAlignment( Progress, AlignCenter );
  statusBar()->setItemAlignment( Version, AlignAuto );
}


void Smb4KApp::setupView()
{
  //
  // Main widget (shares view):
  //
  KLibFactory *shares_factory = NULL;

  m_current_shares_view = Smb4KSettings::sharesView();

  switch ( Smb4KSettings::sharesView() )
  {
    case Smb4KSettings::EnumSharesView::IconView:
    {
      shares_factory = KLibLoader::self()->factory( "libsmb4ksharesiconview" );

      break;
    }
    case Smb4KSettings::EnumSharesView::ListView:
    {
      shares_factory = KLibLoader::self()->factory( "libsmb4kshareslistview" );

      break;
    }
    default:
    {
      break;
    }
  }

  if ( shares_factory )
  {
    m_shares_part = static_cast<KParts::Part *>( shares_factory->create( TQT_TQOBJECT(this), "SharesPart", "KParts::Part" ) );

    if ( m_shares_part )
    {
      KDockWidget *main = createDockWidget( "SharesView", SmallIcon( "drive-harddisk-mounted" ), 0L );
      main->setWidget( m_shares_part->widget() );
      main->setDockSite( KDockWidget::DockCorner );
      main->setEnableDocking( KDockWidget::DockNone );
      setView( main );
      setMainDockWidget( main );

      createGUI( m_shares_part );

      accel()->insert( i18n( "Jump to shares view" ), CTRL+Key_3, TQT_TQOBJECT(this), TQT_SLOT( slotJumpToSharesView() ), false, true );
    }
  }
  else
  {
    KMessageBox::error( 0, "<qt>"+KLibLoader::self()->lastErrorMessage()+"</qt>" );

    TDEApplication::exit( 0 );

    return;
  }

  //
  // Browser widget:
  //
  KLibFactory *browser_factory = KLibLoader::self()->factory( "libsmb4knetworkbrowser" );

  if ( browser_factory )
  {
    m_browser_part = static_cast<KParts::Part *>( browser_factory->create( TQT_TQOBJECT(this), "BrowserPart", "KParts::Part" ) );

    if ( m_browser_part )
    {
      KDockWidget *network = createDockWidget( "NetworkBrowser", SmallIcon( "network" ), 0L, i18n( "Network Browser" ) );
      network->setWidget( m_browser_part->widget() );
      network->manualDock( getMainDockWidget(), KDockWidget::DockLeft, 45 );

      factory()->addClient( m_browser_part );

      accel()->insert( i18n( "Jump to network browser" ), CTRL+Key_1, TQT_TQOBJECT(this), TQT_SLOT( slotJumpToNetworkBrowser() ), false, true );
    }
  }
  else
  {
    KMessageBox::error( 0, "<qt>"+KLibLoader::self()->lastErrorMessage()+"</qt>" );

    TDEApplication::exit( 0 );

    return;
  }

  //
  // Search dialog
  //
  KLibFactory *search_factory = KLibLoader::self()->factory( "libsmb4ksearchdialog" );

  if ( search_factory )
  {
    m_search_part = static_cast<KParts::Part *>( search_factory->create( TQT_TQOBJECT(this), "SearchDialogPart", "KParts::Part" ) );

    if ( m_search_part )
    {
      KDockWidget *search = createDockWidget( "SearchDialog", SmallIcon( "edit-find" ), 0L, i18n( "Search Dialog" ) );
      search->setWidget( m_search_part->widget() );

      KDockWidget *network = manager()->getDockWidgetFromName( "NetworkBrowser" );

      if ( network )
      {
        search->manualDock( network, KDockWidget::DockCenter, 0 );
      }
      else
      {
        search->manualDock( getMainDockWidget(), KDockWidget::DockLeft, 45 );
      }

      factory()->addClient( m_search_part );

      accel()->insert( i18n( "Jump to search dialog" ), CTRL+Key_2, TQT_TQOBJECT(this), TQT_SLOT( slotJumpToSearchDialog() ), false, true );
    }
  }
  else
  {
    KMessageBox::error( 0, "<qt>"+KLibLoader::self()->lastErrorMessage()+"</qt>" );

    TDEApplication::exit( 0 );

    return;
  }
}


void Smb4KApp::setupSystemTray()
{
  if ( !m_system_tray )
  {
    m_system_tray = new Smb4KSystemTray( this, "SystemTray" );
  }

  connect( m_system_tray, TQT_SIGNAL( quitSelected() ),
           this,          TQT_SLOT( slotQuit() ) );

  connect( m_system_tray, TQT_SIGNAL( settingsChanged() ),
           this,          TQT_SLOT( slotSettingsChanged() ) );

  m_system_tray->embed( Smb4KSettings::embedIntoSystemTray() );
}


void Smb4KApp::changeSharesView()
{
  // Change the information about the current view:
  m_current_shares_view = Smb4KSettings::sharesView();

  // Save dock widget settings:
  writeDockConfig( Smb4KSettings::self()->config(), "MainWindow" );

  // Clear the shares view
  factory()->removeClient( m_shares_part );

  KDockWidget *dock = NULL;

  if ( (dock = manager()->getDockWidgetFromName( "SharesView" )) != NULL )
  {
    delete dock;
  }

  // Load the new shares view:
  KLibFactory *shares_factory = NULL;

  switch ( Smb4KSettings::sharesView() )
  {
    case Smb4KSettings::EnumSharesView::IconView:
    {
      shares_factory = KLibLoader::self()->factory( "libsmb4ksharesiconview" );

      break;
    }
    case Smb4KSettings::EnumSharesView::ListView:
    {
      shares_factory = KLibLoader::self()->factory( "libsmb4kshareslistview" );

      break;
    }
    default:
    {
      break;
    }
  }

  if ( shares_factory )
  {
    m_shares_part = static_cast<KParts::Part *>( shares_factory->create( TQT_TQOBJECT(this), "SharesPart", "KParts::Part" ) );

    if ( m_shares_part )
    {
      KDockWidget *main = createDockWidget( "SharesView", SmallIcon( "drive-harddisk-mounted" ), this );
      main->setWidget( m_shares_part->widget() );
      main->setDockSite( KDockWidget::DockCorner );
      main->setEnableDocking( KDockWidget::DockNone );
      setView( main );
      setMainDockWidget( main );

      factory()->addClient( m_shares_part );
    }
  }
  else
  {
    KMessageBox::error( 0, "<qt>"+KLibLoader::self()->lastErrorMessage()+"</qt>" );

    TDEApplication::exit( 0 );

    return;
  }

  // Apply the settings to the dock widgets:
  readDockConfig( Smb4KSettings::self()->config(), "MainWindow" );
}


bool Smb4KApp::queryExit()
{
  Smb4KSettings::setStartMainWindowDocked( !isVisible() );

  saveMainWindowSettings( Smb4KSettings::self()->config(), "MainWindow" );
  writeDockConfig( Smb4KSettings::self()->config(), "MainWindow" );

  hide();

  if ( m_system_tray )
  {
    delete m_system_tray;
    m_system_tray = NULL;
  }

  // Save any options here that have to be written directly
  // before the application exits:

  return true;
}


bool Smb4KApp::queryClose()
{
  if ( !kapp->sessionSaving() && isVisible() && m_system_tray->isEmbedded() &&
       Smb4KSettings::embedIntoSystemTray() )
  {
    // This part has been 'stolen' from JuK application.
    KMessageBox::information(this,
            i18n( "Closing the main window will keep Smb4K running in the system tray. "
                  "Use \"Quit\" from the \"File\" menu to quit the application."),
            i18n( "Docking in System Tray" ), "ClosingMainWindowInfo" );

    hide();

    return false;

  }
  else
  {
    return true;
  }
}


void Smb4KApp::timerEvent( TQTimerEvent * )
{
  KProgress *progress_bar = static_cast<KProgress *>( TQT_TQWIDGET(child( "StatusBarProgressBar", "KProgress", true )) );

  if ( progress_bar )
  {
    progress_bar->setProgress( progress_bar->progress() + 1 );
  }
}


/////////////////////////////////////////////////////////////////////
// TQT_SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////

void Smb4KApp::slotQuit()
{
  statusBar()->changeItem( i18n( "Exiting..." ), Message );

  if ( m_system_tray->isEmbedded() )
  {
    m_system_tray->embed( false );
  }

  TDEApplication::exit( 0 );
}


void Smb4KApp::slotChangeSharesView()
{
  // Let's be sure that the last action that was highlighted is
  // indeed an action that changes the shares view:
  TQString action_name;

  if ( m_action )
  {
    action_name = TQString( m_action->name() );

    if ( TQString::compare( action_name, "icon_view_action" ) != 0 &&
         TQString::compare( action_name, "list_view_action" ) != 0 )
    {
      return;
    }
  }
  else
  {
    return;
  }

  TDERadioAction *shares_view_action = static_cast<TDERadioAction *>( m_action );

  // Set the check mark for the action:
  if ( !shares_view_action->isChecked() )
  {
    shares_view_action->setChecked( true );
  }
  else
  {
    // Do nothing
  }

  // Change the shares view setting and save it:
  if ( TQString::compare( action_name, "icon_view_action" ) == 0 )
  {
    Smb4KSettings::setSharesView( Smb4KSettings::EnumSharesView::IconView );
  }
  else if ( TQString::compare( action_name, "list_view_action" ) == 0 )
  {
    Smb4KSettings::setSharesView( Smb4KSettings::EnumSharesView::ListView );
  }
  else
  {
    return;
  }

  Smb4KSettings::writeConfig();

  changeSharesView();
}


void Smb4KApp::slotConfigDialog()
{
  // If the config dialog is already created and cached,
  // we do not create a new one but show the old instead:
  if ( TDEConfigDialog::showDialog( "ConfigDialog" ) )
  {
    return;
  }

  // Load the factory of the config dialog:
  KLibFactory *config_factory = KLibLoader::self()->factory( "libsmb4tdeconfigdialog" );

  if ( config_factory )
  {
    TDEConfigDialog *dlg = static_cast<TDEConfigDialog *>( TQT_TQWIDGET(config_factory->create( TQT_TQOBJECT(this), "ConfigDialog", "TDEConfigDialog" )) );

    if ( dlg )
    {
      connect( dlg, TQT_SIGNAL( settingsChanged() ), TQT_TQOBJECT(this), TQT_SLOT( slotSettingsChanged() ) );

      dlg->show();
    }
  }
  else
  {
    KMessageBox::error( 0, "<qt>"+KLibLoader::self()->lastErrorMessage()+"</qt>" );

    return;
  }
}


void Smb4KApp::slotSettingsChanged()
{
  // Adjust settings for the system tray widget:
  if ( m_system_tray )
  {
    m_system_tray->loadSettings();
    m_system_tray->embed( Smb4KSettings::embedIntoSystemTray() );
  }

  // (Re-)Load the correct shares view.
  if ( m_current_shares_view != Smb4KSettings::sharesView() )
  {
    changeSharesView();
  }
  else
  {
    // Do nothing
  }

  // Notify the parts to reload the settings.
  // Note: TQApplication::postEvent() will delete the TQCustomEvent pointers.
  TQApplication::postEvent( m_browser_part, new TQCustomEvent( EVENT_LOAD_SETTINGS ) );
  TQApplication::postEvent( m_search_part, new TQCustomEvent( EVENT_LOAD_SETTINGS ) );
  TQApplication::postEvent( m_shares_part, new TQCustomEvent( EVENT_LOAD_SETTINGS ) );
}


void Smb4KApp::slotBookmarkEditor()
{
  Smb4KBookmarkEditor *dlg = static_cast<Smb4KBookmarkEditor *>( TQT_TQWIDGET(child( "BookmarkEditor", "Smb4KBookmarkEditor", true )) );

  if ( !dlg )
  {
    dlg = new Smb4KBookmarkEditor( this, "BookmarkEditor" );
  }

  dlg->show();
}


void Smb4KApp::slotRunStateChanged()
{
  // Get the progress bar:
  KProgress *progress_bar = static_cast<KProgress *>( TQT_TQWIDGET(child( "StatusBarProgressBar", "KProgress", true )) );

  // Clear the status bar:
  statusBar()->clear();

  // Set the status bar message:
  switch( Smb4KCore::currentState() )
  {
    case SCANNER_INIT:
    {
      switch ( Smb4KSettings::browseList() )
      {
        case Smb4KSettings::EnumBrowseList::LookupDomains:
        {
          statusBar()->changeItem( i18n( "Looking up workgroups and domains..." ), Message );

          break;
        }
        case Smb4KSettings::EnumBrowseList::QueryCurrentMaster:
        {
          statusBar()->changeItem( i18n( "Querying current master browser..." ), Message );

          break;
        }
        case Smb4KSettings::EnumBrowseList::QueryCustomMaster:
        {
          statusBar()->changeItem( i18n( "Querying master browser %1..." ).arg( Smb4KSettings::customMasterBrowser().upper() ), Message );

          break;
        }
        case Smb4KSettings::EnumBrowseList::ScanBroadcastAreas:
        {
          statusBar()->changeItem( i18n( "Scanning broadcast areas..." ), Message );

          break;
        }
        default:
        {
          break;
        }
      }

      break;
    }
    case SCANNER_OPENING_WORKGROUP:
    {
      statusBar()->changeItem( i18n( "Opening workgroup..." ), Message );

      break;
    }
    case SCANNER_OPENING_HOST:
    {
      statusBar()->changeItem( i18n( "Retrieving list of shares..." ), Message );

      break;
    }
    case SCANNER_RETRIEVING_INFO:
    {
      statusBar()->changeItem( i18n( "Retrieving additional information..." ), Message );

      break;
    }
    case SCANNER_SEARCHING:
    {
      statusBar()->changeItem( i18n( "Searching..." ), Message );

      break;
    }
    case SCANNER_RETRYING_OPENING_HOST:
    {
      statusBar()->changeItem( i18n( "Retrying to retrieve list of shares..." ), Message );

      break;
    }
    case SCANNER_STOP:
    {
      statusBar()->changeItem( i18n( "Done." ), Message );

      break;
    }
    case MOUNTER_MOUNTING:
    {
      statusBar()->changeItem( i18n( "Mounting share..." ), Message );

      break;
    }
    case MOUNTER_UNMOUNTING:
    {
      statusBar()->changeItem( i18n( "Unmounting share..." ), Message );

      break;
    }
    case MOUNTER_STOP:
    {
      statusBar()->changeItem( i18n( "Done." ), Message );

      break;
    }
    case PRINT_START:
    {
      statusBar()->changeItem( i18n( "Printing file..." ), Message );

      break;
    }
    case PRINT_STOP:
    {
      statusBar()->changeItem( i18n( "Done." ), Message );

      break;
    }
    case SYNCHRONIZER_START:
    {
      statusBar()->changeItem( i18n( "Synchronizing data..." ), Message );

      break;
    }
    case SYNCHRONIZER_STOP:
    {
      statusBar()->changeItem( i18n( "Done." ), Message );

      break;
    }
    case PREVIEWER_START:
    {
      statusBar()->changeItem( i18n( "Generating preview..." ), Message );

      break;
    }
    case PREVIEWER_STOP:
    {
      statusBar()->changeItem( i18n( "Done." ), Message );

      break;
    }
    case CORE_STOP:
    {
      statusBar()->changeItem( i18n( "Ready." ), Message );

      break;
    }
    default:
    {
      break;
    }
  }

  // Set up the progress bar:
  if ( Smb4KCore::isRunning() )
  {
    // If the progress bar exists and is already set to
    // a totalSteps == 0, we do not need to do anything
    // here.
    if ( progress_bar && progress_bar->totalSteps() != 0 )
    {
      progress_bar->setTotalSteps( 0 );

      m_timer_id = startTimer( TIMER_INTERVAL );
    }
  }
  else
  {
    if ( progress_bar )
    {
      killTimer( m_timer_id );

      progress_bar->setTotalSteps( 100 );
      progress_bar->reset();
    }
  }
}


void Smb4KApp::slotSetupBookmarksMenu()
{
  // Set up bookmark related actions:
  TQValueList<Smb4KBookmark *> bookmarks = Smb4KCore::bookmarkHandler()->getBookmarks();

  if ( !actionCollection()->action( "edit_bookmarks_action" ) )
  {
    unplugActionList( "bookmark_actions" );

    TQPtrList<TDEAction> bookmark_actions;

    // Create the "Edit Bookmarks" action:
    bookmark_actions.append( new TDEAction( i18n( "&Edit Bookmarks" ), "bookmark", CTRL+Key_E,
                             TQT_TQOBJECT(this), TQT_SLOT( slotBookmarkEditor() ), actionCollection(),
                             "edit_bookmarks_action" ) );

    // Get the "Add Bookmark" action from the browser:
    TQPtrList<KXMLGUIClient> clients_list = factory()->clients();

    for ( TQPtrList<KXMLGUIClient>::Iterator it = clients_list.begin(); it != clients_list.end(); ++it )
    {
      if ( (*it)->action( "bookmark_action" ) )
      {
        bookmark_actions.append( (*it)->action( "bookmark_action" ) );

        break;
      }
      else
      {
        continue;
      }
    }

    bookmark_actions.append( new TDEActionSeparator() );

    plugActionList( "bookmark_actions", bookmark_actions );
  }

  // Unplug the action list:
  unplugActionList( "bookmarks" );

  // Get the bookmark action list and delete all entries. We could
  // also try to keep those actions that are not obsolete, but I think
  // this is the cleanest way.
  TDEActionPtrList list = actionCollection()->actions( "Bookmarks" );

  for ( TDEActionPtrList::Iterator it = list.begin(); it != list.end(); ++it )
  {
    actionCollection()->remove( *it );
  }

  // Set up the menu:
  if ( bookmarks.isEmpty() )
  {
    actionCollection()->action( "edit_bookmarks_action" )->setEnabled( false );
  }
  else
  {
    actionCollection()->action( "edit_bookmarks_action" )->setEnabled( true );

    // Work around sorting problems:
    TQStringList display_strings;

    for ( TQValueListIterator<Smb4KBookmark *> it = bookmarks.begin(); it != bookmarks.end(); ++it )
    {
      if ( !(*it)->label().isEmpty() && Smb4KSettings::showCustomBookmarkLabel() )
      {
        display_strings.append( (*it)->label() );
      }
      else
      {
        display_strings.append( (*it)->bookmark() );
      }
    }

    display_strings.sort();

    // Create the bookmark list and plug it into the menu:
    TQPtrList<TDEAction> bookmark_list;

    for ( TQStringList::ConstIterator it = display_strings.begin(); it != display_strings.end(); ++it )
    {
      TDEAction *a = new TDEAction( *it, "folder", TDEShortcut::null(), 0, 0, actionCollection(), *it );
      a->setGroup( "Bookmarks" );
      connect( a, TQT_SIGNAL( activated() ), TQT_TQOBJECT(this), TQT_SLOT( slotBookmarkActivated() ) );
      bookmark_list.append( a );
    }

    plugActionList( "bookmarks", bookmark_list );
  }
}


void Smb4KApp::slotBookmarkActivated()
{
  if ( m_action && TQString::compare( m_action->group(), "Bookmarks" ) == 0 )
  {
    Smb4KBookmark *bookmark = NULL;

    if ( !m_action->plainText().startsWith( "//" ) )
    {
      bookmark = Smb4KCore::bookmarkHandler()->findBookmarkByLabel( m_action->plainText() );
    }
    else
    {
      bookmark = Smb4KCore::bookmarkHandler()->findBookmarkByName( m_action->plainText() );
    }

    if ( bookmark )
    {
      Smb4KCore::mounter()->mountShare( bookmark->workgroup(), bookmark->host(),
                                        bookmark->ip(), bookmark->share() );
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KApp::slotShareListUpdated()
{
  TDEActionPtrList list = actionCollection()->actions( "Bookmarks" );

  if ( !list.isEmpty() )
  {
    TQString name;

    for ( TDEActionPtrList::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      if ( !(*it)->plainText().startsWith( "//" ) )
      {
        Smb4KBookmark *bookmark = Smb4KCore::bookmarkHandler()->findBookmarkByLabel( (*it)->plainText() );

        name = bookmark->bookmark();
      }
      else
      {
        name = (*it)->plainText();
      }

      TQValueList<Smb4KShare> share_list = Smb4KCore::mounter()->findShareByName( name );

      bool enable = true;

      for ( TQValueList<Smb4KShare>::ConstIterator i = share_list.begin(); i != share_list.end(); ++i )
      {
        if ( !(*i).isForeign() )
        {
          enable = false;

          break;
        }
        else
        {
          continue;
        }
      }

      (*it)->setEnabled( enable );
    }
  }
}


void Smb4KApp::slotActionHighlighted( TDEAction *action )
{
  m_action = action;
}


void Smb4KApp::slotJumpToNetworkBrowser()
{
  KDockWidget *dock = NULL;

  if ( (dock = manager()->getDockWidgetFromName( "NetworkBrowser" )) != NULL &&
       dock->isHidden() )
  {
    manager()->getDockWidgetFromName( "NetworkBrowser" )->changeHideShowState();
  }

  // Send a custom focus event to the browser, that tells it to set
  // the focus to the list view widget.
  TQApplication::postEvent( m_browser_part, new TQCustomEvent( EVENT_SET_FOCUS ) );
}


void Smb4KApp::slotJumpToSearchDialog()
{
  KDockWidget *dock = NULL;

  if ( (dock = manager()->getDockWidgetFromName( "SearchDialog" )) != NULL &&
       dock->isHidden() )
  {
    manager()->getDockWidgetFromName( "SearchDialog" )->changeHideShowState();
  }

  // Send a custom focus event to the search dialog, that tells it
  // to set the focus to the combo box in the tool bar.
  TQApplication::postEvent( m_search_part, new TQCustomEvent( EVENT_SET_FOCUS ) );
}


void Smb4KApp::slotJumpToSharesView()
{
  // Send a custom focus event to the shares view, that tells it
  // to set the focus to its main widget.
  TQApplication::postEvent( m_shares_part, new TQCustomEvent( EVENT_SET_FOCUS ) );
}

#include "smb4k.moc"

