/***************************************************************************
    smb4ksynchronizationdialog  -  The synchronization dialog of Smb4K
                             -------------------
    begin                : Sa Mai 19 2007
    copyright            : (C) 2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

// TQt includes
#include <tqlayout.h>
#include <tqframe.h>
#include <tqlabel.h>

// KDE includes
#include <tdelocale.h>
#include <kguiitem.h>
#include <kdebug.h>
#include <kurlrequester.h>
#include <klineedit.h>
#include <kprogress.h>

// application specific includes
#include "smb4ksynchronizationdialog.h"
#include "../core/smb4kshare.h"
#include "../core/smb4kcore.h"
#include "../core/smb4ksynchronizationinfo.h"
#include "../core/smb4ksettings.h"

Smb4KSynchronizationDialog::Smb4KSynchronizationDialog( Smb4KShare *share, TQWidget *parent, const char *name )
: KDialogBase( Plain, i18n( "Synchronization" ), User2|User1|Cancel, User1, parent, name, false, true ),
m_share( share )
{
  setWFlags( TQt::WDestructiveClose );

  setButtonGuiItem( User1, KGuiItem( i18n( "Synchronize" ), "go-bottom", i18n( "Synchronize the destination with the source" ) ) );
  setButtonGuiItem( User2, KGuiItem( i18n( "Swap Paths" ), TQString(), i18n( "Swap source and destination" ) ) );

  TQFrame *frame = plainPage();
  TQGridLayout *layout = new TQGridLayout( frame );
  layout->setSpacing( 5 );
  layout->setMargin( 0 );

  TQLabel *source_label =       new TQLabel( i18n( "Source:" ), frame, "SourceURLLabel" );
  KURLRequester *source =      new KURLRequester( m_share->path()+"/", frame, "SourceURL" );
  source->setShowLocalProtocol( false );
  source->setMode( KFile::Directory | KFile::LocalOnly );

  TQLabel *destination_label =  new TQLabel( i18n( "Destination:" ), frame, "DestinationURLLabel" );
  KURLRequester *destination = new KURLRequester( Smb4KSettings::rsyncPrefix(), frame, "DestinationURL" );
  destination->setShowLocalProtocol( false );
  destination->setMode( KFile::Directory | KFile::LocalOnly );

  KLineEdit *current_file =      new KLineEdit( TQString(), frame, "ProgressInfo" );
  current_file->setEnableSqueezedText( true );
  current_file->setReadOnly( true );

  KProgress *individual =         new KProgress( frame, "IndividualProgress", 0 );
  individual->setEnabled( false );

  KProgress *total =           new KProgress( frame, "TotalProgress", 0 );
  total->setEnabled( false );

  TQWidget *transfer_widget =   new TQWidget( frame, "TransferInfoWidget" );
  TQGridLayout *trans_layout =  new TQGridLayout( transfer_widget );
  trans_layout->setSpacing( 5 );
  trans_layout->setMargin( 0 );

  TQLabel *file_label =         new TQLabel( i18n( "Files transferred:" ), transfer_widget,
                               "FilesTransferredLabel" );
  TQLabel *file_trans_label =   new TQLabel( "0 / 0", transfer_widget,
                               "FilesTransferred" );

  TQLabel *rate_label =         new TQLabel( i18n( "Transfer rate:" ), transfer_widget,
                               "TransferRateLabel" );
  TQLabel *trans_rate_label =   new TQLabel( "0.00 kB/s", transfer_widget,
                               "TransferRate" );

  trans_layout->addWidget( file_label, 0, 0, 0 );
  trans_layout->addWidget( file_trans_label, 0, 1, TQt::AlignRight );
  trans_layout->addWidget( rate_label, 1, 0, 0 );
  trans_layout->addWidget( trans_rate_label, 1, 1, TQt::AlignRight );

  transfer_widget->setEnabled( false );

  layout->addWidget( source_label, 0, 0, 0 );
  layout->addWidget( source, 0, 1, 0 );
  layout->addWidget( destination_label, 1, 0, 0 );
  layout->addWidget( destination, 1, 1, 0 );
  layout->addMultiCellWidget( current_file, 2, 2, 0, 1, 0 );
  layout->addMultiCellWidget( individual, 3, 3, 0, 1, 0 );
  layout->addMultiCellWidget( total, 4, 4, 0, 1, 0 );
  layout->addMultiCellWidget( transfer_widget, 5, 6, 0, 1, 0 );

  // Connections
  connect( Smb4KCore::synchronizer(), TQT_SIGNAL( progress( const Smb4KSynchronizationInfo & ) ),
           this,                      TQT_SLOT( slotProgress( const Smb4KSynchronizationInfo & ) ) );

  connect( Smb4KCore::synchronizer(), TQT_SIGNAL( finished() ),
           this,                      TQT_SLOT( slotSynchronizationFinished() ) );


  setFixedSize( (sizeHint().width() > 350 ? sizeHint().width() : 350), sizeHint().height() );
}


Smb4KSynchronizationDialog::~Smb4KSynchronizationDialog()
{
  // Do *not* delete the share object here.
}


/////////////////////////////////////////////////////////////////////////////
//   TQT_SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


void Smb4KSynchronizationDialog::slotUser1()
{
  // Synchronize!

  // Disable the URL requesters but in a way, that the information
  // proviced in them is still readable:
  KURLRequester *source = static_cast<KURLRequester *>( TQT_TQWIDGET(child( "SourceURL", "KURLRequester", true )) );
  source->lineEdit()->setReadOnly( true );
  source->button()->setEnabled( false );

  KURLRequester *destination = static_cast<KURLRequester *>( TQT_TQWIDGET(child( "DestinationURL", "KURLRequester", true )) );
  destination->lineEdit()->setReadOnly( true );
  destination->button()->setEnabled( false );

  TQWidget *transfer_widget = static_cast<TQWidget *>( TQT_TQWIDGET(child( "TransferInfoWidget", "TQWidget", true )) );
  transfer_widget->setEnabled( true );

  enableButton( User1, false );
  enableButton( User2, false );

  // Enable the progress bars and the information widgets:
  static_cast<KProgress *>( TQT_TQWIDGET(child( "IndividualProgress", "KProgress", true )) )->setEnabled( true );
  static_cast<KProgress *>( TQT_TQWIDGET(child( "TotalProgress", "KProgress", true )) )->setEnabled( true );

  Smb4KCore::synchronizer()->synchronize( source->url(), destination->url() );
}


void Smb4KSynchronizationDialog::slotUser2()
{
  // Swap URLs.

  KURLRequester *source = static_cast<KURLRequester *>( TQT_TQWIDGET(child( "SourceURL", "KURLRequester", true )) );
  KURLRequester *destination = static_cast<KURLRequester *>( TQT_TQWIDGET(child( "DestinationURL", "KURLRequester", true )) );

  TQString sourceURL = source->url();
  TQString destinationURL = destination->url();

  source->setURL( destinationURL );
  destination->setURL( sourceURL );
}


void Smb4KSynchronizationDialog::slotCancel()
{
  Smb4KCore::synchronizer()->abort();

  KDialogBase::slotCancel();
}


void Smb4KSynchronizationDialog::slotProgress( const Smb4KSynchronizationInfo &info )
{
  KLineEdit *progress =   static_cast<KLineEdit *>( TQT_TQWIDGET(child( "ProgressInfo", "KLineEdit", true )) );
  KProgress *individual = static_cast<KProgress *>( TQT_TQWIDGET(child( "IndividualProgress", "KProgress", true )) );
  KProgress *total =      static_cast<KProgress *>( TQT_TQWIDGET(child( "TotalProgress", "KProgress", true )) );
  TQLabel *transferred =   static_cast<TQLabel *>( TQT_TQWIDGET(child( "FilesTransferred", "TQLabel", true )) );
  TQLabel *rate =          static_cast<TQLabel *>( TQT_TQWIDGET(child( "TransferRate", "TQLabel", true )) );

  if ( !info.text().isEmpty() )
  {
    progress->setSqueezedText( info.text() );
  }

  if ( info.individualProgress() != -1 )
  {
    individual->setProgress( info.individualProgress() );
  }

  if ( info.totalProgress() != -1 )
  {
    total->setProgress( info.totalProgress() );
  }

  if ( info.totalFileNumber() != -1 && info.processedFileNumber() != -1 )
  {
    transferred->setText( TQString( "%1 / %2" ).arg( info.processedFileNumber() ).arg( info.totalFileNumber() ) );
  }

  if ( !info.transferRate().isEmpty() )
  {
    rate->setText( info.transferRate() );
  }
}


void Smb4KSynchronizationDialog::slotSynchronizationFinished()
{
  KProgress *individual = static_cast<KProgress *>( TQT_TQWIDGET(child( "IndividualProgress", "KProgress", true )) );
  KProgress *total =      static_cast<KProgress *>( TQT_TQWIDGET(child( "TotalProgress", "KProgress", true )) );

  if ( individual && individual->progress() != 100 )
  {
    individual->setProgress( 100 );
  }

  if ( total && total->progress() != 100 )
  {
    total->setProgress( 100 );
  }

  // Change the "Cancel" button to a "Close" button.
  setButtonGuiItem( Cancel, KStdGuiItem::close() );
}

#include "smb4ksynchronizationdialog.moc"
