/*
 * This file is part of the Polkit-tqt project
 * Copyright (C) 2009 Daniel Nicoletti <dantti85-pk@yahoo.com.br>
 * Copyright (C) 2009 Dario Freddi <drf@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB. If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "polkit-tqt-gui-actionbutton.h"
#include "polkit-tqt-gui-actionbutton_p.h"

#include <tqbutton.h>
#include <tqstring.h>
#include <tqvaluelist.h>


namespace PolkitTQt
{

namespace Gui
{

//--------------------------------------
// ActionButtonPrivate
//--------------------------------------

void ActionButtonPrivate::addButton(TQButton *button)
{
  buttons.append(button);
  TQObject::connect(button, TQT_SIGNAL(clicked()), q, TQT_SLOT(streamClicked()));
  TQObject::connect(q, TQT_SIGNAL(toggled(bool)), button, TQT_SLOT(toggle()));
  q->updateButton();
}

void ActionButtonPrivate::removeButton(TQButton *button)
{
  if (buttons.contains(button))
  {
    TQObject::disconnect(button, SIGNAL(clicked(bool)), q, SLOT(streamClicked()));
    TQObject::disconnect(q, SIGNAL(toggled(bool)), button, SLOT(toggle()));
    TQValueList<TQButton*>::iterator butIt = buttons.find(button);
    if (butIt != buttons.end())
    {
      buttons.remove(butIt);
    }
  }
}

//--------------------------------------
// ActionButton
//--------------------------------------

ActionButton::ActionButton(ActionButtonPrivate &dd, const TQString &actionId, TQObject *parent)
        : Action(actionId, parent), d(&dd)
{
  d->q = this;
  connect(this, TQT_SIGNAL(dataChanged()), this, TQT_SLOT(updateButton()));
}

ActionButton::ActionButton(TQButton *button, const TQString &actionId, TQObject *parent)
        : Action(actionId, parent), d(new ActionButtonPrivate())
{
  d->q = this;
  d->buttons.append(button);

  setButton(button);
  connect(this, TQT_SIGNAL(dataChanged()), this, TQT_SLOT(updateButton()));
}

ActionButton::~ActionButton()
{
  delete d;
}

void ActionButton::streamClicked()
{
  emit clicked(::tqt_cast<TQButton*>(this->sender()));
}

void ActionButton::updateButton()
{
  for (TQButton *&ent : d->buttons)
  {
    if (isVisible())
    {
      ent->show();
    }
    else
    {
      ent->hide();
    }
    ent->setEnabled(isEnabled());
    //if (!toolTip().isNull())
    //{
    //  ent->setToolTip(toolTip());
    //}
    //if (!whatsThis().isNull())
    //{
    //  ent->setWhatsThis(whatsThis());
    //}
    ent->setPixmap(iconSet().pixmap());
    // setPixmap() clears the button text, so setText() must be called afterwards
    ent->setText(text());
    // if the item cannot do the action anymore
    // lets revert to the initial state
    if (ent->isToggleButton())
    {
      ent->setDown(isOn());
    }
  }
}

bool ActionButton::activate()
{
  bool tg = false;
  for (TQButton *&ent : d->buttons)
  {
    if (ent->isToggleButton())
    {
      // we set the the current Action state
      ent->setDown(isOn());
      // toggle the action cause we are not directly connected there..
      tg = true;
    }
  }

  if (tg)
  {
    toggle();
  }

  return Action::activate();
}

void ActionButton::setButton(TQButton *button)
{
  // First, let's clear the list
  while (!d->buttons.isEmpty())
  {
    d->removeButton(d->buttons.first());
  }

  // And then add it
  d->addButton(button);
}

TQButton* ActionButton::button() const
{
  return d->buttons.first();
}

}

}

#include "polkit-tqt-gui-actionbutton.moc"
