/*
 * info_netbsd.cpp is part of the KDE program kcminfo.  This displays
 * various information about the OpenBSD system it's running on.
 *
 * Originally written by Jaromir Dolecek <dolecek@ics.muni.cz>. CPU info
 * code has been imported from implementation of processor.cpp for KDE 1.0
 * by David Brownlee <abs@NetBSD.org> as found in NetBSD packages collection.
 * Hubert Feyer <hubertf@NetBSD.org> enhanced the sound information printing
 * quite a lot, too.
 *
 * The code is placed into public domain. Do whatever you want with it.
 */

#define INFO_CPU_AVAILABLE
#define INFO_IRQ_AVAILABLE
#define INFO_DMA_AVAILABLE
#define INFO_PCI_AVAILABLE
#define INFO_IOPORTS_AVAILABLE
#define INFO_SOUND_AVAILABLE
#define INFO_DEVICES_AVAILABLE
#define INFO_SCSI_AVAILABLE
#define INFO_PARTITIONS_AVAILABLE
#define INFO_XSERVER_AVAILABLE


/*
 * all following functions should return TRUE, when the Information
 * was filled into the lBox-Widget. Returning FALSE indicates that
 * information was not available.
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/sysctl.h>
#include <stdio.h>	/* for NULL */
#include <stdlib.h>	/* for malloc(3) */

#include <tqfile.h>
#include <tqfontmetrics.h>
#include <tqstrlist.h>
#include <tqtextstream.h>

#include <kdebug.h>

typedef struct
  {
  int	string;
  int	name;
  const char	*title;
  } hw_info_mib_list_t;

bool GetInfo_CPU(TQListView *lBox)
{
  static hw_info_mib_list_t hw_info_mib_list[]= {
	{ 1, HW_MODEL,		"Model" },
	{ 1, HW_MACHINE,	"Machine" },
	{ 0, HW_NCPU,		"Number of CPUs" },
	{ 0, HW_PAGESIZE,	"Pagesize" },
	{ 0,0,0 }
	};
  hw_info_mib_list_t *hw_info_mib;

  int mib[2], num;
  char *buf;
  size_t len;
  TQString value;

  lBox->addColumn(i18n("Information"));
  lBox->addColumn(i18n("Value"));

  for ( hw_info_mib = hw_info_mib_list ;  hw_info_mib->title ; ++hw_info_mib )
  {
	mib[0] = CTL_HW;
	mib[1] = hw_info_mib->name;
	if ( hw_info_mib->string ) {
		sysctl(mib,2,NULL,&len,NULL,0);
		if ( (buf = (char*)malloc(len)) ) {
			sysctl(mib,2,buf,&len,NULL,0);
			value = TQString::fromLocal8Bit(buf);
			free(buf);
		}
		else {
			value = TQString("Unknown");
		}
	}
	else {
		len = sizeof(num);
		sysctl(mib,2,&num,&len,NULL,0);
		value.sprintf("%d", num);
	}
	new TQListViewItem(lBox, hw_info_mib->title, value);
   }

   return true;
}

// this is used to find out which devices are currently
// on system
static bool GetDmesgInfo(TQListView *lBox, const char *filter,
	void func(TQListView *, TQString s, void **, bool))
{
        TQFile *dmesg = new TQFile("/var/run/dmesg.boot");
	bool usepipe=false;
	FILE *pipe=NULL;
	TQTextStream *t;
	bool seencpu=false;
	void *opaque=NULL;
	TQString s;
	bool found=false;

	if (dmesg->exists() && dmesg->open(IO_ReadOnly)) {
		t = new TQTextStream(dmesg);
	}
	else {
		delete dmesg;
		pipe = popen("/sbin/dmesg", "r");
		if (!pipe) return false;
		usepipe = true;
		t = new TQTextStream(pipe, IO_ReadOnly);
	}

	TQListViewItem *olditem = NULL;
	while(!(s = t->readLine()).isNull()) {
		if (!seencpu) {
			if (s.contains("cpu"))
				seencpu = true;
			else
				continue;
		}
		if (s.contains("boot device") ||
			s.contains("WARNING: old BSD partition ID!"))
			break;

		if (!filter || s.contains(filter)) {
			if (func) {
				func(lBox, s, &opaque, false);
			}
			else {
				olditem = new TQListViewItem(lBox, olditem, s);
			}
			found = true;
		}
	}
	if (func) {
		func(lBox, s, &opaque, true);
	}
	//lBox->triggerUpdate();

	delete t;
	if (pipe) {
		pclose(pipe);
	}
	else {
		dmesg->close();
		delete dmesg;
	}

	return found;
}


void AddIRQLine(TQListView *lBox, TQString s, void **opaque, bool ending)
{
	TQStrList *strlist = (TQStrList *) *opaque;
	const char *str;
	int pos, irqnum=0;
	const char *p;
	p = s.latin1();

	if (!strlist) {
		strlist = new TQStrList();
		*opaque = (void *) strlist;
	}
	if (ending) {
		str = strlist->first();
		for(;str; str = strlist->next()) {
			new TQListViewItem(lBox, str);
		}
		delete strlist;
		return;
	}

	pos = s.find(" irq ");
	irqnum = (pos < 0) ? 0 : atoi(&p[pos+5]);
	if (irqnum) {
		s.sprintf("%02d%s", irqnum, p);
	}
	else {
		s.sprintf("??%s", p);
	}
	strlist->inSort(s.latin1());
}

bool GetInfo_IRQ (TQListView *lBox)
{
	lBox->addColumn(i18n("IRQ"));
	lBox->addColumn(i18n("Device"));
	(void) GetDmesgInfo(lBox, " irq ", AddIRQLine);
	return true;
}

bool GetInfo_DMA (TQListView *)
{
  return false;
}

bool GetInfo_PCI (TQListView *lbox)
{
	if (!GetDmesgInfo(lbox, "at pci", NULL))
		new TQListViewItem(lbox, i18n("No PCI devices found."));
	return true;
}

bool GetInfo_IO_Ports (TQListView *lbox)
{
	if (!GetDmesgInfo(lbox, "port 0x", NULL))
		new TQListViewItem(lbox, i18n("No I/O port devices found."));
	return true;
}

bool GetInfo_Sound (TQListView *lbox)
{
	if (!GetDmesgInfo(lbox, "audio", NULL))
		new TQListViewItem(lbox, i18n("No audio devices found."));

	// append information on any audio devices found
	TQListViewItem *lvitem = lbox->firstChild();
	for(; lvitem; lvitem = lvitem->nextSibling()) {
		TQString s;
		int pos, len;
		const char *start, *end;
		char *dev;

		s = lvitem->text(0);
		if ((pos = s.find("at ")) >= 0) {
			pos += 3;	// skip "at "
			start = end = s.ascii();
			for(; *end && (*end!=':') && (*end!='\n'); end++);
			len = end - start;
			dev = (char *) malloc(len + 1);
			strncpy(dev, start, len);
			dev[len] = '\0';

			GetDmesgInfo(lbox, dev, NULL);

			free(dev);
		}
	}

	return true;
}

bool GetInfo_Devices (TQListView *lBox)
{
	(void) GetDmesgInfo(lBox, NULL, NULL);
	return true;
}

bool GetInfo_SCSI (TQListView *lbox)
{
	if (!GetDmesgInfo(lbox, "scsibus", NULL))
		new TQListViewItem(lbox, i18n("No SCSI devices found."));
	return true;
}

bool GetInfo_Partitions (TQListView *lbox)
{
	TQString s;
	char *line, *orig_line;
	const char *device, *mountpoint, *type, *flags;
	FILE *pipe = popen("/sbin/mount", "r");
	TQTextStream *t;

	if (!pipe) {
		kdError(0) << i18n("Unable to run /sbin/mount.") << endl;
		return false;
	}
	t = new TQTextStream(pipe, IO_ReadOnly);

	lbox->addColumn(i18n("Device"));
	lbox->addColumn(i18n("Mount Point"));
	lbox->addColumn(i18n("FS Type"));
	lbox->addColumn(i18n("Mount Options"));

	TQListViewItem *olditem = 0;
	while (!(s = t->readLine()).isNull()) {
		orig_line = line = strdup(s.latin1());

		device = strsep(&line, " ");

		(void) strsep(&line, " "); // consume word "on"
		mountpoint = strsep(&line, " ");

		(void) strsep(&line, " "); // consume word "type"
		type = strsep(&line, " ");

		flags = line;

		olditem = new TQListViewItem(lbox, olditem, device, mountpoint,
					type, flags);

		free(orig_line);
	}

	delete t;
	pclose(pipe);
	return true;
}

bool GetInfo_XServer_and_Video (TQListView *lBox)
{
	return GetInfo_XServer_Generic( lBox );
}
