/***************************************************************************
 *   Copyright (C) 2005 by Danny Kukawka                                   *
 *                         <dkukawka@suse.de>, <danny.kukawka@web.de>      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of version 2 of the GNU General Public License     *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/
 
/*! \file configuredialog.cpp
 * All here displayed file members of configureDialog.cpp are related to operations with the 
 * configure dialog for kpowersave
 * \brief 	In this file can be found all configure dialog related code. 
 * \author 	Danny Kukawka, <dkukawka@suse.de>, <danny.kukawka@web.de>
 * \date    	2005
 */

 // KDE - Headers
 #include <kaudioplayer.h>
 #include <tdeconfig.h>
 #include <kiconloader.h>
 #include <tdelocale.h>
 #include <tdemessagebox.h>
 #include <kinputdialog.h>
 #include <tdeaccelmanager.h>

 // QT - Headers
 #include <tqcheckbox.h>
 #include <tqcombobox.h>
 #include <tqdialog.h>
 #include <tqgroupbox.h>
 #include <tqlabel.h>
 #include <tqlineedit.h>
 #include <tqlistbox.h>
 #include <tqpushbutton.h>
 #include <tqslider.h>
 #include <tqspinbox.h>
 #include <tqstringlist.h>
 #include <tqtabwidget.h>
 #include <tqtoolbox.h>
 #include <tqtooltip.h>
 #include <tqstring.h>

 #include "configuredialog.h"

/*! This is the default constructor of the class ConfigureDialog. */
ConfigureDialog::ConfigureDialog( TDEConfig *_config, HardwareInfo *_hwinfo, Settings *_settings,
				  TQWidget *parent, const char *name)
				 :configure_Dialog(parent, name, false, WDestructiveClose )
{
	kdDebugFuncIn(trace);

	tdeconfig = _config;
	settings = _settings;
	hwinfo = _hwinfo;
	suspend = hwinfo->getSuspendSupport();

	TQString session = getenv("DESKTOP_SESSION");
	if(session.startsWith("gnome")) gnome_session = true;
        else gnome_session = false;

	initalised = false;
	general_changed = false;
	scheme_changed = false;
	displayed_WARN_autosuspend = false;

	// check if brightness is supporte
	if(hwinfo->supportBrightness()) {
		brightnessLevels = hwinfo->getMaxBrightnessLevel() -1;
		brightness_last = hwinfo->getCurrentBrightnessLevel();
	} else { 
		brightnessLevels = -1;
		brightness_last = -1;
	}
	brightness_changed = false;

	currentScheme = -1;

	// get the correct available suspend types
	SuspendStates suspend = hwinfo->getSuspendSupport();
	if( suspend.suspend2ram && (suspend.suspend2ram_allowed || suspend.suspend2ram_allowed == -1))
		actions.append("Suspend");
	if ( suspend.suspend2disk && (suspend.suspend2disk_allowed || suspend.suspend2disk_allowed == -1))
		actions.append("Hibernate");
	if ( suspend.standby && (suspend.standby_allowed || suspend.standby_allowed == -1))
		actions.append("Standby");

	setIcons();
	setTooltips();
	getSchemeList();
	setSchemeList();	
	setGeneralSettings();
	setInactivityBox();
	selectScheme(settings->currentScheme);
	
	tL_valueBrightness->setFrameStyle(TQFrame::Panel | TQFrame::Sunken);
	tL_valueBrightness->setBackgroundMode(TQt::PaletteBase);
	tL_valueBrightness->setAlignment(TQt::AlignCenter);
	tL_valueBrightness->setFocusPolicy(TQ_NoFocus);

	// hide Lid item if this is not a laptop
	if (!hwinfo->isLaptop())
		cB_lockLid->hide();
	// hide CPU Freq page if not supported
	if (!hwinfo->supportCPUFreq()) {
		tB_scheme->setItemLabel( 4, tB_scheme->itemLabel(4) + i18n(" - not supported"));
		tB_scheme->setItemEnabled( 4, false);
	}

	if (!hwinfo->supportBrightness()) {
		tB_scheme->setItemLabel( 3, tB_scheme->itemLabel(3) + i18n(" - not supported"));
		tB_scheme->setItemEnabled( 3, false);
	}

	initalised = true;
	kdDebugFuncOut(trace);
}

/*! This is the default destructor of the class ConfigureDialog. */
ConfigureDialog::~ConfigureDialog(){
	kdDebugFuncIn(trace);
	 // no need to delete child widgets, TQt does it all for us
}

/* ---- START General setup SECTION ---- */

/*! 
 * This used to fill the TQListBox listBox_schemes and the scheme related TQComboBoxes
 * with the existing schemes.
 */
void ConfigureDialog::setSchemeList(){
	kdDebugFuncIn(trace);

	listBox_schemes->clear();
	cB_acScheme->clear();
	cB_batteryScheme->clear();

	for ( TQStringList::Iterator it = schemes.begin(); it != schemes.end(); ++it ) {
		const char *_tmp = (*it).utf8();
		if(_tmp == "Performance" || _tmp == i18n("Performance")) {
			listBox_schemes->insertItem(SmallIcon("scheme_power", TQIconSet::Automatic), i18n(_tmp));
			cB_acScheme->insertItem(i18n(_tmp));
			cB_batteryScheme->insertItem(i18n(_tmp));
		} else if(_tmp == "Powersave" || _tmp == i18n("Powersave")) {
			listBox_schemes->insertItem(SmallIcon("scheme_powersave", TQIconSet::Automatic),
						    i18n(_tmp));
			cB_acScheme->insertItem(i18n(_tmp));
			cB_batteryScheme->insertItem(i18n(_tmp));
		} else if(_tmp == "Presentation" || _tmp == i18n("Presentation")){
			listBox_schemes->insertItem(SmallIcon("scheme_presentation", TQIconSet::Automatic),
						    i18n(_tmp));
			cB_acScheme->insertItem(i18n(_tmp));
			cB_batteryScheme->insertItem(i18n(_tmp));
		} else if(_tmp == "Acoustic" || _tmp == i18n("Acoustic")) {
			listBox_schemes->insertItem(SmallIcon("scheme_acoustic", TQIconSet::Automatic),
						    i18n(_tmp));
			cB_acScheme->insertItem(i18n(_tmp));
			cB_batteryScheme->insertItem(i18n(_tmp));
		} else if(_tmp == "AdvancedPowersave" || _tmp == i18n("Advanced Powersave")) {
			listBox_schemes->insertItem(SmallIcon("scheme_advanced_powersave", 
						    TQIconSet::Automatic), i18n("Advanced Powersave"));
			cB_acScheme->insertItem(i18n("Advanced Powersave"));
			cB_batteryScheme->insertItem(i18n("Advanced Powersave"));
		} else {
			listBox_schemes->insertItem(i18n(_tmp));
			cB_acScheme->insertItem(i18n(_tmp));
			cB_batteryScheme->insertItem(i18n(_tmp));
		}
    	}

	TDEAcceleratorManager::manage(pB_newScheme);
	kdDebugFuncOut(trace);
}

/*! 
 * This used to get the list of schemes in from the config
 */
void ConfigureDialog::getSchemeList(){
	kdDebugFuncIn(trace);

	if (tdeconfig->hasGroup("General")) {
		tdeconfig->setGroup("General");
		schemes = tdeconfig->readListEntry("schemes", ',');
	}

	kdDebugFuncOut(trace);
}

/*! 
 * This used to set the current scheme based on the name of the scheme
 * \param _scheme 	TQString with the name of the scheme
 */
void ConfigureDialog::selectScheme (TQString _scheme){
	kdDebugFuncIn(trace);

	// select the current scheme in the listbox
	if(!_scheme.isEmpty()) {
		int pos = schemes.findIndex(_scheme);
		if(pos > -1) {
			listBox_schemes->setCurrentItem(pos);
			currentScheme = pos;
		} else {
			listBox_schemes->setCurrentItem(0);
		}
	} else {
		listBox_schemes->setCurrentItem(0);
	}
	
	kdDebugFuncOut(trace);
}

/*! 
 * This used to set all needed Icons for the dialog.
 */
void ConfigureDialog::setIcons(){
	kdDebugFuncIn(trace);

	/* set all Icons */
	this->setIcon(SmallIcon("kpowersave", TQIconSet::Automatic));
	buttonApply->setIconSet(SmallIconSet("apply", TQIconSet::Automatic));
	buttonCancel->setIconSet(SmallIconSet("cancel", TQIconSet::Automatic));
	buttonOk->setIconSet(SmallIconSet("ok", TQIconSet::Automatic));
	buttonHelp->setIconSet(SmallIconSet("help", TQIconSet::Automatic));

	pB_editBlacklist->setIconSet(SmallIconSet("configure", TQIconSet::Automatic));
	pB_editBlacklistDimm->setIconSet(SmallIconSet("configure", TQIconSet::Automatic));
	pB_editAutosuspendGBlacklist->setIconSet(SmallIconSet("configure", TQIconSet::Automatic));
	pB_editAutodimmGBlacklist->setIconSet(SmallIconSet("configure", TQIconSet::Automatic));

	tB_scheme->setItemIconSet( 0 ,SmallIcon("tdescreensaver", TQIconSet::Automatic));
	tB_scheme->setItemIconSet( 1 ,SmallIcon("display", TQIconSet::Automatic));
	
	if(actions[0] == "Hibernate") 
		tB_scheme->setItemIconSet( 2 ,SmallIcon("suspend_to_disk", TQIconSet::Automatic));
	else if(actions[0] == "Suspend") 
		tB_scheme->setItemIconSet( 2 ,SmallIcon("suspend_to_ram", TQIconSet::Automatic));
	else if(actions[0] == "Standby") 
		tB_scheme->setItemIconSet( 2 ,SmallIcon("stand_by", TQIconSet::Automatic));
	
	tB_scheme->setItemIconSet( 3 ,SmallIcon("autodimm", TQIconSet::Automatic));
	tB_scheme->setItemIconSet( 4 ,SmallIcon("processor", TQIconSet::Automatic));
	tB_scheme->setItemIconSet( 5 ,SmallIcon("misc", TQIconSet::Automatic));

	tB_general->setItemIconSet( 0, SmallIcon( "scheme_powersave", TQIconSet::Automatic));
	tB_general->setItemIconSet( 1, SmallIcon( "button", TQIconSet::Automatic));
	tB_general->setItemIconSet( 2, SmallIcon( "scheme_power", TQIconSet::Automatic));
	tB_general->setItemIconSet( 3, SmallIcon( "system-lock-screen", TQIconSet::Automatic));
	tB_general->setItemIconSet( 4, SmallIcon( "misc", TQIconSet::Automatic));
	pB_configNotify->setIconSet(SmallIconSet("knotify", TQIconSet::Automatic));

	kdDebugFuncOut(trace);
}

/*! 
 * This used to set all needed Tooltips for the dialog.
 */
void ConfigureDialog::setTooltips(){
	kdDebugFuncIn(trace);

	// TQToolTip::setWakeUpDelay ( 1000 );
	TQToolTip::add(cB_specificSettings, i18n("This enables specific screen saver settings. \n"
						"Note: If selected, the global screen saver settings are \n"
						"overwritten while kpowersave runs."));
	TQToolTip::add(cB_disable_Ss, i18n("This disables the screen saver. \n"
					  "Note: If selected, the global screen saver settings are \n"
					  "overwritten while kpowersave runs."));
	TQToolTip::add(cB_blankScreen, i18n("This blanks the screen instead of using a specific screen saver. \n"
					   "Note: This may work only with KScreensaver."));
	TQToolTip::add(cB_SpecificPM, i18n("This enables specific DPMS settings. \n"
					  "Note: If selected, the global DPMS settings are \n"
					  "overwritten while kpowersave runs."));
	TQToolTip::add(cB_disablePM, i18n("This disables DPMS support."));
	TQToolTip::add(cB_lockSuspend, i18n("If selected, the screen is locked on suspend or standby."));
	TQToolTip::add(cB_lockLid, i18n("If selected, the screen is locked if the lid close event is triggered."));
	
	TQToolTip::add(cB_autoSuspend, i18n("Check this box to enable or disable automatic suspension of "
					   "the computer."));
	TQToolTip::add(cB_autoInactivity, i18n("Activate this action if the user was inactive for the defined "
					      "time \n(greater than 0 minutes). If empty, nothing happens."));
	
	tB_scheme->setItemToolTip ( 0, i18n("All scheme-related screen saver and DPMS settings."));
	tB_scheme->setItemToolTip ( 1, i18n("All scheme-related display brightness settings."));
	tB_scheme->setItemToolTip ( 2, i18n("All scheme-related automatic suspend settings."));

	TQToolTip::add(brightnessSlider, i18n("Here you can change the brightness of your display. \n"
					     "Use the slider to change the brightness directly for \n"
					     "testing. To reset back to previous level, please press \n"
					     "the 'Reset' button. "));
	TQToolTip::add(pB_resetBrightness, i18n("Use this button to set back the slider and the "
					       "display brightness."));

	kdDebugFuncOut(trace);
}

/*!
 * This function is used to get the real name of the predefined schemes.
 * This is a workaround for the case if Yast translated the name of the 
 * scheme In this case is a new config section created and the old settings
 * would be lost.
 * \param s_scheme TQString with the 'current' name of the scheme
 * \return TQString with the english name of the scheme 
 */
TQString ConfigureDialog::getSchemeRealName(TQString s_scheme) {
	kdDebugFuncIn(trace);
	TQString ret = s_scheme;

	if( s_scheme == "Performance" || s_scheme == i18n("Performance"))
		ret = "Performance";
	else if( s_scheme == "Powersave" || s_scheme == i18n("Powersave"))
		ret = "Powersave";
	else if( s_scheme == "Presentation" || s_scheme == i18n("Presentation"))
		ret = "Presentation";
	else if( s_scheme == "Acoustic" || s_scheme == i18n("Acoustic"))
		ret = "Acoustic";

	kdDebugFuncOut(trace);
	return ret;
}

/*!
 * This function is used to read the settings from configfile and
 * set the values to the related dialog elements.
 * \param schemeID Integer value, represent the position of the 
 *		   scheme in the TQListBox and within the pdaemon-list.
 */
void ConfigureDialog::setConfigToDialog( int schemeID ){
	if (trace) kdDebug() << funcinfo << "IN , scheme: " << schemeID << endl;

	initalised = false;
	
	TQString s_scheme = getSchemeRealName(schemes[schemeID]);
	if (tdeconfig->hasGroup(s_scheme)){
		settings->load_general_settings();
		
		// set the delete scheme button enabled if needed
		if (!s_scheme.startsWith("Performance") && !s_scheme.startsWith("Powersave") && 
		    !s_scheme.startsWith("Presentation") && !s_scheme.startsWith("Acoustic")) {
			if (!s_scheme.startsWith(settings->ac_scheme) || 
			    !s_scheme.startsWith(settings->battery_scheme)) {
				// enable delete button
				pB_deleteScheme->setEnabled( true );
				TQToolTip::add(pB_deleteScheme, i18n("Press this button to delete the "
								    "selected scheme."));
			} else {
				// disable button
				pB_deleteScheme->setEnabled( false );
				TQToolTip::add(pB_deleteScheme, i18n("You can't delete the current AC "
								    "or battery scheme."));
			}
		} else {
			// disable button, can't delete these scheme, they are default
			pB_deleteScheme->setEnabled( false );
			TQToolTip::add(pB_deleteScheme, i18n("You can't delete this default scheme."));
		}

		tdeconfig->setGroup(s_scheme);
	}
	// no configuration found, set to default values ?!
	else { 
		if(tdeconfig->hasGroup("default-scheme"))
			tdeconfig->setGroup("default-scheme");
		else { 
			kdDebugFuncOut(trace);
			return;
		}
	}
	
	cB_disable_Ss_toggled(tdeconfig->readBoolEntry("disableSs",false));
	cB_blankScreen->setChecked(tdeconfig->readBoolEntry("blankSs",false));
	cB_specificSettings_toggled(tdeconfig->readBoolEntry("specSsSettings",false));
	
	int i_standby = tdeconfig->readNumEntry("standbyAfter", -1);
	if(i_standby >= 0) {
		sB_standby->setValue(i_standby);
	}
	else {
		// load value from default-scheme
		if(tdeconfig->hasGroup("default-scheme")){
			tdeconfig->setGroup("default-scheme");
			i_standby = tdeconfig->readNumEntry("standbyAfter", -1);
			if(i_standby >= 0) {
				sB_standby->setValue(i_standby);
			}
			else sB_standby->setValue(0);
		}
		else{
			sB_standby->setValue(0);
		}
		// reset group to selected scheme
		if (tdeconfig->hasGroup(s_scheme)) tdeconfig->setGroup(s_scheme);
		else tdeconfig->setGroup("default-scheme");
	}
	
	int i_suspend = tdeconfig->readNumEntry("suspendAfter", -1);
	if(i_suspend >= 0) {
		sB_suspend->setValue(i_suspend);
	}
	else {
		// load value from default-scheme
		if(tdeconfig->hasGroup("default-scheme")){
			tdeconfig->setGroup("default-scheme");
			i_standby = tdeconfig->readNumEntry("suspendAfter", -1);
			if(i_standby >= 0) {
				sB_standby->setValue(i_standby);
			}
			else sB_standby->setValue(0);
		}
		else{
			sB_standby->setValue(0);
		}
		// reset group to selected scheme
		if (tdeconfig->hasGroup(s_scheme)) tdeconfig->setGroup(s_scheme);
		else tdeconfig->setGroup("default-scheme");
	}
	
	int i_poweroff = tdeconfig->readNumEntry("powerOffAfter", -1);
	if(i_poweroff >= 0) {
		sB_powerOff->setValue(i_poweroff);
	}
	else {
		// load value from default-scheme
		if(tdeconfig->hasGroup("default-scheme")){
			tdeconfig->setGroup("default-scheme");
			i_standby = tdeconfig->readNumEntry("powerOffAfter", -1);
			if(i_poweroff >= 0) {
				sB_powerOff->setValue(i_poweroff);
			}
			else sB_powerOff->setValue(0);
		}
		else{
			sB_powerOff->setValue(0);
		}
		// reset group to selected scheme
		if (tdeconfig->hasGroup(s_scheme)) tdeconfig->setGroup(s_scheme);
		else tdeconfig->setGroup("default-scheme");
	}
	
	cB_SpecificPM_toggled(tdeconfig->readBoolEntry("specPMSettings",false));
	cB_disablePM_toggled(tdeconfig->readBoolEntry("disableDPMS",false));
	
	// set autosuspend related settings
	TQString _action = tdeconfig->readEntry("autoInactiveAction", "NULL");
	if( _action != "NULL") {
		int _index = actions.findIndex(_action);
		if( _index != -1) { 
			cB_autoInactivity->setCurrentItem( _index );
			cB_autoInactivity_activated( _index );
			cB_Blacklist->setEnabled(true);
		}
		else {
			// set to empty element if not supported by current machine or
			// if the value is "_NONE_"
			cB_autoInactivity->setCurrentItem( 0 );
			cB_autoInactivity_activated( 0 );
			cB_Blacklist->setEnabled(false);
		}
	}
	else {
		// set to disabled (to reduce code), if a entry found set to enabled !
		cB_Blacklist->setEnabled(false);
		
		if(tdeconfig->hasGroup("default-scheme")){
			tdeconfig->setGroup("default-scheme");
			 _action = tdeconfig->readEntry("autoInactiveAction", "NULL");
			if(_action != "NULL") {
				int _index = actions.findIndex(_action);
				if( _index != -1) { 
					cB_autoInactivity->setCurrentItem( _index );
					tL_autoInactivity_After->setEnabled(true);
					cB_Blacklist->setEnabled(true);
				}
				else {
					cB_autoInactivity->setCurrentItem( 0 );
				}
			}
			else {
				cB_autoInactivity->setCurrentItem( 0 );
			}
			// reset group to selected scheme
			if (tdeconfig->hasGroup(s_scheme)) tdeconfig->setGroup(s_scheme);
			else tdeconfig->setGroup("default-scheme");
		}
		else{
			cB_autoInactivity->setCurrentItem( 0 );
		}
	}
	if( cB_autoInactivity->currentItem() == 0 ) {
		sB_autoInactivity->setEnabled(false);
		tL_autoInactivity_After->setEnabled(false);
	}
	
	int i_autoInactivityAfter = tdeconfig->readNumEntry("autoInactiveActionAfter", -1);
	if(i_autoInactivityAfter >= 0) {
		sB_autoInactivity->setValue(i_autoInactivityAfter);
	}
	else {
		// load value from default-scheme
		if(tdeconfig->hasGroup("default-scheme")){
			tdeconfig->setGroup("default-scheme");
			i_autoInactivityAfter = tdeconfig->readNumEntry("autoInactiveActionAfter", -1);
			if(i_autoInactivityAfter >= 0) {
				sB_autoInactivity->setValue(i_autoInactivityAfter);
			}
			else sB_autoInactivity->setValue(0);
		}
		else{
			sB_autoInactivity->setValue(0);
		}
		// reset group to selected scheme
		if (tdeconfig->hasGroup(s_scheme)) tdeconfig->setGroup(s_scheme);
		else tdeconfig->setGroup("default-scheme");
	}

	if (hwinfo->supportBrightness()) {
		// enable the widgets
		cB_Brightness->setEnabled(true);
		cB_Brightness_toggled(tdeconfig->readBoolEntry("enableBrightness",false));

		if (brightnessLevels > 33) 
			brightnessSlider->setLineStep(3);
		else 
			brightnessSlider->setLineStep(100/brightnessLevels);

		brightnessSlider->setPageStep(10);

		int i_brightnessPercent = tdeconfig->readNumEntry("brightnessPercent", -1);
		if(i_brightnessPercent >= 0) {
			brightnessSlider->setValue(i_brightnessPercent);
			tL_valueBrightness->setText(TQString::number(i_brightnessPercent) + " %");
		}
		else {
			brightnessSlider->setValue(100);
			tL_valueBrightness->setText(TQString::number(100)+ " %");	
		}
		
		tL_brightness->setText(i18n("Your hardware supports to change the brightness. The "
					    "values of the slider are in percent and mapped "
					    "to the available brightness levels of your hardware."));
	}
	else {
		cB_Brightness->setEnabled(false);
		gB_Brightness->setEnabled(false);
		tL_brightness->setText(i18n("Your Hardware currently not support changing the brightness "
					    "of your display."));
	}

	// no need to enable autodimm if not support change brightness 
	if (hwinfo->supportBrightness()) {
		int i_autoDimmAfter = tdeconfig->readNumEntry("autoDimmAfter", -1);
		if(i_autoDimmAfter >= 0) {
			sB_autoDimmTime->setValue(i_autoDimmAfter);
		}
		else {
			// load value from default-scheme
			if(tdeconfig->hasGroup("default-scheme")){
				tdeconfig->setGroup("default-scheme");
				i_autoDimmAfter = tdeconfig->readNumEntry("autoDimmAfter", -1);
				if(i_autoDimmAfter >= 0) {
					sB_autoDimmTime->setValue(i_autoDimmAfter);
				}
				else sB_autoDimmTime->setValue(0);
			}
			else{
				sB_autoDimmTime->setValue(0);
			}
			// reset group to selected scheme
			if (tdeconfig->hasGroup(s_scheme)) tdeconfig->setGroup(s_scheme);
			else tdeconfig->setGroup("default-scheme");
		}

		int i_autoDimmTo = tdeconfig->readNumEntry("autoDimmTo", -1);
		if(i_autoDimmTo >= 0) {
			sB_autoDimmTo->setValue(i_autoDimmTo);
		}
		else {
			// load value from default-scheme
			if(tdeconfig->hasGroup("default-scheme")){
				tdeconfig->setGroup("default-scheme");
				i_autoDimmTo = tdeconfig->readNumEntry("autoDimmTo", -1);
				if(i_autoDimmAfter >= 0) {
					sB_autoDimmTo->setValue(i_autoDimmTo);
				}
				else sB_autoDimmTo->setValue(0);
			}
			else{
				sB_autoDimmTo->setValue(0);
			}
			// reset group to selected scheme
			if (tdeconfig->hasGroup(s_scheme)) tdeconfig->setGroup(s_scheme);
			else tdeconfig->setGroup("default-scheme");
		}
	}

	if (hwinfo->supportCPUFreq()) {
		TQString cpuFreqPolicy = tdeconfig->readEntry("cpuFreqPolicy");
		
		if (cpuFreqPolicy.isEmpty()) {
			// load value from default-scheme
			if(tdeconfig->hasGroup("default-scheme")){
				tdeconfig->setGroup("default-scheme");
				cpuFreqPolicy = tdeconfig->readEntry("cpuFreqPolicy");	
			} 
			if (cpuFreqPolicy.isEmpty()) {
				cpuFreqPolicy = "DYNAMIC";
			}
			// reset group to selected scheme
			if (tdeconfig->hasGroup(s_scheme)) tdeconfig->setGroup(s_scheme);
			else tdeconfig->setGroup("default-scheme");
		}

		if (cpuFreqPolicy.startsWith("PERFORMANCE")) {
			comboB_cpuFreq->setCurrentItem(0);
		} else if (cpuFreqPolicy.startsWith("DYNAMIC")) {
			comboB_cpuFreq->setCurrentItem(1);
		} else if (cpuFreqPolicy.startsWith("POWERSAVE")) {
			comboB_cpuFreq->setCurrentItem(2);
		} else {
			kdWarning() << "Could not read/map CPU Freq Policy, set to dynamic" << endl;
			comboB_cpuFreq->setCurrentItem(1);
		}
	}

	cB_disableNotifications->setChecked(tdeconfig->readBoolEntry("disableNotifications",false));
	
	cB_Blacklist_toggled(tdeconfig->readBoolEntry("autoInactiveSchemeBlacklistEnabled",false));
	cB_autoSuspend_toggled(tdeconfig->readBoolEntry("autoSuspend",false));
	cB_BlacklistDimm_toggled(tdeconfig->readBoolEntry("autoDimmSchemeBlacklistEnabled",false));
	cB_autoDimm_toggled(tdeconfig->readBoolEntry("autoDimm",false));
		
	initalised = true;
	scheme_changed = false;
	if(!general_changed) buttonApply->setEnabled(false);
	currentScheme = schemeID;

	kdDebugFuncOut(trace);
}

/*!
 * This is used to set the values from the section general in configfile to the 
 * related items in the dialog.
 */
void ConfigureDialog::setGeneralSettings() {
	kdDebugFuncIn(trace);

	tdeconfig->setGroup("General");
	
	cB_lockSuspend->setChecked(tdeconfig->readBoolEntry("lockOnSuspend",false));
	cB_lockLid->setChecked(tdeconfig->readBoolEntry("lockOnLidClose",false));
	cB_autostart->setChecked(tdeconfig->readBoolEntry("Autostart",false));
	cB_autostart_neverAsk->setChecked(tdeconfig->readBoolEntry("AutostartNeverAsk",false));
	
	TQString lockmethod = tdeconfig->readEntry("lockMethod", "NULL");
	if(comboB_lock->count() == 0 ){
		comboB_lock->insertItem(i18n("Select Automatically"),0); 
		comboB_lock->insertItem(i18n("KScreensaver"),1);
		comboB_lock->insertItem(i18n("XScreensaver"),2);
		comboB_lock->insertItem(i18n("xlock"),3);
		if (gnome_session) comboB_lock->insertItem(i18n("GNOME Screensaver"),4);
	}
	
	if (lockmethod == "automatic") comboB_lock->setCurrentItem(0);
	else if (lockmethod == "tdescreensaver")  comboB_lock->setCurrentItem(1);
	else if (lockmethod == "xscreensaver") comboB_lock->setCurrentItem(2);
	else if (lockmethod == "xlock") comboB_lock->setCurrentItem(3);
	else if (gnome_session && (lockmethod == "gnomescreensaver")) comboB_lock->setCurrentItem(4);
	else comboB_lock->setCurrentItem(0);
	
	if(cB_lockSuspend->isOn() || cB_lockLid->isOn() ) {
		tL_lockWith->setEnabled(true);
		comboB_lock->setEnabled(true);
	}
	else {
		tL_lockWith->setEnabled(false);
		comboB_lock->setEnabled(false);
	}

	// battery tab
	BatteryCollection* _primBats = hwinfo->getPrimaryBatteries();
	if (_primBats->getNumBatteries() > 0) {
		// set widgets in the tab
		sB_batWarning->setValue(tdeconfig->readNumEntry("batteryWarning",0));
		sB_batLow->setValue(tdeconfig->readNumEntry("batteryLow",0));
		sB_batCritical->setValue(tdeconfig->readNumEntry("batteryCritical",0));
		// hide them and enable only if needed
		sB_batWarnAction_value->hide();
		sB_batLowAction_value->hide();
		sB_batCritAction_value->hide();

		TQString _select;
		TQStringList _actions = tdeconfig->readListEntry("batteryAllowedActions", TQString());
		_select = tdeconfig->readEntry("batteryWarningAction", TQString());
		fillActionComboBox(cB_batWarning, _actions, _select);
		if (_select == "BRIGHTNESS" && hwinfo->supportBrightness()) {
			sB_batWarnAction_value->show();
			sB_batWarnAction_value->setValue(tdeconfig->readNumEntry( "batteryWarningActionValue",0));
		}
		_select = tdeconfig->readEntry("batteryLowAction", TQString());
		fillActionComboBox(cB_batLow, _actions, _select);
		if (_select == "BRIGHTNESS" && hwinfo->supportBrightness()) {
			sB_batLowAction_value->show();
			sB_batLowAction_value->setValue( tdeconfig->readNumEntry( "batteryLowActionValue",0));
		}
		_select = tdeconfig->readEntry("batteryCriticalAction", TQString());
		fillActionComboBox(cB_batCritical, _actions, _select);
		if (_select == "BRIGHTNESS" && hwinfo->supportBrightness()) {
			sB_batCritAction_value->show();
			sB_batCritAction_value->setValue(tdeconfig->readNumEntry("batteryCriticalActionValue",0));
		}
	} else {
		// disable tab 
		tB_general->setItemLabel( 0, tB_general->itemLabel(0) + i18n(" - not supported"));
		tB_general->setItemEnabled( 0, false);
	}

	// buttons tab:
	TQStringList _actions = tdeconfig->readListEntry("buttonsAllowedActions", TQString());
	fillActionComboBox(cB_PowerButton, _actions, tdeconfig->readEntry("ActionOnPowerButton", TQString()));
	fillActionComboBox(cB_SuspendButton, _actions, tdeconfig->readEntry("ActionOnSuspendButton", TQString()));
	fillActionComboBox(cB_S2DiskButton, _actions, tdeconfig->readEntry("ActionOnS2DiskButton", TQString()));
	// avoid logout dialog for lidclose - this make no sense
	_actions.remove("LOGOUT_DIALOG");
	fillActionComboBox(cB_LidcloseButton, _actions, tdeconfig->readEntry("ActionOnLidClose", TQString()));

	// default scheme tab:
	TQString _ac_scheme = tdeconfig->readEntry( "ac_scheme", "Performance");
	TQString _bat_scheme = tdeconfig->readEntry( "battery_scheme", "Powersave");
	cB_acScheme->setCurrentItem(schemes.findIndex(_ac_scheme));
	cB_batteryScheme->setCurrentItem(schemes.findIndex(_bat_scheme));

	kdDebugFuncOut(trace);
}

/* ---- END General setup SECTION ---- */

/* ---- START store settings SECTION ---- */

/*! 
 * This used to save changes in settings of the current scheme.
 */
void ConfigureDialog::saveSchemeSettings() {
	kdDebugFuncIn(trace);

	TQString s_scheme = getSchemeRealName(schemes[currentScheme]);
	tdeconfig->setGroup(s_scheme);

	tdeconfig->writeEntry("specSsSettings",cB_specificSettings->isOn());
	tdeconfig->writeEntry("disableSs",cB_disable_Ss->isOn());
	tdeconfig->writeEntry("blankSs",cB_blankScreen->isOn());
	tdeconfig->writeEntry("specPMSettings",cB_SpecificPM->isOn());
	tdeconfig->writeEntry("disableDPMS",cB_disablePM->isOn());
	
	tdeconfig->writeEntry("standbyAfter",sB_standby->value());
	tdeconfig->writeEntry("suspendAfter",sB_suspend->value());
	tdeconfig->writeEntry("powerOffAfter",sB_powerOff->value());

	tdeconfig->writeEntry("disableNotifications",cB_disableNotifications->isOn());
		
	if(cB_autoInactivity->currentText() == " "){
		tdeconfig->writeEntry("autoInactiveAction", "_NONE_");
	}
	else{
		int _index = cB_autoInactivity->currentItem();
		if(_index > 0) {
			/*if(_index == (cB_autoInactivity->count()-1)) {
				tdeconfig->writeEntry("autoInactiveAction","shutdown");	
			}
			else {
				tdeconfig->writeEntry("autoInactiveAction",actions[(_index)]);
			}*/
			tdeconfig->writeEntry("autoInactiveAction",actions[(_index)]);
		}
		tdeconfig->writeEntry("autoInactiveActionAfter",sB_autoInactivity->value());
	}
	tdeconfig->writeEntry("autoSuspend",cB_autoSuspend->isOn());
	tdeconfig->writeEntry("autoInactiveSchemeBlacklistEnabled",cB_Blacklist->isOn());
	
	tdeconfig->writeEntry("autoDimm",cB_autoDimm->isOn());
	tdeconfig->writeEntry("autoDimmAfter", sB_autoDimmTime->value());
	tdeconfig->writeEntry("autoDimmTo", sB_autoDimmTo->value());
	tdeconfig->writeEntry("autoDimmSchemeBlacklistEnabled",cB_BlacklistDimm->isOn());

	tdeconfig->writeEntry("enableBrightness",cB_Brightness->isOn());
	if(brightness_changed)
		tdeconfig->writeEntry("brightnessPercent",brightnessSlider->value());


	if(hwinfo->supportCPUFreq()) {
		switch( comboB_cpuFreq->currentItem()) {
			case 0:
				tdeconfig->writeEntry("cpuFreqPolicy", "PERFORMANCE");
				break;
			case 2:
				tdeconfig->writeEntry("cpuFreqPolicy", "POWERSAVE");
				break;
			case 1:
			default:
				tdeconfig->writeEntry("cpuFreqPolicy", "DYNAMIC");
				break;
		}
	}

	tdeconfig->sync();
	scheme_changed = false;
	if(!general_changed) buttonApply->setEnabled(false);

	kdDebugFuncOut(trace);
}

/*! 
 * This used to save changes in settings of the current scheme.
 */
void ConfigureDialog::saveGeneralSettings() {
	kdDebugFuncIn(trace);

	tdeconfig->setGroup("General");
	
	tdeconfig->writeEntry("lockOnSuspend",cB_lockSuspend->isOn());
	tdeconfig->writeEntry("lockOnLidClose",cB_lockLid->isOn());
	tdeconfig->writeEntry("Autostart",cB_autostart->isOn());
	tdeconfig->writeEntry("AutostartNeverAsk",cB_autostart_neverAsk->isOn());
	
	TQString selected_method = "";
	int _selected = comboB_lock->currentItem();
	if(_selected == 0) selected_method = "automatic";
	else if(_selected == 1) selected_method = "tdescreensaver";
	else if(_selected == 2) selected_method = "xscreensaver";
	else if(_selected == 3) selected_method = "xlock";
	else if(gnome_session && (_selected == 4)) selected_method = "gnomescreensaver";
	tdeconfig->writeEntry( "lockMethod", selected_method );
	
	tdeconfig->writeEntry("batteryWarning", sB_batWarning->value());
	tdeconfig->writeEntry("batteryLow", sB_batLow->value());
	tdeconfig->writeEntry("batteryCritical", sB_batCritical->value());

	// battery level tab
	TQString _action = mapDescriptionToAction(cB_batWarning->currentText());
	tdeconfig->writeEntry("batteryWarningAction", _action);
	if (_action == "BRIGHTNESS") {
		tdeconfig->writeEntry("batteryWarningActionValue", sB_batWarnAction_value->value());
	}
	_action = mapDescriptionToAction(cB_batLow->currentText());
	tdeconfig->writeEntry("batteryLowAction", _action);
	if (_action == "BRIGHTNESS") {
		tdeconfig->writeEntry("batteryLowActionValue", sB_batLowAction_value->value());
	}
	_action = mapDescriptionToAction(cB_batCritical->currentText());
	tdeconfig->writeEntry("batteryCriticalAction", _action);
	if (_action == "BRIGHTNESS") {
		tdeconfig->writeEntry("batteryCriticalActionValue", sB_batCritAction_value->value());
	}

	// button tab
	tdeconfig->writeEntry("ActionOnPowerButton", mapDescriptionToAction(cB_PowerButton->currentText()));
	tdeconfig->writeEntry("ActionOnLidClose", mapDescriptionToAction(cB_LidcloseButton->currentText()));
	tdeconfig->writeEntry("ActionOnSuspendButton", mapDescriptionToAction(cB_SuspendButton->currentText()));
	tdeconfig->writeEntry("ActionOnS2DiskButton", mapDescriptionToAction(cB_S2DiskButton->currentText()));

	// schemes tab
	tdeconfig->writeEntry("ac_scheme", getSchemeRealName(schemes[cB_acScheme->currentItem()]));
	tdeconfig->writeEntry("battery_scheme", getSchemeRealName(schemes[cB_batteryScheme->currentItem()]));

	tdeconfig->sync();
	general_changed = false;
	if(!scheme_changed) buttonApply->setEnabled(false);

	kdDebugFuncOut(trace);
}

/* ---- END store settings SECTION ---- */

/* ---- START monitor changes SECTION ---- */

/*!
 * SLOT: Called if a value within the Tab 'General Settings' is changed.
 */
void ConfigureDialog::general_valueChanged(){
	kdDebugFuncIn(trace);

	if(initalised) {
		general_changed = true;
		buttonApply->setEnabled(true);
		// enable/disable the comboB_lock and tL_lockWith
		if(cB_lockSuspend->isOn() || cB_lockLid->isOn() ) {
			tL_lockWith->setEnabled(true);
			comboB_lock->setEnabled(true);
		}
		else {
			tL_lockWith->setEnabled(false);
			comboB_lock->setEnabled(false);
		}
	}

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: Called if a value within the Tab 'Scheme Settings' is changed.
 */
void ConfigureDialog::scheme_valueChanged(){
	kdDebugFuncIn(trace);

	if(initalised) {
		scheme_changed = true;
		buttonApply->setEnabled(true);
	}

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: called if the current scheme in the TQListBox listBox_schemes is
 * changed/ a other scheme was selected.
 */
void ConfigureDialog::listBox_schemes_currentChanged(){
	kdDebugFuncIn(trace);

	// store current settings if something changed
	if(initalised && scheme_changed ) {
		int res = KMessageBox::warningYesNo(this,
						    i18n("There are unsaved changes in the active scheme.\n"
							 "Apply the changes before jumping to the next scheme "
							 "or discard the changes?"), 
						    i18n("Unsaved Changes"), KStdGuiItem::apply(),
						    KStdGuiItem::discard());
		
		if (res == KMessageBox::Yes) {
			// Save changes
			saveSchemeSettings();
		}
		if (res == KMessageBox::No) {
			// discard changes and reset trigger
			scheme_changed = false;
		}
	}
	// set to new Item
	setConfigToDialog(listBox_schemes->currentItem());

	kdDebugFuncOut(trace);
}

/* ---- END monitor changes SECTION ---- */

/* ---- START BUTTON SECTION ---- */

/*!
 * SLOT: called if the 'Apply' button is clicked.
 */
void ConfigureDialog::buttonApply_clicked(){
	kdDebugFuncIn(trace);

	if(initalised && scheme_changed ) {
		saveSchemeSettings();
		scheme_changed = false;
	}
	if(initalised && general_changed ) {
		saveGeneralSettings();
		general_changed = false;
	}

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: called if the 'Chancel' button is clicked.
 */
void ConfigureDialog::buttonCancel_clicked(){
	kdDebugFuncOut(trace);

	if (scheme_changed || general_changed) {
		int res = KMessageBox::warningYesNoCancel(this,
             						  i18n("There are unsaved changes.\nApply the changes "
							       "before cancel or discard the changes?"), 
						          i18n("Unsaved Changes"), KStdGuiItem::apply(),
							  KStdGuiItem::discard());
		if (res == KMessageBox::Yes) {
			buttonApply_clicked();
		}
		else if (res == KMessageBox::Cancel) return;
	}

	kdDebugFuncOut(trace);
	close();
}

/*!
 * SLOT: called if the 'OK' button is clicked.
 */
void ConfigureDialog::buttonOk_clicked(){
	kdDebugFuncIn(trace);

	buttonApply_clicked();
	buttonApply->setEnabled(false);
	
	kdDebugFuncOut(trace);
	close();
}

/*!
 * SLOT: called if the 'Help' button is clicked.
 */
void ConfigureDialog::buttonHelp_clicked(){
	kdDebugFuncIn(trace);

	emit openHelp();

	kdDebugFuncOut(trace);
}

/* ----  END BUTTON SECTION ---- */

/* ---- START SCHEME ADD/DELETE SECTION ---- */

/*!
 * SLOT: called if the 'New' scheme button is clicked.
 */
void ConfigureDialog::pB_newScheme_clicked(){
	kdDebugFuncIn(trace);

	bool _ok = false;
	bool _end = false;
	TQString _new;
	TQString _text = i18n("Please insert a name for the new scheme:");
	TQString _error;

	getSchemeList();

	while (!_end) {
		_new = KInputDialog::getText( i18n("KPowersave Configuration"), 
					      _error + _text, TQString(), &_ok, this);
		if (!_ok ) {
			_end = true;
		} else {
			_error = TQString();
			if (!_new.isEmpty()) {
				if ( schemes.contains(_new))
					_error = i18n("Error: A scheme with this name already exist.\n");
				else 
					_end = true;
			}
		}
	}

	if (!_new.isEmpty()) {
		// write append new scheme to list
		schemes.append(_new);
		tdeconfig->setGroup("General");
		tdeconfig->writeEntry("schemes", schemes, ",");
		tdeconfig->sync();

		// update GUI
		setSchemeList();
		selectScheme(_new);
		saveSchemeSettings();
	}
	
	kdDebugFuncOut(trace);
}

/*!
 * SLOT: called if the 'Delete' scheme button is clicked.
 */
void ConfigureDialog::pB_deleteScheme_clicked(){
	kdDebugFuncIn(trace);

	if (pB_deleteScheme->isEnabled()) {
		int answer = KMessageBox::questionYesNo( this, i18n("Do you really want to delete the "
							         "%1 scheme?").arg(schemes[currentScheme]),
							 i18n("Confirm delete scheme"),
							 i18n("Delete"), i18n("Cancel"));
		if (answer == KMessageBox::Yes) {
			// delete the scheme, we can be sure this is a userscheme
			TQString _s_tmp = getSchemeRealName(schemes[currentScheme]);
			if (tdeconfig->hasGroup(_s_tmp) && tdeconfig->deleteGroup(_s_tmp)) {
				schemes.remove(_s_tmp);
				tdeconfig->setGroup("General");
				tdeconfig->writeEntry("schemes", schemes, ",");
				tdeconfig->sync();

				// update GUI
				setSchemeList();
				selectScheme(settings->currentScheme);
			} else {
				// could not delete the scheme ... error case
				KMessageBox::queuedMessageBox(this, KMessageBox::Error, 
							      i18n("Could not delete the selected scheme."));
			}
		}
	}

	kdDebugFuncOut(trace);
}

/* ----  END SCHEME ADD/DELETE SECTION ---- */

/* ---- START SCREENSAVER SECTION ---- */

/*!
 * SLOT: called if TQCheckBox cB_specificSettings is toggled.
 * \param state boolean, true if toggled on
 *		   	 false if toggled off
 */
void ConfigureDialog::cB_specificSettings_toggled(bool state){
	kdDebugFuncIn(trace);

	if(!initalised) cB_specificSettings->setChecked(state);
	
	/* set widgets visible/disabled */
	if(state){
		cB_disable_Ss->setEnabled(true);
		if(cB_disable_Ss->isOn()){
			cB_blankScreen->setEnabled(false);
		}
		else {
			cB_blankScreen->setEnabled(true);
		}
	}
	else {
		cB_disable_Ss->setEnabled(false);
		cB_blankScreen->setEnabled(false);
	}

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: called if TQCheckBox cB_disable_Ss is toggled.
 * \param state boolean, true if toggled on
 *		   	 false if toggled off
 */
void ConfigureDialog::cB_disable_Ss_toggled(bool state){
	kdDebugFuncIn(trace);

	/* set widgets visible/disabled */
	if(!initalised) cB_disable_Ss->setChecked(state);
	cB_blankScreen->setEnabled(!state);
	
	kdDebugFuncOut(trace);
}


/* ---- END SCREENSAVER SECTION ---- */

/* ---- START DPMS SECTION ---- */

/*!
 * SLOT: called if TQCheckBox cB_SpecificPM is toggled.
 * \param state boolean, true if toggled on
 *		   	 false if toggled offtrue
 */
void ConfigureDialog::cB_SpecificPM_toggled(bool state){
	kdDebugFuncIn(trace);

	if(!initalised) cB_SpecificPM->setChecked(state);
	
	/* set widgets visible/disabled */
	cB_disablePM->setEnabled(state);
	if(cB_disablePM->isOn()) {
		state = false;
	}
	tL_standbyAfter->setEnabled(state);
	sB_standby->setEnabled(state);
	tL_suspendAfter->setEnabled(state);
	sB_suspend->setEnabled(state);
	tL_powerOffAfter->setEnabled(state);
	sB_powerOff->setEnabled(state);

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: called if TQCheckBox cB_disablePM is toggled.
 * \param state boolean, true if toggled on
 *			 false if toggled off
 */
void ConfigureDialog::cB_disablePM_toggled(bool state){
	kdDebugFuncIn(trace);

	if(!initalised) cB_disablePM->setChecked(state);

	/* set widgets visible/disabled */
	tL_standbyAfter->setEnabled(!state);
	sB_standby->setEnabled(!state);
	tL_suspendAfter->setEnabled(!state);
	sB_suspend->setEnabled(!state);
	tL_powerOffAfter->setEnabled(!state);
	sB_powerOff->setEnabled(!state);

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: called if the standby TQSpinBoxes for DPMS timout is changed.
 */
void ConfigureDialog::sB_standby_valueChanged() {
	kdDebugFuncIn(trace);

	if (initalised) {
		if (sB_standby->value() == 0 ) 
			return;
		if ( sB_standby->value() > sB_suspend->value()) {
			sB_suspend->setValue(sB_standby->value());
		}
	}

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: called if the suspend TQSpinBoxes for DPMS timout is changed.
 */
void ConfigureDialog::sB_suspend_valueChanged() {
	kdDebugFuncIn(trace);

	if (initalised) {
		if (sB_suspend->value() == 0 ) 
			return;

		if ( sB_suspend->value() < sB_standby->value()) {
			sB_standby->setValue(sB_suspend->value());
		}
		if ( sB_suspend->value() > sB_powerOff->value()) {
			sB_powerOff->setValue(sB_suspend->value());
		}
	}
	
	kdDebugFuncOut(trace);
}

/*!
 * SLOT: called if the powerOff TQSpinBoxes for DPMS timout is changed.
 */
void ConfigureDialog::sB_powerOff_valueChanged() {
	kdDebugFuncIn(trace);

	if (initalised) {
		if (sB_powerOff->value() == 0 ) 
			return;
		if ( sB_powerOff->value() < sB_suspend->value()) {
			sB_suspend->setValue(sB_powerOff->value());
		}
	}

	kdDebugFuncOut(trace);
}

/* ----  END DPMS SECTION ---- */

/* ---- START Inactivity SECTION ---- */

/*! 
 * This used to set the inactivity related combobox cB_autoInactivity .
 */
void ConfigureDialog::setInactivityBox(){
	kdDebugFuncIn(trace);

	cB_autoInactivity->clear();
	// add "nothing" at start of the list
	actions.push_front(" ");
	// add "Turn Off Computer" at end of the list
	// TQString _to_i18n = i18n("Turn Off Computer");
	// actions.append("Turn Off Computer");
	
	for ( TQStringList::Iterator it = actions.begin(); it != actions.end(); ++it ) {
		cB_autoInactivity->insertItem( i18n( (*it).utf8() ) );
	}

	kdDebugFuncOut(trace);
}

/*! 
 * This used to set the autosuspend related widgets.
 */
void ConfigureDialog::cB_autoSuspend_toggled( bool toggled ) {
	kdDebugFuncIn(trace);

	if(cB_autoSuspend->isOn() != toggled)
		cB_autoSuspend->setChecked(toggled);	

	tL_autoInactivity_explain->setEnabled(toggled);
	cB_autoInactivity->setEnabled(toggled);
	if(cB_autoInactivity->currentItem() > 0) {
		cB_autoInactivity->setEnabled(true);
		tL_autoInactivity_After->setEnabled(true);
		sB_autoInactivity->setEnabled(true);
		cB_Blacklist->setEnabled(true);
		cB_Blacklist_toggled(cB_Blacklist->isOn());
	}
	
	if(!toggled) {
		cB_autoSuspend->setChecked(false);
		tL_autoInactivity_After->setEnabled(false);
		cB_autoInactivity->setEnabled(false);
		sB_autoInactivity->setEnabled(false);
		cB_Blacklist->setEnabled(false);
		pB_editBlacklist->setEnabled(false);
	}
	
	scheme_changed = true;
	buttonApply->setEnabled(true);

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: called if a new item in cB_autoInactivity is selected.
 * \param selectedItem Integer, contains the ID of the new item in the
 *				comboBox.
 */
void ConfigureDialog::cB_autoInactivity_activated( int selectedItem ) {
	kdDebugFuncIn(trace);

	if( actions[selectedItem] != " " ) {
		if(!displayed_WARN_autosuspend && initalised) {
			TQString _msg = "<qt>" + i18n("<b>Note:</b> If you select this option, the computer "
						     "will suspend or standby if the current user is "
						     "inactive for the defined time even if somebody is "
						     "logged in remotely to the X server.<br><br> This "
						     "feature can also produce problems with some programs, "
						     "such as video players or cd burner. These programs can "
						     "be blacklisted by checking <b>Enable scheme-specific "
						     "blacklist</b> and click <b>Edit Blacklist...</b>. If "
						     "this does not help, report the problem or deactivate "
						     "autosuspend.<br><br> Really use this option?") +
				       "</qt>";

			int tmp = KMessageBox::warningContinueCancel(this, _msg);
			if (tmp ==  KMessageBox::Cancel) {
				selectedItem = 0;
			}
			displayed_WARN_autosuspend = true;
		}
		sB_autoInactivity->setEnabled(true);
		tL_autoInactivity_After->setEnabled(true);
		cB_Blacklist->setEnabled(true);
		if(cB_Blacklist->isChecked()) pB_editBlacklist->setEnabled(true);
	
	}
	else {
		sB_autoInactivity->setEnabled(false);
		tL_autoInactivity_After->setEnabled(false);
		cB_Blacklist->setEnabled(false);
		pB_editBlacklist->setEnabled(false);
	}
	
	cB_autoInactivity->setCurrentItem( selectedItem );
	
	if(initalised) {
		scheme_changed = true;
		buttonApply->setEnabled(true);
	}

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: Called if the 'Enable scheme specific blacklist' is toggled.
 * \param toggled boolean, true if toggled on
 *			   false if toggled off
 */
void ConfigureDialog::cB_Blacklist_toggled( bool toggled ){
	kdDebugFuncIn(trace);

	pB_editBlacklist->setEnabled(toggled);
	
	if(initalised) {
		buttonApply->setEnabled(true);
		scheme_changed = true;
	} else {
		cB_Blacklist->setChecked(toggled);
	}

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: Called if the 'edit blacklist' button is toggled.
 */
void ConfigureDialog::pB_editBlacklistSuspend_clicked(){
	kdDebugFuncIn(trace);

	TQString _top_text = "";
	bool initialiseImport = false;

	if(tabWidget->currentPageIndex() == 0 ) {
		TQString s_scheme = getSchemeRealName(schemes[listBox_schemes->currentItem()]);
		_top_text = listBox_schemes->currentText();
		if (tdeconfig->hasGroup(s_scheme)){
			tdeconfig->setGroup(s_scheme);
		}
		blacklist = tdeconfig->readListEntry("autoInactiveSchemeBlacklist", ',');
		if( blacklist.empty()) {
				TQString _msg = i18n("The blacklist of the selected scheme is empty. "
					    "Import the general blacklist?");
			int tmp = KMessageBox::questionYesNo(this, _msg, TQString(), i18n("Import"), i18n("Do Not Import"));
			if (tmp ==  KMessageBox::Yes) {
				initialiseImport = true;
				if(tdeconfig->hasGroup("General")){
					tdeconfig->setGroup("General");
					blacklist = tdeconfig->readListEntry("autoInactiveBlacklist", ',');
				}
			}
		}
	}
	else {
		if(tdeconfig->hasGroup("General")){
			_top_text = i18n("General Autosuspend Blacklist");
			tdeconfig->setGroup("General");
			blacklist = tdeconfig->readListEntry("autoInactiveBlacklist", ',');
		}
	}
	blacklistEDlgAS = new blacklistEditDialog(blacklist, _top_text, initialiseImport, this);
	
	connect( blacklistEDlgAS, TQT_SIGNAL(config_finished(TQStringList)), this, 
		 TQT_SLOT(saveSchemeSuspendBlacklist(TQStringList)));
	blacklistEDlgAS->exec();

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: Called if the signal config_finished(TQStringList) recieved
 * and the autosuspend blacklist is edited.
 * \param new_blacklist TQStringlist with the edited blacklisted processes
 */

void ConfigureDialog::saveSchemeSuspendBlacklist( TQStringList new_blacklist){
	kdDebugFuncIn(trace);

	if(tabWidget->currentPageIndex() == 0 ) {
		TQString s_scheme = getSchemeRealName(schemes[currentScheme]);
		tdeconfig->setGroup(s_scheme);
		tdeconfig->writeEntry("autoInactiveSchemeBlacklist", new_blacklist, ',');
	}
	else {
		tdeconfig->setGroup("General");
		tdeconfig->writeEntry("autoInactiveBlacklist", new_blacklist, ',');
	}

	tdeconfig->sync();
	kdDebugFuncOut(trace);
}

/* ---- END Inactivity SECTION ---- */

/* ---- START Autodimm SECTION ---- */

/*! 
 * This used to set the autodimm related widgets.
 */
void ConfigureDialog::cB_autoDimm_toggled( bool toggled ) {
	kdDebugFuncIn(trace);

	if(cB_autoDimm->isOn() != toggled)
		cB_autoDimm->setChecked(toggled);	

	if (toggled)
	tL_autoDimmExplain->setEnabled(toggled);
	tL_autoDimmAfter->setEnabled(toggled);
	sB_autoDimmTime->setEnabled(toggled);
	tL_autoDimmTo->setEnabled(toggled);
	sB_autoDimmTo->setEnabled(toggled);

	if (sB_autoDimmTime->value() > 0)
		cB_BlacklistDimm->setEnabled(toggled);

	if (cB_BlacklistDimm->isOn())
		pB_editBlacklistDimm->setEnabled(toggled);
	
	if (toggled) {
		if (sB_autoDimmTime->value() > 0) {
			tL_autoDimmTo->setEnabled(true);
			sB_autoDimmTo->setEnabled(true);
		} else {
			tL_autoDimmTo->setEnabled(false);
			sB_autoDimmTo->setEnabled(false);
		}
	}

	scheme_changed = true;
	buttonApply->setEnabled(true);

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: Called if the 'Enable scheme specific blacklist' is toggled.
 * \param toggled boolean, true if toggled on
 *			   false if toggled off
 */
void ConfigureDialog::cB_BlacklistDimm_toggled( bool toggled ){
	kdDebugFuncIn(trace);

	pB_editBlacklistDimm->setEnabled(toggled);
	
	if(initalised) {
		buttonApply->setEnabled(true);
		scheme_changed = true;
	} else {
		cB_BlacklistDimm->setChecked(toggled);
	}

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: Called if the 'edit blacklist' button for autodimm is toggled.
 */
void ConfigureDialog::pB_editBlacklistDimm_clicked(){
	kdDebugFuncIn(trace);

	TQString _top_text = "";
	bool initialiseImport = false;

	if(tabWidget->currentPageIndex() == 0 ) {
		TQString s_scheme = getSchemeRealName(schemes[listBox_schemes->currentItem()]);
		_top_text = listBox_schemes->currentText();
		if (tdeconfig->hasGroup(s_scheme)){
			tdeconfig->setGroup(s_scheme);
		}
		blacklist = tdeconfig->readListEntry("autoDimmSchemeBlacklist", ',');
		if( blacklist.empty()) {
				TQString _msg = i18n("The blacklist of the selected scheme is empty. "
						    "Import the general blacklist?");
			int tmp = KMessageBox::questionYesNo(this, _msg, TQString(), i18n("Import"), i18n("Do Not Import"));
			if (tmp ==  KMessageBox::Yes) {
				initialiseImport = true;
				if(tdeconfig->hasGroup("General")){
					tdeconfig->setGroup("General");
					blacklist = tdeconfig->readListEntry("autoDimmBlacklist", ',');
				}
			}
		}
	}
	else {
		if(tdeconfig->hasGroup("General")){
			_top_text = i18n("General Autodimm Blacklist");
			tdeconfig->setGroup("General");
			blacklist = tdeconfig->readListEntry("autoDimmBlacklist", ',');
		}
	}
	blacklistEDlgAD = new blacklistEditDialog(blacklist, _top_text, initialiseImport, this);
	
	connect( blacklistEDlgAD, TQT_SIGNAL(config_finished(TQStringList)), this, 
		 TQT_SLOT(saveSchemeDimmBlacklist(TQStringList)));
	blacklistEDlgAD->exec();

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: Called if the signal config_finished(TQStringList) recieved
 * and the autdimm blacklist is edited.
 * \param new_blacklist TQStringlist with the edited blacklisted processes
 */
void ConfigureDialog::saveSchemeDimmBlacklist( TQStringList new_blacklist){
	kdDebugFuncIn(trace);

	if(tabWidget->currentPageIndex() == 0 ) {
		TQString s_scheme = getSchemeRealName(schemes[currentScheme]);
		tdeconfig->setGroup(s_scheme);
		tdeconfig->writeEntry("autoDimmSchemeBlacklist", new_blacklist, ',');
	}
	else {
		tdeconfig->setGroup("General");
		tdeconfig->writeEntry("autoDimmBlacklist", new_blacklist, ',');
	}

	tdeconfig->sync();
	kdDebugFuncOut(trace);
}

/*!
 * SLOT: Called if there \ref sB_autoDimmTime get changed
 * \param value Integer with the new value
 */
void ConfigureDialog::sB_autoDimmTime_valueChanged( int value ) {
	kdDebugFuncIn(trace);

	if (value > 0) {
		if (!tL_autoDimmTo->isEnabled()) {
			tL_autoDimmTo->setEnabled(true);
			sB_autoDimmTo->setEnabled(true);
			cB_BlacklistDimm->setEnabled(true);
			if (cB_BlacklistDimm->isOn())
				pB_editBlacklistDimm->setEnabled(true);
		}
	} else {
		if (tL_autoDimmTo->isEnabled()) {
			tL_autoDimmTo->setEnabled(false);
			sB_autoDimmTo->setEnabled(false);
			cB_BlacklistDimm->setEnabled(false);
			pB_editBlacklistDimm->setEnabled(false);
		}
	}
	kdDebugFuncOut(trace);
}


/* ---- END Autodimm SECTION ---- */

/* ---- START Brightness SECTION ---- */
/*! \b SLOT: to enable the brigthness related widgets */
void ConfigureDialog::cB_Brightness_toggled( bool toggled ) {
	kdDebugFuncIn(trace);

	gB_Brightness->setEnabled(toggled);
	cB_Brightness->setChecked(toggled);
	connect(brightnessSlider, TQT_SIGNAL(valueChanged (int)), this, TQT_SLOT(brightnessSlider_sliderMoved(int)));

	kdDebugFuncOut(trace);
}

/*! \b SLOT: to change the brightness if the slider is changed */
void ConfigureDialog::brightnessSlider_sliderMoved( int new_value ) {
	kdDebugFuncIn(trace);

	if (cB_Brightness->isEnabled() && cB_Brightness->isChecked()) {
		scheme_valueChanged();
		tL_valueBrightness->setText(TQString::number(new_value) + " %");
		hwinfo->setBrightness(-1, new_value);
		pB_resetBrightness->setEnabled(true);
		brightness_changed = true;
	}

	kdDebugFuncOut(trace);
}

/*! \b SLOT: to reset the brightness if the reset button clicked */
void ConfigureDialog::pB_resetBrightness_clicked( ) {
	kdDebugFuncIn(trace);

	hwinfo->setBrightness(brightness_last, -1);
	brightnessSlider->setValue(brightness_last);
	pB_resetBrightness->setEnabled(false);
	brightness_changed = false;

	kdDebugFuncOut(trace);
}

/* ---- END Brightness SECTION ---- */

/* ---- START battery level SECTION ---- */
/*!
 * SLOT: called if the warning TQSpinBoxes for battery level is changed.
 */
void ConfigureDialog::sB_batWarning_valueChanged() {
	kdDebugFuncIn(trace);
	
	if (initalised) {
		if (sB_batWarning->value() == 0 ) 
			sB_batWarning->setValue(1);
		if ( sB_batWarning->value() <= sB_batLow->value()) {
			sB_batLow->setValue(sB_batWarning->value()-1);
		}
	}

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: called if the low TQSpinBoxes for battery level is changed.
 */
void ConfigureDialog::sB_batLow_valueChanged() {
	kdDebugFuncIn(trace);

	if (initalised) {
		if (sB_batLow->value() == 0 ) 
			sB_batLow->setValue(1);

		if ( sB_batLow->value() >= sB_batWarning->value()) {
			sB_batWarning->setValue(sB_batLow->value()+1);
		}
		if ( sB_batLow->value() <= sB_batCritical->value()) {
			sB_batCritical->setValue(sB_batLow->value()-1);
		}
	}

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: called if the critical TQSpinBoxes for battery level is changed.
 */
void ConfigureDialog::sB_batCritical_valueChanged() {
	kdDebugFuncIn(trace);

	if (initalised) {
		if (sB_batCritical->value() == 0 ) 
			sB_batCritical->setValue(1);
		if ( sB_batCritical->value() >= sB_batLow->value()) {
			sB_batLow->setValue(sB_batCritical->value()+1);
		}
	}

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: called if the TQComboBox for the battery warning level action changed,
 *       used to hide/show the related TQSpinboxes if needed.
 */
void ConfigureDialog::cB_batWarning_activated() {
	kdDebugFuncIn(trace);

	if ( mapDescriptionToAction(cB_batWarning->currentText()) == "BRIGHTNESS") {
		sB_batWarnAction_value->show();
	} else {
		sB_batWarnAction_value->hide();
	}

	kdDebugFuncOut(trace);
}

//! called if the TQComboBox for the battery low level action changed
/*!
 *       used to hide/show the related TQSpinboxes if needed.
 */
void ConfigureDialog::cB_batLow_activated() {
	kdDebugFuncIn(trace);

	if ( mapDescriptionToAction(cB_batLow->currentText()) == "BRIGHTNESS") {
		sB_batLowAction_value->show();
	} else {
		sB_batLowAction_value->hide();
	}

	kdDebugFuncOut(trace);
}

/*!
 * SLOT: called if the TQComboBox for the battery critical level action changed,
 *       used to hide/show the related TQSpinboxes if needed.
 */
void ConfigureDialog::cB_batCritical_activated() {
	kdDebugFuncIn(trace);

	if ( mapDescriptionToAction(cB_batCritical->currentText()) == "BRIGHTNESS") {
		sB_batCritAction_value->show();
	} else {
		sB_batCritAction_value->hide();
	}

	kdDebugFuncOut(trace);
}

/* ---- END battery level SECTION ---- */

/*! \b SLOT: to open the KNotify config dialog */
void ConfigureDialog::pB_configNotify_released( ) {
	kdDebugFuncIn(trace);

	emit openKNotify();

	kdDebugFuncOut(trace);
}

/* ---- START helper functions SECTION ---- */
/*!
 * Map the key from config for a action to a descriptive name.
 * \param action	TQString with the config key value
 * \return 		TQString with the description
 */
TQString ConfigureDialog::mapActionToDescription( TQString action ) {
	kdDebugFuncIn(trace);

	TQString ret;

	if (action.startsWith("SHUTDOWN")) {
		ret = i18n("Shutdown");
	} else if (action.startsWith("LOGOUT_DIALOG")) {
		ret = i18n("Logout Dialog");
	} else if (action.startsWith("SUSPEND2DISK")) {
		if (actions.contains("Hibernate"))
			ret = i18n("Hibernate");
	} else if (action.startsWith("SUSPEND2RAM")) {
		if (actions.contains("Suspend"))
			ret = i18n("Suspend");
	} else if (action.startsWith("CPUFREQ_POWERSAVE")) {
		if (hwinfo->supportCPUFreq())
			ret = i18n("CPU Powersave policy");
	} else if (action.startsWith("CPUFREQ_DYNAMIC")) {
		if (hwinfo->supportCPUFreq())
			ret = i18n("CPU Dynamic policy");
	} else if (action.startsWith("CPUFREQ_PERFORMANCE")) {
		if (hwinfo->supportCPUFreq())
			ret = i18n("CPU Performance policy");
	} else if (action.startsWith("BRIGHTNESS")) {
		if (hwinfo->supportBrightness())
			ret = i18n("Set Brightness to");
	} 

	kdDebugFuncOut(trace);
	return ret;
}

/*!
 * Map the action description (maybe translated) back to a TQString key for the config.
 * \param description	TQString with the description
 * \return 		TQString with the config key value
 */
TQString ConfigureDialog::mapDescriptionToAction( TQString description ) {
	kdDebugFuncIn(trace);

	TQString ret;

	if (description.startsWith("Shutdown") || description.startsWith(i18n("Shutdown"))) {
		ret = "SHUTDOWN";
	} else if (description.startsWith("Logout Dialog") || 
		   description.startsWith(i18n("Logout Dialog"))) {
		ret = "LOGOUT_DIALOG";
	} else if (description.startsWith("Hibernate") || 
		   description.startsWith(i18n("Hibernate"))) {
		ret = "SUSPEND2DISK";
	} else if (description.startsWith("Suspend") || 
		   description.startsWith(i18n("Suspend"))) {
		ret = "SUSPEND2RAM";
	} else if (description.startsWith("CPU Powersave policy") || 
		   description.startsWith(i18n("CPU Powersave policy"))) {
		ret = "CPUFREQ_POWERSAVE";
	} else if (description.startsWith("CPU Dynamic policy") || 
		   description.startsWith(i18n("CPU Dynamic policy"))) {
		ret = "CPUFREQ_DYNAMIC";
	} else if (description.startsWith("CPU Performance policy") || 
		   description.startsWith(i18n("CPU Performance policy"))) {
		ret = "CPUFREQ_PERFORMANCE";
	} else if (description.startsWith("Set Brightness to") || 
		   description.startsWith(i18n("Set Brightness to"))) {
		ret = "BRIGHTNESS";
	} 

	kdDebugFuncOut(trace);
	return ret;
}

/*!
 * Helper to fill a TQComboBox with their text and select the correct item
 * \param _cb 		Pointer to the TQComboBox
 * \param _actions	TQStringList with the allowed actions
 * \param _select	TQString with the action to select
 */
void ConfigureDialog::fillActionComboBox(TQComboBox *_cb, TQStringList _actions, TQString _select) {
	kdDebugFuncIn(trace);

	_cb->clear();
	_cb->insertItem("");

	for ( TQStringList::Iterator it = _actions.begin(); it != _actions.end(); ++it ) {
		TQString _tmp = *it;

		TQString _desc = mapActionToDescription( _tmp );
		if (!_desc.isEmpty()) {
			_cb->insertItem( _desc );
			if (_tmp == _select) {
				_cb->setCurrentItem(_cb->count()-1);
			}
		}
	}

	kdDebugFuncOut(trace);
}

/* ---- END helper functions SECTION ---- */

#include "configuredialog.moc"
