/***************************************************************************
 *
 * Copyright (C) 2008 by Kevin Krammer <k.krammer@gmx.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include "xmlmarshaller.h"

#include <tqdbusdata.h>
#include <tqdbusdatalist.h>
#include <tqdbusdatamap.h>
#include <tqdbusobjectpath.h>
#include <tqdbusvariant.h>

#include <tqdom.h>
#include <tqvaluelist.h>

#include <kdebug.h>

static TQDomElement fromBool(bool value, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("bool");
    TQDomText    text    = doc.createTextNode((value ? "true" : "false"));

    element.appendChild(text);

    return element;
}

static TQDomElement fromByte(TQ_UINT8 value, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("byte");
    TQDomText    text    = doc.createTextNode(TQString::number(value));

    element.appendChild(text);

    return element;
}

static TQDomElement fromInt16(TQ_INT16 value, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("int16");
    TQDomText    text    = doc.createTextNode(TQString::number(value));

    element.appendChild(text);

    return element;
}

static TQDomElement fromUInt16(TQ_UINT16 value, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("uin16");
    TQDomText    text    = doc.createTextNode(TQString::number(value));

    element.appendChild(text);

    return element;
}

static TQDomElement fromInt32(TQ_INT32 value, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("int32");
    TQDomText    text    = doc.createTextNode(TQString::number(value));

    element.appendChild(text);

    return element;
}

static TQDomElement fromUInt32(TQ_UINT32 value, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("uint32");
    TQDomText    text    = doc.createTextNode(TQString::number(value));

    element.appendChild(text);

    return element;
}

static TQDomElement fromInt64(TQ_INT64 value, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("int64");
    TQDomText    text    = doc.createTextNode(TQString::number(value));

    element.appendChild(text);

    return element;
}

static TQDomElement fromUInt64(TQ_UINT64 value, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("uint64");
    TQDomText    text    = doc.createTextNode(TQString::number(value));

    element.appendChild(text);

    return element;
}

static TQDomElement fromDouble(double value, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("double");
    TQDomText    text    = doc.createTextNode(TQString::number(value));

    element.appendChild(text);

    return element;
}

static TQDomElement fromString(const TQString& value, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("string");
    TQDomText    text    = doc.createTextNode(value); // TODO: espace

    element.appendChild(text);

    return element;
}

static TQDomElement fromObjectPath(const TQT_DBusObjectPath& value, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("objectpath");
    TQDomText    text    = doc.createTextNode(value);

    element.appendChild(text);

    return element;
}

static TQDomElement fromByteKeyMap(const TQT_DBusDataMap<TQ_UINT8>& map, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("map");

    if (map.isEmpty()) return element;

    TQT_DBusDataMap<TQ_UINT8>::const_iterator it    = map.begin();
    TQT_DBusDataMap<TQ_UINT8>::const_iterator endIt = map.end();
    for (; it != endIt; ++it)
    {
        TQDomElement entryElement = doc.createElement("entry");

        entryElement.appendChild(fromByte(it.key(), ownerDoc));
        entryElement.appendChild(XMLMarshaller::fromTQT_DBusData(it.data(), ownerDoc));

        element.appendChild(entryElement);
    }

    return element;
}

static TQDomElement fromInt16KeyMap(const TQT_DBusDataMap<TQ_INT16>& map, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("map");

    if (map.isEmpty()) return element;

    TQT_DBusDataMap<TQ_INT16>::const_iterator it    = map.begin();
    TQT_DBusDataMap<TQ_INT16>::const_iterator endIt = map.end();
    for (; it != endIt; ++it)
    {
        TQDomElement entryElement = doc.createElement("entry");

        entryElement.appendChild(fromInt16(it.key(), ownerDoc));
        entryElement.appendChild(XMLMarshaller::fromTQT_DBusData(it.data(), ownerDoc));

        element.appendChild(entryElement);
    }

    return element;
}

static TQDomElement fromUInt16KeyMap(const TQT_DBusDataMap<TQ_UINT16>& map, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("map");

    if (map.isEmpty()) return element;

    TQT_DBusDataMap<TQ_UINT16>::const_iterator it    = map.begin();
    TQT_DBusDataMap<TQ_UINT16>::const_iterator endIt = map.end();
    for (; it != endIt; ++it)
    {
        TQDomElement entryElement = doc.createElement("entry");

        entryElement.appendChild(fromUInt16(it.key(), ownerDoc));
        entryElement.appendChild(XMLMarshaller::fromTQT_DBusData(it.data(), ownerDoc));

        element.appendChild(entryElement);
    }

    return element;
}

static TQDomElement fromInt32KeyMap(const TQT_DBusDataMap<TQ_INT32>& map, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("map");

    if (map.isEmpty()) return element;

    TQT_DBusDataMap<TQ_INT32>::const_iterator it    = map.begin();
    TQT_DBusDataMap<TQ_INT32>::const_iterator endIt = map.end();
    for (; it != endIt; ++it)
    {
        TQDomElement entryElement = doc.createElement("entry");

        entryElement.appendChild(fromInt32(it.key(), ownerDoc));
        entryElement.appendChild(XMLMarshaller::fromTQT_DBusData(it.data(), ownerDoc));

        element.appendChild(entryElement);
    }

    return element;
}

static TQDomElement fromUInt32KeyMap(const TQT_DBusDataMap<TQ_UINT32>& map, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("map");

    if (map.isEmpty()) return element;

    TQT_DBusDataMap<TQ_UINT32>::const_iterator it    = map.begin();
    TQT_DBusDataMap<TQ_UINT32>::const_iterator endIt = map.end();
    for (; it != endIt; ++it)
    {
        TQDomElement entryElement = doc.createElement("entry");

        entryElement.appendChild(fromUInt32(it.key(), ownerDoc));
        entryElement.appendChild(XMLMarshaller::fromTQT_DBusData(it.data(), ownerDoc));

        element.appendChild(entryElement);
    }

    return element;
}

static TQDomElement fromInt64KeyMap(const TQT_DBusDataMap<TQ_INT64>& map, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("map");

    if (map.isEmpty()) return element;

    TQT_DBusDataMap<TQ_INT64>::const_iterator it    = map.begin();
    TQT_DBusDataMap<TQ_INT64>::const_iterator endIt = map.end();
    for (; it != endIt; ++it)
    {
        TQDomElement entryElement = doc.createElement("entry");

        entryElement.appendChild(fromInt16(it.key(), ownerDoc));
        entryElement.appendChild(XMLMarshaller::fromTQT_DBusData(it.data(), ownerDoc));

        element.appendChild(entryElement);
    }

    return element;
}

static TQDomElement fromUInt64KeyMap(const TQT_DBusDataMap<TQ_UINT64>& map, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("map");

    if (map.isEmpty()) return element;

    TQT_DBusDataMap<TQ_UINT64>::const_iterator it    = map.begin();
    TQT_DBusDataMap<TQ_UINT64>::const_iterator endIt = map.end();
    for (; it != endIt; ++it)
    {
        TQDomElement entryElement = doc.createElement("entry");

        entryElement.appendChild(fromUInt64(it.key(), ownerDoc));
        entryElement.appendChild(XMLMarshaller::fromTQT_DBusData(it.data(), ownerDoc));

        element.appendChild(entryElement);
    }

    return element;
}

static TQDomElement fromStringKeyMap(const TQT_DBusDataMap<TQString>& map, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("map");

    if (map.isEmpty()) return element;

    TQT_DBusDataMap<TQString>::const_iterator it    = map.begin();
    TQT_DBusDataMap<TQString>::const_iterator endIt = map.end();
    for (; it != endIt; ++it)
    {
        TQDomElement entryElement = doc.createElement("entry");

        entryElement.appendChild(fromString(it.key(), ownerDoc));
        entryElement.appendChild(XMLMarshaller::fromTQT_DBusData(it.data(), ownerDoc));

        element.appendChild(entryElement);
    }

    return element;
}

static TQDomElement fromObjectPathKeyMap(const TQT_DBusDataMap<TQT_DBusObjectPath>& map, const TQDomDocument& ownerDoc)
{
    TQDomDocument doc    = ownerDoc;
    TQDomElement element = doc.createElement("map");

    if (map.isEmpty()) return element;

    TQT_DBusDataMap<TQT_DBusObjectPath>::const_iterator it    = map.begin();
    TQT_DBusDataMap<TQT_DBusObjectPath>::const_iterator endIt = map.end();
    for (; it != endIt; ++it)
    {
        TQDomElement entryElement = doc.createElement("entry");

        entryElement.appendChild(fromObjectPath(it.key(), ownerDoc));
        entryElement.appendChild(XMLMarshaller::fromTQT_DBusData(it.data(), ownerDoc));

        element.appendChild(entryElement);
    }

    return element;
}

static bool toBool(const TQDomElement& element)
{
    return element.text() == "true";
}

static TQ_UINT8 toByte(const TQDomElement& element)
{
    uint number = element.text().toUInt();
    if (number > 255) return 0;
    return number;
}

static TQ_INT16 toInt16(const TQDomElement& element)
{
    return element.text().toShort();
}

static TQ_UINT16 toUInt16(const TQDomElement& element)
{
    return element.text().toUShort();
}

static TQ_INT32 toInt32(const TQDomElement& element)
{
    return element.text().toInt();
}

static TQ_UINT32 toUInt32(const TQDomElement& element)
{
    return element.text().toUInt();
}

static TQ_INT64 toInt64(const TQDomElement& element)
{
    return element.text().toLongLong();
}

static TQ_UINT64 toUInt64(const TQDomElement& element)
{
    return element.text().toULongLong();
}

static double toDouble(const TQDomElement& element)
{
    return element.text().toDouble();
}

static TQString toString(const TQDomElement& element)
{
    return element.text();
}

static TQT_DBusObjectPath toObjectPath(const TQDomElement& element)
{
    return element.text();
}

static TQT_DBusData::Type typeFromEntry(const TQDomElement& element)
{
    for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
    {
        if (!child.isElement()) continue;

        TQDomElement childElement = child.toElement();

        if (childElement.tagName() == "byte")       return TQT_DBusData::Byte;
        if (childElement.tagName() == "int16")      return TQT_DBusData::Int16;
        if (childElement.tagName() == "uint16")     return TQT_DBusData::UInt16;
        if (childElement.tagName() == "int32")      return TQT_DBusData::Int32;
        if (childElement.tagName() == "uint32")     return TQT_DBusData::UInt32;
        if (childElement.tagName() == "int64")      return TQT_DBusData::Int64;
        if (childElement.tagName() == "uint64")     return TQT_DBusData::UInt64;
        if (childElement.tagName() == "string")     return TQT_DBusData::String;
        if (childElement.tagName() == "objectpath") return TQT_DBusData::ObjectPath;
    }

    return TQT_DBusData::Invalid;
}

static TQT_DBusData toByteKeyMap(const TQDomElement& element)
{
    TQT_DBusDataMap<TQ_UINT8> map;

    for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
    {
        if (!child.isElement()) continue;

        TQDomElement entryElement = child.toElement();
        if (entryElement.tagName() != "entry") continue;

        for (TQDomNode entryChild = entryElement.firstChild(); !entryChild.isNull();
             entryChild = entryChild.nextSibling())
        {
            if (!entryChild.isElement()) continue;

            TQDomElement childElement = entryChild.toElement();
            if (childElement.tagName() != "byte") continue;

            TQ_UINT8 key = toByte(childElement);

            TQT_DBusData data;
            for (entryChild = entryChild.nextSibling(); !entryChild.isNull();
                 entryChild = entryChild.nextSibling())
            {
                if (!entryChild.isElement()) continue;

                data = XMLMarshaller::toTQT_DBusData(entryChild.toElement());
                if (data.isValid()) break;
            }

            if (!data.isValid()) return TQT_DBusData();

            map.insert(key, data);
            break;
        }
    }

    return TQT_DBusData::fromByteKeyMap(map);
}

static TQT_DBusData toInt16KeyMap(const TQDomElement& element)
{
    TQT_DBusDataMap<TQ_INT16> map;

    for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
    {
        if (!child.isElement()) continue;

        TQDomElement entryElement = child.toElement();
        if (entryElement.tagName() != "entry") continue;

        for (TQDomNode entryChild = entryElement.firstChild(); !entryChild.isNull();
             entryChild = entryChild.nextSibling())
        {
            if (!entryChild.isElement()) continue;

            TQDomElement childElement = entryChild.toElement();
            if (childElement.tagName() != "int16") continue;

            TQ_INT16 key = toInt16(childElement);

            TQT_DBusData data;
            for (entryChild = entryChild.nextSibling(); !entryChild.isNull();
                 entryChild = entryChild.nextSibling())
            {
                if (!entryChild.isElement()) continue;

                data = XMLMarshaller::toTQT_DBusData(entryChild.toElement());
                if (data.isValid()) break;
            }

            if (!data.isValid()) return TQT_DBusData();

            map.insert(key, data);
            break;
        }
    }

    return TQT_DBusData::fromInt16KeyMap(map);
}

static TQT_DBusData toUInt16KeyMap(const TQDomElement& element)
{
    TQT_DBusDataMap<TQ_UINT16> map;

    for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
    {
        if (!child.isElement()) continue;

        TQDomElement entryElement = child.toElement();
        if (entryElement.tagName() != "entry") continue;

        for (TQDomNode entryChild = entryElement.firstChild(); !entryChild.isNull();
             entryChild = entryChild.nextSibling())
        {
            if (!entryChild.isElement()) continue;

            TQDomElement childElement = entryChild.toElement();
            if (childElement.tagName() != "uint16") continue;

            TQ_UINT16 key = toUInt16(childElement);

            TQT_DBusData data;
            for (entryChild = entryChild.nextSibling(); !entryChild.isNull();
                 entryChild = entryChild.nextSibling())
            {
                if (!entryChild.isElement()) continue;

                data = XMLMarshaller::toTQT_DBusData(entryChild.toElement());
                if (data.isValid()) break;
            }

            if (!data.isValid()) return TQT_DBusData();

            map.insert(key, data);
            break;
        }
    }

    return TQT_DBusData::fromUInt16KeyMap(map);
}

static TQT_DBusData toInt32KeyMap(const TQDomElement& element)
{
    TQT_DBusDataMap<TQ_INT32> map;

    for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
    {
        if (!child.isElement()) continue;

        TQDomElement entryElement = child.toElement();
        if (entryElement.tagName() != "entry") continue;

        for (TQDomNode entryChild = entryElement.firstChild(); !entryChild.isNull();
             entryChild = entryChild.nextSibling())
        {
            if (!entryChild.isElement()) continue;

            TQDomElement childElement = entryChild.toElement();
            if (childElement.tagName() != "int32") continue;

            TQ_INT32 key = toInt32(childElement);

            TQT_DBusData data;
            for (entryChild = entryChild.nextSibling(); !entryChild.isNull();
                 entryChild = entryChild.nextSibling())
            {
                if (!entryChild.isElement()) continue;

                data = XMLMarshaller::toTQT_DBusData(entryChild.toElement());
                if (data.isValid()) break;
            }

            if (!data.isValid()) return TQT_DBusData();

            map.insert(key, data);
            break;
        }
    }

    return TQT_DBusData::fromInt32KeyMap(map);
}

static TQT_DBusData toUInt32KeyMap(const TQDomElement& element)
{
    TQT_DBusDataMap<TQ_UINT32> map;

    for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
    {
        if (!child.isElement()) continue;

        TQDomElement entryElement = child.toElement();
        if (entryElement.tagName() != "entry") continue;

        for (TQDomNode entryChild = entryElement.firstChild(); !entryChild.isNull();
             entryChild = entryChild.nextSibling())
        {
            if (!entryChild.isElement()) continue;

            TQDomElement childElement = entryChild.toElement();
            if (childElement.tagName() != "uint32") continue;

            TQ_UINT32 key = toUInt32(childElement);

            TQT_DBusData data;
            for (entryChild = entryChild.nextSibling(); !entryChild.isNull();
                 entryChild = entryChild.nextSibling())
            {
                if (!entryChild.isElement()) continue;

                data = XMLMarshaller::toTQT_DBusData(entryChild.toElement());
                if (data.isValid()) break;
            }

            if (!data.isValid()) return TQT_DBusData();

            map.insert(key, data);
            break;
        }
    }

    return TQT_DBusData::fromUInt32KeyMap(map);
}

static TQT_DBusData toInt64KeyMap(const TQDomElement& element)
{
    TQT_DBusDataMap<TQ_INT64> map;

    for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
    {
        if (!child.isElement()) continue;

        TQDomElement entryElement = child.toElement();
        if (entryElement.tagName() != "entry") continue;

        for (TQDomNode entryChild = entryElement.firstChild(); !entryChild.isNull();
             entryChild = entryChild.nextSibling())
        {
            if (!entryChild.isElement()) continue;

            TQDomElement childElement = entryChild.toElement();
            if (childElement.tagName() != "int64") continue;

            TQ_INT64 key = toInt64(childElement);

            TQT_DBusData data;
            for (entryChild = entryChild.nextSibling(); !entryChild.isNull();
                 entryChild = entryChild.nextSibling())
            {
                if (!entryChild.isElement()) continue;

                data = XMLMarshaller::toTQT_DBusData(entryChild.toElement());
                if (data.isValid()) break;
            }

            if (!data.isValid()) return TQT_DBusData();

            map.insert(key, data);
            break;
        }
    }

    return TQT_DBusData::fromInt64KeyMap(map);
}

static TQT_DBusData toUInt64KeyMap(const TQDomElement& element)
{
    TQT_DBusDataMap<TQ_UINT64> map;

    for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
    {
        if (!child.isElement()) continue;

        TQDomElement entryElement = child.toElement();
        if (entryElement.tagName() != "entry") continue;

        for (TQDomNode entryChild = entryElement.firstChild(); !entryChild.isNull();
             entryChild = entryChild.nextSibling())
        {
            if (!entryChild.isElement()) continue;

            TQDomElement childElement = entryChild.toElement();
            if (childElement.tagName() != "uint64") continue;

            TQ_UINT64 key = toUInt64(childElement);

            TQT_DBusData data;
            for (entryChild = entryChild.nextSibling(); !entryChild.isNull();
                 entryChild = entryChild.nextSibling())
            {
                if (!entryChild.isElement()) continue;

                data = XMLMarshaller::toTQT_DBusData(entryChild.toElement());
                if (data.isValid()) break;
            }

            if (!data.isValid()) return TQT_DBusData();

            map.insert(key, data);
            break;
        }
    }

    return TQT_DBusData::fromUInt64KeyMap(map);
}

static TQT_DBusData toStringKeyMap(const TQDomElement& element)
{
    TQT_DBusDataMap<TQString> map;

    for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
    {
        if (!child.isElement()) continue;

        TQDomElement entryElement = child.toElement();
        if (entryElement.tagName() != "entry") continue;

        for (TQDomNode entryChild = entryElement.firstChild(); !entryChild.isNull();
             entryChild = entryChild.nextSibling())
        {
            if (!entryChild.isElement()) continue;

            TQDomElement childElement = entryChild.toElement();
            if (childElement.tagName() != "string") continue;

            TQString key = toString(childElement);

            TQT_DBusData data;
            for (entryChild = entryChild.nextSibling(); !entryChild.isNull();
                 entryChild = entryChild.nextSibling())
            {
                if (!entryChild.isElement()) continue;

                data = XMLMarshaller::toTQT_DBusData(entryChild.toElement());
                if (data.isValid()) break;
            }

            if (!data.isValid()) return TQT_DBusData();

            map.insert(key, data);
            break;
        }
    }

    return TQT_DBusData::fromStringKeyMap(map);
}

static TQT_DBusData toObjectPathKeyMap(const TQDomElement& element)
{
    TQT_DBusDataMap<TQT_DBusObjectPath> map;

    for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
    {
        if (!child.isElement()) continue;

        TQDomElement entryElement = child.toElement();
        if (entryElement.tagName() != "entry") continue;

        for (TQDomNode entryChild = entryElement.firstChild(); !entryChild.isNull();
             entryChild = entryChild.nextSibling())
        {
            if (!entryChild.isElement()) continue;

            TQDomElement childElement = entryChild.toElement();
            if (childElement.tagName() != "objectpath") continue;

            TQT_DBusObjectPath key = toObjectPath(childElement);

            TQT_DBusData data;
            for (entryChild = entryChild.nextSibling(); !entryChild.isNull();
                 entryChild = entryChild.nextSibling())
            {
                if (!entryChild.isElement()) continue;

                data = XMLMarshaller::toTQT_DBusData(entryChild.toElement());
                if (data.isValid()) break;
            }

            if (!data.isValid()) return TQT_DBusData();

            map.insert(key, data);
            break;
        }
    }

    return TQT_DBusData::fromObjectPathKeyMap(map);
}

TQDomElement XMLMarshaller::fromTQT_DBusData(const TQT_DBusData& data, const TQDomDocument& ownerDoc)
{
    switch (data.type())
    {
        case TQT_DBusData::Invalid:
            kdWarning() << "XMLMarsaller::fromTQT_DBusData: data is invalid" << endl;
            return TQDomElement();

        case TQT_DBusData::Bool:
            return fromBool(data.toBool(), ownerDoc);

        case TQT_DBusData::Byte:
            return fromByte(data.toByte(), ownerDoc);

        case TQT_DBusData::Int16:
            return fromInt16(data.toInt16(), ownerDoc);

        case TQT_DBusData::UInt16:
            return fromUInt16(data.toUInt16(), ownerDoc);

        case TQT_DBusData::Int32:
            return fromInt32(data.toInt32(), ownerDoc);

        case TQT_DBusData::UInt32:
            return fromUInt32(data.toUInt32(), ownerDoc);

        case TQT_DBusData::Int64:
            return fromInt64(data.toInt64(), ownerDoc);

        case TQT_DBusData::UInt64:
            return fromUInt64(data.toUInt64(), ownerDoc);

        case TQT_DBusData::Double:
            return fromDouble(data.toDouble(), ownerDoc);

        case TQT_DBusData::String:
            return fromString(data.toString(), ownerDoc);

        case TQT_DBusData::ObjectPath:
            return fromObjectPath(data.toObjectPath(), ownerDoc);

        case TQT_DBusData::List:
        {
            TQDomDocument doc    = ownerDoc;
            TQDomElement element = doc.createElement("list");
            TQT_DBusDataList list  = data.toList();
            if (list.isEmpty())
            {
                element.setAttribute("signature", data.buildDBusSignature().data());
            }
            else
            {
                TQValueList<TQT_DBusData> items = list.toTQValueList();
                TQValueList<TQT_DBusData>::const_iterator it    = items.begin();
                TQValueList<TQT_DBusData>::const_iterator endIt = items.end();
                for (; it != endIt; ++it)
                {
                    TQDomElement itemElement = fromTQT_DBusData(*it, ownerDoc);

                    if (!itemElement.isNull()) element.appendChild(itemElement);
                }
            }

            return element;
        }

        case TQT_DBusData::Struct:
        {
            TQDomDocument doc    = ownerDoc;
            TQDomElement element = doc.createElement("struct");

            TQValueList<TQT_DBusData> members = data.toStruct();
            TQValueList<TQT_DBusData>::const_iterator it    = members.begin();
            TQValueList<TQT_DBusData>::const_iterator endIt = members.end();
            for (; it != endIt; ++it)
            {
                TQDomElement memberElement = fromTQT_DBusData(*it, ownerDoc);

                if (!memberElement.isNull()) element.appendChild(memberElement);
            }
            return element;
        }

        case TQT_DBusData::Variant:
        {
            TQDomDocument doc    = ownerDoc;
            TQDomElement element = doc.createElement("variant");

            TQT_DBusVariant variant = data.toVariant();

            element.setAttribute("signature", variant.signature);

            TQDomElement dataElement = fromTQT_DBusData(variant.value, ownerDoc);
            if (!dataElement.isNull()) element.appendChild(dataElement);

            return element;
        }

        case TQT_DBusData::Map:
        {
            TQDomElement mapElement;

            switch (data.keyType())
            {
                case TQT_DBusData::Byte:
                    mapElement = fromByteKeyMap(data.toByteKeyMap(), ownerDoc);
                    break;

                case TQT_DBusData::Int16:
                    mapElement = fromInt16KeyMap(data.toInt16KeyMap(), ownerDoc);
                    break;

                case TQT_DBusData::UInt16:
                    mapElement = fromUInt16KeyMap(data.toUInt16KeyMap(), ownerDoc);
                    break;

                case TQT_DBusData::Int32:
                    mapElement = fromInt32KeyMap(data.toInt32KeyMap(), ownerDoc);
                    break;

                case TQT_DBusData::UInt32:
                    mapElement = fromUInt32KeyMap(data.toUInt32KeyMap(), ownerDoc);
                    break;

                case TQT_DBusData::Int64:
                    mapElement = fromInt64KeyMap(data.toInt64KeyMap(), ownerDoc);
                    break;

                case TQT_DBusData::UInt64:
                    mapElement = fromUInt64KeyMap(data.toUInt64KeyMap(), ownerDoc);
                    break;

                case TQT_DBusData::String:
                    mapElement = fromStringKeyMap(data.toStringKeyMap(), ownerDoc);
                    break;

                case TQT_DBusData::ObjectPath:
                    mapElement = fromObjectPathKeyMap(data.toObjectPathKeyMap(), ownerDoc);
                    break;

                default:
                    return TQDomElement();
            }

            if (!mapElement.hasChildNodes())
            {
                mapElement.setAttribute("signature", data.buildDBusSignature().data());
            }

            return mapElement;
        }
    }

    return TQDomElement();
}

TQT_DBusData XMLMarshaller::toTQT_DBusData(const TQDomElement& element)
{
    if (element.isNull()) return TQT_DBusData();

    if (element.tagName() == "bool")       return TQT_DBusData::fromBool(toBool(element));
    if (element.tagName() == "byte")       return TQT_DBusData::fromByte(toByte(element));
    if (element.tagName() == "int16")      return TQT_DBusData::fromInt16(toInt16(element));
    if (element.tagName() == "uint16")     return TQT_DBusData::fromUInt16(toUInt16(element));
    if (element.tagName() == "int32")      return TQT_DBusData::fromInt32(toInt32(element));
    if (element.tagName() == "uint32")     return TQT_DBusData::fromUInt32(toUInt32(element));
    if (element.tagName() == "int64")      return TQT_DBusData::fromInt64(toInt64(element));
    if (element.tagName() == "uint64")     return TQT_DBusData::fromUInt64(toUInt64(element));
    if (element.tagName() == "double")     return TQT_DBusData::fromDouble(toDouble(element));
    if (element.tagName() == "string")     return TQT_DBusData::fromString(toString(element));
    if (element.tagName() == "objectpath") return TQT_DBusData::fromObjectPath(toObjectPath(element));

    if (element.tagName() == "list")
    {
        if (element.hasChildNodes())
        {
            TQValueList<TQT_DBusData> list;

            for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
            {
                if (!child.isElement()) continue;

                TQT_DBusData itemData = toTQT_DBusData(child.toElement());
                if (itemData.isValid()) list << itemData;
            }

            return TQT_DBusData::fromList(list);
        }

        // TODO handle empty list with signatures as hint
    }

    if (element.tagName() == "struct")
    {
        TQValueList<TQT_DBusData> members;

        for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
        {
            if (!child.isElement()) continue;

            TQT_DBusData memberData = toTQT_DBusData(child.toElement());
            if (memberData.isValid()) members << memberData;
        }

        return TQT_DBusData::fromStruct(members);
    }

    if (element.tagName() == "variant")
    {
        TQT_DBusData data;
        for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
        {
            if (!child.isElement()) continue;

            TQT_DBusData childData = toTQT_DBusData(child.toElement());
            if (childData.isValid())
            {
                data = childData;
                break;
            }
        }

        if (!data.isValid()) return TQT_DBusData();

        TQT_DBusVariant variant;
        variant.signature = element.attribute("signature");
        variant.value     = data;

        return TQT_DBusData::fromVariant(variant);
    }

    if (element.tagName() == "map")
    {
        TQDomElement entryElement;
        for (TQDomNode child = element.firstChild(); !child.isNull(); child = child.nextSibling())
        {
            if (!child.isElement()) continue;

            TQDomElement childElement = child.toElement();
            if (childElement.tagName() == "entry")
            {
                entryElement = childElement;
                break;
            }
        }

        if (entryElement.isNull())
        {
            // TODO: empty map
            return TQT_DBusData();
        }

        switch (typeFromEntry(entryElement))
        {
            case TQT_DBusData::Byte:
                return toByteKeyMap(element);

            case TQT_DBusData::Int16:
                return toInt16KeyMap(element);

            case TQT_DBusData::UInt16:
                return toUInt16KeyMap(element);

            case TQT_DBusData::Int32:
                return toInt32KeyMap(element);

            case TQT_DBusData::UInt32:
                return toUInt32KeyMap(element);

            case TQT_DBusData::Int64:
                return toInt64KeyMap(element);

            case TQT_DBusData::UInt64:
                return toUInt64KeyMap(element);

            case TQT_DBusData::String:
                return toStringKeyMap(element);

            case TQT_DBusData::ObjectPath:
                return toObjectPathKeyMap(element);

            default:
                return TQT_DBusData();
        }
    }

    return TQT_DBusData();
}

TQString XMLMarshaller::fromTQT_DBusData(const TQT_DBusData& data)
{
    TQDomDocument doc;

    TQDomElement element = fromTQT_DBusData(data, doc);
    if (element.isNull()) return TQString();

    doc.appendChild(element);
    return doc.toString();
}

TQT_DBusData XMLMarshaller::toTQT_DBusData(const TQString& xmlString)
{
    TQDomDocument doc;

    if (!doc.setContent(xmlString)) return TQT_DBusData();

    return toTQT_DBusData(doc.documentElement());
}

