/***************************************************************************
*   Copyright (C) 2003 by Unai Garro (ugarro@users.sourceforge.net)       *
*                                                                         *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/
#include "paneldeco.h"

#include <ntqpainter.h>
#include <ntqpoint.h>
#include <ntqrect.h>

#include <kiconloader.h>
#include <kpixmap.h>
#include <kpixmapeffect.h>


// Panel decoration

PanelDeco::PanelDeco( TQWidget *parent, const char *name, const TQString &title, const TQString &iconName ) : TQVBox( parent, name )
{

	// Top decoration
	tDeco = new TopDeco( this, "TopDecoration", title, iconName );

	hbox = new TQHBox( this );

	//Left decoration
	lDeco = new LeftDeco( hbox, "LeftDecoration" );

	//The widget stack (panels)
	stack = new TQWidgetStack( hbox );
	stack->setSizePolicy( TQSizePolicy( TQSizePolicy::MinimumExpanding, TQSizePolicy::MinimumExpanding ) );

}


PanelDeco::~PanelDeco()
{}

void PanelDeco::childEvent( TQChildEvent *e )
{
	if ( e->type() == TQEvent::ChildInserted ) {
		TQObject * obj = e->child();
		if ( obj->inherits( "TQWidget" ) ) {
			TQWidget * w = ( TQWidget* ) obj;
			if ( w != hbox && w != tDeco )
				w->reparent( stack, TQPoint( 0, 0 ) );
		}
	}
}


int PanelDeco::id( TQWidget* w )
{
	return ( stack->id( w ) );
}

void PanelDeco::raise( TQWidget *w )
{
	TQWidget * old_w = visiblePanel();

	stack->raiseWidget( w );

	if ( old_w != w )
		emit panelRaised( w, old_w );
}

TQWidget* PanelDeco::visiblePanel( void )
{
	return ( stack->visibleWidget() );
}

void PanelDeco::setHeader( const TQString &title, const TQString &icon )
{
	tDeco->setHeader( title, icon );
}

// Left part of the decoration

LeftDeco::LeftDeco( TQWidget *parent, const char *name ) :
#if TQT_VERSION >= 0x030200
		TQWidget( parent, name, TQt::WNoAutoErase )
#else
		TQWidget( parent, name )
#endif
{}

LeftDeco::~LeftDeco()
{}

// Top part of the decoration

TopDeco::TopDeco( TQWidget *parent, const char *name, const TQString &title, const TQString &iconName ) :
#if TQT_VERSION >= 0x030200
		TQWidget( parent, name, TQt::WNoAutoErase )
#else
		TQWidget( parent, name )
#endif
{
	setMinimumHeight( 30 );
	icon = 0;
	panelTitle = TQString::null;
	if ( !iconName.isNull() ) {
		TDEIconLoader il;
		icon = new TQPixmap( il.loadIcon( iconName, TDEIcon::NoGroup, 22 ) );
	}

	if ( !title.isNull() ) {
		panelTitle = title;
	}
}

TopDeco::~TopDeco()
{
	delete icon;
}


void TopDeco::paintEvent( TQPaintEvent * )
{
	// Get gradient colors
	TQColor c1 = colorGroup().button().light( 120 );
	TQColor c2 = paletteBackgroundColor();

	// Draw the gradient
	KPixmap kpm;
	kpm.resize( size() );
	KPixmapEffect::unbalancedGradient ( kpm, c1, c2, KPixmapEffect::VerticalGradient, 150, 150 );

	// Add a line on top
	TQPainter painter( &kpm );
	painter.setPen( colorGroup().button().dark( 130 ) );
	painter.drawLine( 0, 0, width(), 0 );

	// Now Add the icon
	int xpos = 0, ypos = 0;
	if ( icon ) {
		xpos = 20;
		ypos = ( height() - icon->height() ) / 2 - 1;
		painter.drawPixmap( xpos, ypos, *icon );
		xpos += icon->width(); // Move it so that later we can easily place the text
	}

	// Finally, draw the text besides the icon
	if ( !panelTitle.isNull() ) {
		xpos += 15;
		TQRect r = rect();
		r.setLeft( xpos );
		painter.setPen( TQColor( 0x00, 0x00, 0x00 ) );
		TQFont ft = font();
		ft.setBold( true );
		painter.setFont( ft );
		painter.drawText( r, TQt::AlignVCenter, panelTitle );
	}
	painter.end();
	// Copy the pixmap to the widget
	bitBlt( this, 0, 0, &kpm );
}

void TopDeco::setHeader( const TQString &title, const TQString &iconName )
{
	if ( !title.isNull() )
		panelTitle = title;
	if ( !iconName.isNull() ) {
		TDEIconLoader il;
		icon = new TQPixmap( il.loadIcon( iconName, TDEIcon::NoGroup, 22 ) );
	}
	if ( !title.isNull() || !iconName.isNull() )
		update();
}

TQSize TopDeco::sizeHint( void )
{
	return ( TQSize( parentWidget() ->width(), 30 ) );
}

#include "paneldeco.moc"
