#ifndef _DOWNLOADER_H
#define _DOWNLOADER_H

#include <kurl.h>
#include <tqobject.h>
#include <tqptrlist.h>

class TQFile;
class TQTimer;
class Downloader;


namespace TDEIO
{
	class TransferJob;
	class Job;
}

/**
 * Item to download, usually queued up in Downloader
 **/
class DownloadItem
{
	friend class Downloader;
public:
	DownloadItem();
	virtual ~DownloadItem();

	bool isDownloaded() const;

	/**
	 * @return the local filename this item will be saved to
	 **/
	TQString localFilename() const;

	virtual void setLocalFilename(const TQString &filename);

	/**
	 * Called if this item has been fully downloaded
	 **/
	virtual void downloadFinished();
	/**
	 * Called at regular intervals while downloading this item
	 **/
	virtual void downloaded(int percent);
	/**
	 * Called when downloading this item timed out
	 **/
	virtual void downloadTimeout();
	/**
	 * @return true if the download was scheduled, false if the file is local
	 **/
	bool enqueue(const KURL &url);
	/**
	 * Remove this item from Downloader queue
	 **/
	void dequeue();

private:
	TQString mLocalFilename;
};

/**
 * download playlistitems, in a queue based fasion
 **/
class Downloader : public TQObject
{
Q_OBJECT
  
	struct QueueItem
	{
		DownloadItem *notifier;
		KURL file;
		TQString local;
	};

public:
	Downloader(TQObject *parent=0);
	virtual ~Downloader();

public slots:
	TQString enqueue(DownloadItem *notifier, const KURL &file);
	void dequeue(DownloadItem *notifier);

	/**
	 * @internal
	 **/
	void start();

signals:
	/**
	 * Emitted when a new DownloadItem was added to the queue
	 * @p notifier is the added item
	 **/
	void enqueued(DownloadItem *notifier, const KURL &file);
	/**
	 * Emitted when a DownloadItem was removed from the queue
	 * @p notifier is the removed item
	 **/
	void dequeued(DownloadItem *notifier);

private slots:
	void getNext();

	void data( TDEIO::Job *, const TQByteArray &data);
	void percent( TDEIO::Job *, unsigned long percent);
	void jobDone( TDEIO::Job *);
	void giveUpWithThisDownloadServerIsRunningNT();

private:
	TQPtrList<Downloader::QueueItem> mQueue;
	TQPtrList<Downloader::QueueItem> *mUnstartedQueue;
	TQFile *localfile;
	Downloader::QueueItem *current;
	TDEIO::TransferJob *mJob;
	TQTimer *mTimeout;
	bool mStarted;
};

#endif

