/*
    This file is part of KAddressBook.
    Copyright (c) 2002 Mike Pilone <mpilone@slac.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

    As a special exception, permission is given to link this program
    with any edition of TQt, and distribute the resulting executable,
    without including the source code for TQt in the source distribution.
*/

#ifndef VIEWMANAGER_H
#define VIEWMANAGER_H

#include <tqdict.h>
#include <tqwidget.h>
#include <kaddressbookview.h>

class TQDropEvent;
class TQWidgetStack;

class TDEAction;
class TDESelectAction;

class FilterSelectionWidget;

namespace KAB { class Core; }
namespace TDEABC { class AddressBook; }

/**
  The view manager manages the views and everything related to them. The
  manager will load the views at startup and display a view when told to
  make one active.

  The view manager will also create and manage all dialogs directly related to
  views (ie: AddView, ConfigureView, DeleteView, etc).
 */
class ViewManager : public TQWidget
{
  Q_OBJECT
  

  public:
    ViewManager( KAB::Core *core, TQWidget *parent, const char *name = 0 );
    ~ViewManager();

    void restoreSettings();
    void saveSettings();

    void unloadViews();

    TQStringList selectedUids() const;
    TQStringList selectedEmails() const;
    TDEABC::Addressee::List selectedAddressees() const;

    void setFilterSelectionWidget( FilterSelectionWidget *wdg );

    TDEABC::Field *currentSortField() const;
    TDEABC::Field::List viewFields() const;

  public slots:
    void setSelected( const TQString &uid = TQString(), bool selected = true );
    void setFirstSelected( bool selected = true );

    void refreshView( const TQString &uid = TQString() );
    void editView();
    void deleteView();
    void addView();

    void scrollUp();
    void scrollDown();

  protected slots:
    /**
      Called whenever the user drops something in the active view.
      This method will try to decode what was dropped, and if it was
      a valid addressee, add it to the addressbook.
     */
    void dropped( TQDropEvent* );

    /**
      Called whenever the user attempts to start a drag in the view.
      This method will convert all the selected addressees into text (vcard)
      and create a drag object.
     */
    void startDrag();

  signals:
    /**
      Emitted whenever the user selects an entry in the view.
     */
    void selected( const TQString &uid );

    /**
      Emitted whenever the user activates an entry in the view.
     */
    void executed( const TQString &uid );

    /**
      Emitted whenever the address book is modified in some way.
     */
    void modified();

    /**
      Emitted whenever a url is dragged on a view.
     */
    void urlDropped( const KURL& );

    /**
      Emitted whenever the sort field of a view has changed.
     */
    void sortFieldChanged();

    /**
      Emitted whenever the view fields changed.
     */
    void viewFieldsChanged();

  private slots:
    void setActiveView( const TQString &name );
    void setActiveFilter( int index );
    void configureFilters();

  private:
    void createViewFactories();
    TQStringList filterNames() const;
    int filterPosition( const TQString &name ) const;
    TQStringList viewNames() const;
    int viewPosition( const TQString &name ) const;
    void initActions();
    void initGUI();

    KAB::Core *mCore;

    Filter mCurrentFilter;
    Filter::List mFilterList;

    TQDict<KAddressBookView> mViewDict;
    TQDict<ViewFactory> mViewFactoryDict;
    TQStringList mViewNameList;

    TQWidgetStack *mViewWidgetStack;
    KAddressBookView *mActiveView;

    TDEAction *mActionDeleteView;
    TDESelectAction *mActionSelectView;

    FilterSelectionWidget *mFilterSelectionWidget;
};

#endif
