#!/usr/bin/env python

# TQt tutorial 14.

import sys
import math
import random
from python_tqt import qt


class LCDRange(qt.TQWidget):
    def __init__(self, s=None, parent=None, name=None):
        qt.TQWidget.__init__(self, parent, name)

        lcd = qt.TQLCDNumber(2, self, "lcd")
        self.slider = qt.TQSlider(qt.TQt.Horizontal, self, "slider")
        self.slider.setRange(0, 99)
        self.slider.setValue(0)

        self.label = qt.TQLabel(" ", self, "label")
        self.label.setAlignment(qt.TQt.AlignCenter)

        self.connect(self.slider, qt.SIGNAL("valueChanged(int)"), lcd, qt.SLOT("display(int)"))
        self.connect(self.slider, qt.SIGNAL("valueChanged(int)"), self, qt.PYSIGNAL("valueChanged(int)"))

        self.setFocusProxy(self.slider)

        l = qt.TQVBoxLayout(self)
        l.addWidget(lcd, 1)
        l.addWidget(self.slider)
        l.addWidget(self.label)

        if s is not None:
            self.setText(s)

    def value(self):
        return self.slider.value()

    def setValue(self, value):
        self.slider.setValue(value)

    def setRange(self, minVal, maxVal):
        if minVal < 0 or maxVal > 99 or minVal > maxVal:
            raise ValueError("LCDRange.setRange(): invalid range")

        self.slider.setRange(minVal, maxVal)

    def text(self):
        return self.label.text()

    def setText(self, s):
        self.label.setText(s)


class CannonField(qt.TQWidget):
    def __init__(self, parent=None, name=None):
        qt.TQWidget.__init__(self, parent, name)

        self.ang = 45
        self.f = 0
        self.timerCount = 0

        self.autoShootTimer = qt.TQTimer(self, "movement handler")
        self.connect(self.autoShootTimer, qt.SIGNAL("timeout()"), self.moveShot)

        self.shoot_ang = 0
        self.shoot_f = 0
        self.target = qt.TQPoint(0, 0)
        self.gameEnded = 0
        self.barrelPressed = 0

        self.setPalette(qt.TQPalette(qt.TQColor(250, 250, 200)))

        self.barrelRect = qt.TQRect(33, -4, 15, 8)

        self.newTarget()

    def angle(self):
        return self.ang

    def setAngle(self, degrees):
        if degrees < 5:
            degrees = 5
        if degrees > 70:
            degrees = 70
        if self.ang == degrees:
            return
        self.ang = degrees
        self.repaint(self.cannonRect(), 0)
        self.emit(qt.PYSIGNAL("angleChanged(int)"), (self.ang, ))

    def force(self):
        return self.f

    def setForce(self, newton):
        if newton < 0:
            newton = 0
        if self.f == newton:
            return
        self.f = newton
        self.emit(qt.PYSIGNAL("forceChanged(int)"), (self.f, ))

    def shoot(self):
        if self.isShooting():
            return

        self.timerCount = 0
        self.shoot_ang = self.ang
        self.shoot_f = self.f
        self.autoShootTimer.start(50)
        self.emit(qt.PYSIGNAL("canShoot(bool)"), (0, ))

    def newTarget(self):
        r = qt.TQRegion(self.targetRect())
        self.target = qt.TQPoint(random.randint(200, 390), random.randint(10, 265))
        self.repaint(r.unite(qt.TQRegion(self.targetRect())))

    def gameOver(self):
        return self.gameEnded

    def setGameOver(self):
        if self.gameEnded:
            return
        if self.isShooting():
            self.autoShootTime.stop()
        self.gameEnded = 1
        self.repaint()

    def restartGame(self):
        if self.isShooting():
            self.autoShootTime.stop()
        self.gameEnded = 0
        self.repaint()
        self.emit(qt.PYSIGNAL("canShoot(bool)"), (1, ))

    def moveShot(self):
        r = qt.TQRegion(self.shotRect())
        self.timerCount = self.timerCount + 1

        shotR = self.shotRect()

        if shotR.intersects(self.targetRect()):
            self.autoShootTimer.stop()
            self.emit(qt.PYSIGNAL("hit()"), ())
            self.emit(qt.PYSIGNAL("canShoot(bool)"), (1, ))
        elif shotR.x() > self.width() or shotR.y() > self.height() or shotR.intersects(self.barrierRect()):
            self.autoShootTimer.stop()
            self.emit(qt.PYSIGNAL("missed()"), ())
            self.emit(qt.PYSIGNAL("canShoot(bool)"), (1, ))
        else:
            r = r.unite(qt.TQRegion(shotR))

        self.repaint(r)

    def mousePressEvent(self, ev):
        if ev.button() != qt.TQt.LeftButton:
            return
        if self.barrelHit(ev.pos()):
            self.barrelPressed = 1

    def mouseMoveEvent(self, ev):
        if not self.barrelPressed:
            return
        pnt = ev.pos()
        if pnt.x() <= 0:
            pnt.setX(1)
        if pnt.y() >= self.height():
            pnt.setY(self.height() - 1)
        rad = math.atan(float(self.rect().bottom() - pnt.y()) / pnt.x())
        self.setAngle(int(round(rad * 180 / math.pi)))

    def mouseReleaseEvent(self, ev):
        if ev.button() == qt.TQt.LeftButton:
            self.barrelPressed = 0

    def paintEvent(self, ev):
        updateR = ev.rect()
        p = qt.TQPainter(self)

        if self.gameEnded:
            p.setPen(qt.TQt.black)
            p.setFont(qt.TQFont("Courier", 48, qt.TQFont.Bold))
            p.drawText(self.rect(), qt.TQt.AlignCenter, "Game Over")

        if updateR.intersects(self.cannonRect()):
            self.paintCannon(p)

        if updateR.intersects(self.barrierRect()):
            self.paintBarrier(p)

        if self.isShooting() and updateR.intersects(self.shotRect()):
            self.paintShot(p)

        if not self.gameEnded and updateR.intersects(self.targetRect()):
            self.paintTarget(p)

    def paintShot(self, p):
        p.setBrush(qt.TQt.black)
        p.setPen(qt.TQt.NoPen)
        p.drawRect(self.shotRect())

    def paintTarget(self, p):
        p.setBrush(qt.TQt.red)
        p.setPen(qt.TQt.black)
        p.drawRect(self.targetRect())

    def paintBarrier(self, p):
        p.setBrush(qt.TQt.yellow)
        p.setPen(qt.TQt.black)
        p.drawRect(self.barrierRect())

    def paintCannon(self, p):
        cr = self.cannonRect()
        pix = qt.TQPixmap(cr.size())
        pix.fill(self, cr.topLeft())

        tmp = qt.TQPainter(pix)
        tmp.setBrush(qt.TQt.blue)
        tmp.setPen(qt.TQt.NoPen)

        tmp.translate(0, pix.height() - 1)
        tmp.drawPie(qt.TQRect(-35, -35, 70, 70), 0, 90 * 16)
        tmp.rotate(-self.ang)
        tmp.drawRect(self.barrelRect)
        tmp.end()

        p.drawPixmap(cr.topLeft(), pix)

    def cannonRect(self):
        r = qt.TQRect(0, 0, 50, 50)
        r.moveBottomLeft(self.rect().bottomLeft())
        return r

    def shotRect(self):
        gravity = 4.0

        time = self.timerCount / 4.0
        velocity = self.shoot_f
        radians = self.shoot_ang * math.pi / 180

        velx = velocity * math.cos(radians)
        vely = velocity * math.sin(radians)
        x0 = (self.barrelRect.right() + 5) * math.cos(radians)
        y0 = (self.barrelRect.right() + 5) * math.sin(radians)
        x = x0 + velx * time
        y = y0 + vely * time - 0.5 * gravity * time * time

        r = qt.TQRect(0, 0, 6, 6)
        r.moveCenter(qt.TQPoint(int(x), int(self.height() - 1 - y)))
        return r

    def targetRect(self):
        r = qt.TQRect(0, 0, 20, 10)
        r.moveCenter(qt.TQPoint(self.target.x(), self.height() - 1 - self.target.y()))
        return r

    def barrierRect(self):
        return qt.TQRect(145, self.height() - 100, 15, 100)

    def barrelHit(self, p):
        mtx = qt.TQWMatrix()
        mtx.translate(0, self.height() - 1)
        mtx.rotate(-self.ang)
        (mtx,  invertable) = mtx.invert()
        return self.barrelRect.contains(mtx.map(p))

    def isShooting(self):
        return self.autoShootTimer.isActive()

    def sizePolicy(self):
        return qt.TQSizePolicy(qt.TQSizePolicy.Expanding, qt.TQSizePolicy.Expanding)


class GameBoard(qt.TQWidget):
    def __init__(self, parent=None, name=None):
        qt.TQWidget.__init__(self, parent, name)

        quit = qt.TQPushButton("&Quit", self, "quit")
        quit.setFont(qt.TQFont("Times", 18, qt.TQFont.Bold))
        self.connect(quit, qt.SIGNAL("clicked()"), qt.tqApp, qt.SLOT("quit()"))

        self.angle = LCDRange("ANGLE", self, "angle")
        self.angle.setRange(5, 70)

        self.force = LCDRange("FORCE", self, "force")
        self.force.setRange(10, 50)

        box = qt.TQVBox(self, "cannonFrame")
        box.setFrameStyle(qt.TQFrame.WinPanel | qt.TQFrame.Sunken)

        self.cannonField = CannonField(box, "cannonField")

        self.connect(self.angle, qt.PYSIGNAL("valueChanged(int)"), self.cannonField.setAngle)
        self.connect(self.cannonField, qt.PYSIGNAL("angleChanged(int)"), self.angle.setValue)

        self.connect(self.force, qt.PYSIGNAL("valueChanged(int)"), self.cannonField.setForce)
        self.connect(self.cannonField, qt.PYSIGNAL("forceChanged(int)"), self.force.setValue)

        self.connect(self.cannonField, qt.PYSIGNAL("hit()"), self.hit)
        self.connect(self.cannonField, qt.PYSIGNAL("missed()"), self.missed)

        self.shoot = qt.TQPushButton("&Shoot", self, "shoot")
        self.shoot.setFont(qt.TQFont("Times", 18, qt.TQFont.Bold))
        self.connect(self.shoot, qt.SIGNAL("clicked()"), self.fire)
        self.connect(self.cannonField, qt.PYSIGNAL("canShoot(bool)"), self.shoot, qt.SLOT("setEnabled(bool)"))

        restart = qt.TQPushButton("&New Game", self, "newgame")
        restart.setFont(qt.TQFont("Times", 18, qt.TQFont.Bold))
        self.connect(restart, qt.SIGNAL("clicked()"), self.newGame)

        self.hits = qt.TQLCDNumber(2, self, "hits")
        self.shotsLeft = qt.TQLCDNumber(2, self, "shotsleft")
        hitsL = qt.TQLabel("HITS", self, "hitsLabel")
        shotsLeftL = qt.TQLabel("SHOTS LEFT", self, "shotsleftLabel")

        accel = qt.TQAccel(self)
        accel.connectItem(accel.insertItem(qt.TQt.Key_Enter), self.fire)
        accel.connectItem(accel.insertItem(qt.TQt.Key_Return), self.fire)
        accel.connectItem(accel.insertItem(qt.TQt.CTRL + qt.TQt.Key_Q), qt.tqApp, qt.SLOT("quit()"))

        grid = qt.TQGridLayout(self, 2, 2, 10)
        grid.addWidget(quit, 0, 0)
        grid.addWidget(box, 1, 1)
        grid.setColStretch(1, 10)

        leftBox = qt.TQVBoxLayout()
        grid.addLayout(leftBox, 1, 0)
        leftBox.addWidget(self.angle)
        leftBox.addWidget(self.force)

        topBox = qt.TQHBoxLayout()
        grid.addLayout(topBox, 0, 1)
        topBox.addWidget(self.shoot)
        topBox.addWidget(self.hits)
        topBox.addWidget(hitsL)
        topBox.addWidget(self.shotsLeft)
        topBox.addWidget(shotsLeftL)
        topBox.addStretch(1)
        topBox.addWidget(restart)

        self.angle.setValue(60)
        self.force.setValue(25)
        self.angle.setFocus()

        self.newGame()

    def fire(self):
        if self.cannonField.gameOver() or self.cannonField.isShooting():
            return
        self.shotsLeft.display(self.shotsLeft.intValue() - 1)
        self.cannonField.shoot()

    def hit(self):
        self.hits.display(self.hits.intValue() + 1)
        if self.shotsLeft.intValue() == 0:
            self.cannonField.setGameOver()
        else:
            self.cannonField.newTarget()

    def missed(self):
        if self.shotsLeft.intValue() == 0:
            self.cannonField.setGameOver()

    def newGame(self):
        self.shotsLeft.display(15)
        self.hits.display(0)
        self.cannonField.restartGame()
        self.cannonField.newTarget()


qt.TQApplication.setColorSpec(qt.TQApplication.CustomColor)
a = qt.TQApplication(sys.argv)

gb = GameBoard()
gb.setGeometry(100, 100, 500, 355)
a.setMainWidget(gb)
gb.show()
sys.exit(a.exec_loop())
