/*
 *  Copyright (c) 2006 Boudewijn Rempt <boud@valdyas.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#include "kis_filter.h"

#include <kdebug.h>
#include <tqdom.h>
#include <tqstring.h>

#include "kis_filter_registry.h"
#include "kis_transaction.h"
#include "kis_undo_adapter.h"
#include "kis_painter.h"
#include "kis_selection.h"
#include "kis_id.h"
#include "kis_canvas_subject.h"
#include "kis_progress_display_interface.h"
#include "kis_types.h"
#include "kis_filter_config_widget.h"


KisFilterConfiguration::KisFilterConfiguration(const KisFilterConfiguration & rhs)
{
    m_name = rhs.m_name;
    m_version = rhs.m_version;
    m_properties = rhs.m_properties;
}

void KisFilterConfiguration::fromXML(const TQString & s )
{
    m_properties.clear();

    TQDomDocument doc;
    doc.setContent( s );
    TQDomElement e = doc.documentElement();
    TQDomNode n = e.firstChild();

    m_name = e.attribute("name");
    m_version = e.attribute("version").toInt();

    while (!n.isNull()) {
        // We don't nest elements in filter configuration. For now...
        TQDomElement e = n.toElement();
        TQString name;
        TQString type;
        TQString value;

        if (!e.isNull()) {
            if (e.tagName() == "property") {
                name = e.attribute("name");
                type = e.attribute("type");
                value = e.text();
                // XXX Convert the variant pro-actively to the right type?
                m_properties[name] = TQVariant(value);
            }
        }
        n = n.nextSibling();
    }
    //dump();
}

TQString KisFilterConfiguration::toString()
{
    TQDomDocument doc = TQDomDocument("filterconfig");
    TQDomElement root = doc.createElement( "filterconfig" );
    root.setAttribute( "name", m_name );
    root.setAttribute( "version", m_version );

    doc.appendChild( root );

    TQMap<TQString, TQVariant>::Iterator it;
    for ( it = m_properties.begin(); it != m_properties.end(); ++it ) {
        TQDomElement e = doc.createElement( "property" );
        e.setAttribute( "name", it.key().latin1() );
        TQVariant v = it.data();
        e.setAttribute( "type", v.typeName() );
        TQString s = v.asString();
        TQDomText text = doc.createCDATASection(v.asString() ); // XXX: Unittest this!
        e.appendChild(text);
        root.appendChild(e);
    }

    return doc.toString();
}

const TQString & KisFilterConfiguration::name() const
{
    return m_name;
}

TQ_INT32 KisFilterConfiguration::version() const
{
    return m_version;
}

void KisFilterConfiguration::setProperty(const TQString & name, const TQVariant & value)
{
    if ( m_properties.find( name ) == m_properties.end() ) {
        m_properties.insert( name, value );
    }
    else {
        m_properties[name] = value;
    }
}

bool KisFilterConfiguration::getProperty(const TQString & name, TQVariant & value)
{
   if ( m_properties.find( name ) == m_properties.end() ) {
       return false;
   }
   else {
       value = m_properties[name];
       return true;
   }
}

TQVariant KisFilterConfiguration::getProperty(const TQString & name)
{
    if ( m_properties.find( name ) == m_properties.end() ) {
        return TQVariant();
    }
    else {
        return m_properties[name];
    }
}


int KisFilterConfiguration::getInt(const TQString & name, int def)
{
    TQVariant v = getProperty(name);
    if (v.isValid())
        return v.asInt();
    else
        return def;

}

double KisFilterConfiguration::getDouble(const TQString & name, double def)
{
    TQVariant v = getProperty(name);
    if (v.isValid())
        return v.asDouble();
    else
        return def;
}

bool KisFilterConfiguration::getBool(const TQString & name, bool def)
{
    TQVariant v = getProperty(name);
    if (v.isValid())
        return v.asBool();
    else
        return def;
}

TQString KisFilterConfiguration::getString(const TQString & name, TQString def)
{
    TQVariant v = getProperty(name);
    if (v.isValid())
        return v.asString();
    else
        return def;
}

void KisFilterConfiguration::dump()
{
    TQMap<TQString, TQVariant>::Iterator it;
    for ( it = m_properties.begin(); it != m_properties.end(); ++it ) {
    }

}
