/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "verifier.h"
#include "misc.h"
#include "listviewitem.h"

#include <tdelocale.h>
#include <tdelistview.h>
#include <tdemessagebox.h>
#include <kprogress.h>
#include <kurl.h>

#include <tqheader.h>
#include <tqlabel.h>

// UI layouts
#include "ui/checksum_verifier.h"

using namespace KFTPGrabberBase;

namespace KFTPWidgets {

Verifier::Verifier(TQWidget *parent, const char *name)
 : KDialogBase(parent, name, true, i18n("Checksum verifier"), Cancel, Cancel, true),
   m_verifier(0)
{
  m_layout = new VerifierLayout(this);
  setMainWidget(m_layout);
  
  // Create columns
  m_layout->fileList->addColumn(i18n("Filename"));
  m_layout->fileList->addColumn(i18n("Checksum"), 100);
  
  m_layout->fileList->setAllColumnsShowFocus(true);
  m_layout->fileList->header()->setStretchEnabled(true, 0);
}

Verifier::~Verifier()
{
  if (m_verifier)
    delete m_verifier;
}

void Verifier::setFile(const TQString &filename)
{
  // Create the verifier
  m_verifier = new KFTPCore::ChecksumVerifier(filename);
  m_layout->currentFile->setText(KURL(filename).fileName());
  
  connect(m_verifier, SIGNAL(fileList(TQValueList<TQPair<TQString, TQString> >)), this, SLOT(slotHaveFileList(TQValueList<TQPair<TQString, TQString> >)));
  connect(m_verifier, SIGNAL(fileDone(const TQString&, KFTPCore::ChecksumVerifier::Result)), this, SLOT(slotFileDone(const TQString&, KFTPCore::ChecksumVerifier::Result)));
  connect(m_verifier, SIGNAL(progress(int)), this, SLOT(slotProgress(int)));
  connect(m_verifier, SIGNAL(error()), this, SLOT(slotError()));
  
  // Start the verification
  m_verifier->verify();
}

void Verifier::slotHaveFileList(TQValueList<TQPair<TQString, TQString> > list)
{
  for (TQValueList<TQPair<TQString, TQString> >::iterator i = list.begin(); i != list.end(); ++i) {
    KFTPWidgets::ListViewItem *item = new KFTPWidgets::ListViewItem(m_layout->fileList);
    item->setText(0, (*i).first);
    item->setText(1, (*i).second);
  }
}

void Verifier::slotFileDone(const TQString &filename, KFTPCore::ChecksumVerifier::Result result)
{
  KFTPWidgets::ListViewItem *item = static_cast<KFTPWidgets::ListViewItem*>(m_layout->fileList->findItem(filename, 0));
  
  if (item) {
    switch (result) {
      case KFTPCore::ChecksumVerifier::Ok: {
        item->setPixmap(0, loadSmallPixmap("ok"));
        item->setTextColor(0, TQColor(0, 200, 0));
        item->setTextColor(1, TQColor(0, 200, 0));
        break;
      }
      case KFTPCore::ChecksumVerifier::NotFound: {
        item->setPixmap(0, loadSmallPixmap("error"));
        item->setTextColor(0, TQColor(128, 128, 128));
        item->setTextColor(1, TQColor(128, 128, 128));
        break;
      }
      case KFTPCore::ChecksumVerifier::Error: {
        item->setPixmap(0, loadSmallPixmap("error"));
        item->setTextColor(0, TQColor(255, 0, 0));
        item->setTextColor(1, TQColor(255, 0, 0));
        break;
      }
    }
    
    m_layout->fileList->ensureItemVisible(item);
  }
}

void Verifier::slotProgress(int percent)
{
  m_layout->checkProgress->setProgress(percent);
  
  if (percent == 100) {
    KMessageBox::information(0, i18n("Verification complete!"));
  }
}

void Verifier::slotError()
{
  KMessageBox::error(0, i18n("Unable to open checksum file or file has an incorrect format!"));
  close();
}

}
#include "verifier.moc"
