/**********************************************************************
** Copyright (C) 2000-2002 Trolltech AS.  All rights reserved.
**
** This file is part of TQt Designer.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid TQt Enterprise Edition or TQt Professional Edition
** licenses may use this file in accordance with the TQt Commercial License
** Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about TQt Commercial License Agreements.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include <tqvariant.h> // HP-UX compiler need this here
#include "widgetfactory.h"
#include <widgetdatabase.h>
#include "metadatabase.h"
#include "mainwindow.h"
#include "formwindow.h"
#include "propertyeditor.h"
#include "tqlayout.h"
#include "listboxeditorimpl.h"
#include "listvieweditorimpl.h"
#include "iconvieweditorimpl.h"
#include "formwindow.h"
#include "multilineeditorimpl.h"
#include "../interfaces/widgetinterface.h"
#ifndef TQT_NO_TABLE
#include "tableeditorimpl.h"
#endif
#include "project.h"
#include "menubareditor.h"
#include "popupmenueditor.h"

#include "kdevdesigner_part.h"

#include <kiconloader.h>

#include <tqfeatures.h>

#include <tqpixmap.h>
#include <tqgroupbox.h>
#include <tqiconview.h>
#ifndef TQT_NO_TABLE
#include <tqtable.h>
#endif
#ifndef TQT_NO_SQL
#include <tqdatatable.h>
#endif
#include <tqdatetimeedit.h>
#include <tqlineedit.h>
#include <tqspinbox.h>
#include <tqmultilineedit.h>
#include <tqtextedit.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqwidgetstack.h>
#include <tqcombobox.h>
#include <tqtabbar.h>
#include <tqlistbox.h>
#include <tqlistview.h>
#include <tqobjectlist.h>
#include <tqlcdnumber.h>
#include <tqslider.h>
#include <tqdial.h>
#include <tqprogressbar.h>
#include <tqtextview.h>
#include <tqtextbrowser.h>
#include <tqframe.h>
#include <tqmetaobject.h>
#include <tqwidgetstack.h>
#include <tqwizard.h>
#include <tqvaluelist.h>
#include <tqtimer.h>
#include <tqscrollbar.h>
#include <tqmainwindow.h>
#include <tqmenubar.h>
#include <tqapplication.h>
#include <tqsplitter.h>
#include <tqtoolbox.h>
#include <tqsizegrip.h>
#ifndef TQT_NO_SQL
#include "database.h"
#endif

#define NO_STATIC_COLORS
#include <globaldefs.h>
#include <tqobject.h>

#include <tdelocale.h>

FormWindow *find_formwindow( TQWidget *w )
{
    if ( !w )
	return 0;
    for (;;) {
	if ( ::tqqt_cast<FormWindow*>(w) )
	    return (FormWindow*)w;
	if ( !w->parentWidget() )
	    return 0;
	w = w->parentWidget();
    }
}

void TQLayoutWidget::paintEvent( TQPaintEvent* )
{
    TQPainter p ( this );
    p.setPen( red );
    p.drawRect( rect() );
}


QDesignerTabWidget::QDesignerTabWidget( TQWidget *parent, const char *name )
    : TQTabWidget( parent, name ), dropIndicator( 0 ), dragPage( 0 ), mousePressed( FALSE )
{
    tabBar()->setAcceptDrops( TRUE );
    tabBar()->installEventFilter( this );
}

int QDesignerTabWidget::currentPage() const
{
    return tabBar()->currentTab();
}

void QDesignerTabWidget::setCurrentPage( int i )
{
    tabBar()->setCurrentTab( i );
}

TQString QDesignerTabWidget::pageTitle() const
{
    return ((TQTabWidget*)this)->tabLabel( TQTabWidget::currentPage() );
}

void QDesignerTabWidget::setPageTitle( const TQString& title )
{
    changeTab( TQTabWidget::currentPage(), title );
}

void QDesignerTabWidget::setPageName( const TQCString& name )
{
    if ( TQTabWidget::currentPage() )
	TQTabWidget::currentPage()->setName( name );
}

TQCString QDesignerTabWidget::pageName() const
{
    if ( !TQTabWidget::currentPage() )
	return 0;
    return TQTabWidget::currentPage()->name();
}

int QDesignerTabWidget::count() const
{
    return tabBar()->count();
}

bool QDesignerTabWidget::eventFilter( TQObject *o, TQEvent *e )
{
    if ( TQT_BASE_OBJECT(o) != TQT_BASE_OBJECT(tabBar()) ) return FALSE;

    switch ( e->type() ) {
    case TQEvent::MouseButtonPress: {
	mousePressed = TRUE;
	TQMouseEvent *me = (TQMouseEvent*)e;
	pressPoint = me->pos();
    }
    break;
    case TQEvent::MouseMove: {
	TQMouseEvent *me = (TQMouseEvent*)e;
	if ( mousePressed && ( pressPoint - me->pos()).manhattanLength() > TQApplication::startDragDistance() ) {
	    TQTextDrag *drg = new TQTextDrag( TQString::number( (long) this ) , this );
	    mousePressed = FALSE;
	    dragPage = TQTabWidget::currentPage();
	    dragLabel = TQTabWidget::tabLabel( dragPage );

	    int index = indexOf( dragPage );

	    removePage( dragPage );
	    if ( !drg->dragMove() ) {
		insertTab( dragPage, dragLabel, index );
		showPage( dragPage );
	    }
	    if ( dropIndicator )
		dropIndicator->hide();
	}
    }
    break;
    case TQEvent::DragLeave:	{
	if ( dropIndicator )
	    dropIndicator->hide();
    }
    break;
    case TQEvent::DragMove: {
	TQDragEnterEvent *de = (TQDragEnterEvent*) e;
	if ( TQTextDrag::canDecode( de ) ) {
	    TQString text;
	    TQTextDrag::decode( de, text );
	    if ( text == TQString::number( (long)this ) )
		de->accept();
	    else
		return FALSE;
	}

	int index = 0;
	TQRect rect;
	for ( ; index < tabBar()->count(); index++ ) {
	    if ( tabBar()->tabAt( index )->rect().contains( de->pos() ) ) {
		rect = tabBar()->tabAt( index )->rect();
		break;
	    }
	}

	if ( index == tabBar()->count() -1 ) {
	    TQRect rect2 = rect;
	    rect2.setLeft( rect2.left() + rect2.width() / 2 );
	    if ( rect2.contains( de->pos() ) )
		index++;
	}

	if ( ! dropIndicator ) {
	    dropIndicator = new TQWidget( this );
	    dropIndicator->setBackgroundColor( red );
	}

	TQPoint pos;
	if ( index == tabBar()->count() )
	    pos = tabBar()->mapToParent( TQPoint( rect.x() + rect.width(), rect.y() ) );
	else
	    pos = tabBar()->mapToParent( TQPoint( rect.x(), rect.y() ) );

	dropIndicator->setGeometry( pos.x(), pos.y() , 3, rect.height() );
	dropIndicator->show();
    }
    break;
    case TQEvent::Drop: {
	TQDragEnterEvent *de = (TQDragEnterEvent*) e;
	if ( TQTextDrag::canDecode( de ) ) {
	    TQString text;
	    TQTextDrag::decode( de, text );
	    if ( text == TQString::number( (long)this ) ) {

		int newIndex = 0;
		for ( ; newIndex < tabBar()->count(); newIndex++ ) {
		    if ( tabBar()->tabAt( newIndex )->rect().contains( de->pos() ) )
			break;
		}

		if ( newIndex == tabBar()->count() -1 ) {
		    TQRect rect2 = tabBar()->tabAt( newIndex )->rect();
		    rect2.setLeft( rect2.left() + rect2.width() / 2 );
		    if ( rect2.contains( de->pos() ) )
			newIndex++;
		}

		int oldIndex = 0;
		for ( ; oldIndex < tabBar()->count(); oldIndex++ ) {
		    if ( tabBar()->tabAt( oldIndex )->rect().contains( pressPoint ) )
			break;
		}

		FormWindow *fw = find_formwindow( this );
		MoveTabPageCommand *cmd =
		    new MoveTabPageCommand( i18n( "Move Tab Page" ), fw, this,
					    dragPage, dragLabel, newIndex, oldIndex );
		fw->commandHistory()->addCommand( cmd );
		cmd->execute();
		de->accept();
	    }
	}
    }
    break;
    default:
	break;
    }
    return FALSE;
}


QDesignerWidgetStack::QDesignerWidgetStack( TQWidget *parent, const char *name )
    : TQWidgetStack( parent, name )
{
    prev = new TQToolButton( TQt::LeftArrow, this, "designer_wizardstack_button" );
    prev->setAutoRaise( TRUE );
    prev->setAutoRepeat( TRUE );
    prev->setSizePolicy( TQSizePolicy( TQSizePolicy::Ignored, TQSizePolicy::Ignored ) );
    next = new TQToolButton( TQt::RightArrow, this, "designer_wizardstack_button" );
    next->setAutoRaise( TRUE );
    next->setAutoRepeat( TRUE );
    next->setSizePolicy( TQSizePolicy( TQSizePolicy::Ignored, TQSizePolicy::Ignored ) );
    connect( prev, TQT_SIGNAL( clicked() ), this, TQT_SLOT( prevPage() ) );
    connect( next, TQT_SIGNAL( clicked() ), this, TQT_SLOT( nextPage() ) );
    updateButtons();
}

void QDesignerWidgetStack::updateButtons()
{
    prev->setGeometry( width() - 31, 1, 15, 15 );
    next->setGeometry( width() - 16, 1, 15, 15 );
    prev->show();
    next->show();
    prev->raise();
    next->raise();
}

void QDesignerWidgetStack::prevPage()
{
    setCurrentPage( currentPage() - 1 );
}

void QDesignerWidgetStack::nextPage()
{
    setCurrentPage( currentPage() + 1 );
}

int QDesignerWidgetStack::currentPage() const
{
    QDesignerWidgetStack* that = (QDesignerWidgetStack*) this;
    return that->pages.find( visibleWidget() );
}

void QDesignerWidgetStack::setCurrentPage( int i )
{
    // help next/prev page commands
    if ( i < 0 )
	i += count();
    if ( i >= count() )
	i -= count();

    if ( i < 0 || i >= count() )
	return;
    raiseWidget( pages.at( i ) );
    updateButtons();
}

TQCString QDesignerWidgetStack::pageName() const
{
    if ( !visibleWidget() )
	return 0;
    return visibleWidget()->name();
}

void QDesignerWidgetStack::setPageName( const TQCString& name )
{
    if ( visibleWidget() )
	visibleWidget()->setName( name );
}

int QDesignerWidgetStack::count() const
{
    return pages.count();
}

TQWidget* QDesignerWidgetStack::page( int i ) const
{
    if ( i < 0 || i >= count() )
	return 0;
    QDesignerWidgetStack* that = (QDesignerWidgetStack*) this;
    return that->pages.at( i );
}


int QDesignerWidgetStack::insertPage( TQWidget *p, int i )
{
    if ( i < 0 )
	pages.append( p );
    else
	pages.insert( (uint) i, p );
    addWidget( p );
    p->show();
    raiseWidget( p );
    TQApplication::sendPostedEvents();
    updateButtons();
    return pages.find( p );
}

int QDesignerWidgetStack::removePage( TQWidget *p )
{
    int i = pages.find( p );
    pages.remove( p );
    removeWidget( p );
    setCurrentPage( 0 );
    updateButtons();
    return i;
}


int QDesignerWizard::currentPageNum() const
{
    for ( int i = 0; i < pageCount(); ++i ) {
	if ( page( i ) == currentPage() )
	    return i;
    }
    return 0;
}





void QDesignerWizard::setCurrentPage( int i )
{
    if ( i < currentPageNum() ) {
	while ( i < currentPageNum() ) {
	    if ( currentPageNum() == 0 )
		break;
	    back();
	}

    } else {
	while ( i > currentPageNum() ) {
	    if ( currentPageNum() == pageCount() - 1 )
		break;
	    next();
	}
    }
}

TQString QDesignerWizard::pageTitle() const
{
    return title( currentPage() );
}

void QDesignerWizard::setPageTitle( const TQString& title )
{
    setTitle( currentPage(), title );
}

void QDesignerWizard::setPageName( const TQCString& name )
{
    if ( TQWizard::currentPage() )
	TQWizard::currentPage()->setName( name );
}

TQCString QDesignerWizard::pageName() const
{
    if ( !TQWizard::currentPage() )
	return 0;
    return TQWizard::currentPage()->name();
}

int QDesignerWizard::pageNum( TQWidget *p )
{
    for ( int i = 0; i < pageCount(); ++i ) {
	if ( page( i ) == p )
	    return i;
    }
    return -1;
}

void QDesignerWizard::addPage( TQWidget *p, const TQString &t )
{
    TQWizard::addPage( p, t );
    if ( removedPages.find( p ) )
	removedPages.remove( p );
}

void QDesignerWizard::removePage( TQWidget *p )
{
    TQWizard::removePage( p );
    removedPages.insert( p, p );
}

void QDesignerWizard::insertPage( TQWidget *p, const TQString &t, int index )
{
    TQWizard::insertPage( p, t, index );
    if ( removedPages.find( p ) )
	removedPages.remove( p );
}

TQMap< int, TQMap< TQString, TQVariant> > *defaultProperties = 0;
TQMap< int, TQStringList > *changedProperties = 0;

/*!
  \class WidgetFactory widgetfactory.h
  \brief Set of static functions for creating widgets, layouts and do other stuff

  The widget factory offers functions to create widgets, create and
  delete layouts find out other details - all based on the
  WidgetDatabase's data. So the functions that use ids use the same
  ids as in the WidgetDatabase.
*/


void WidgetFactory::saveDefaultProperties( TQObject *w, int id )
{
    TQMap< TQString, TQVariant> propMap;
    TQStrList lst = w->metaObject()->propertyNames( TRUE );
    for ( uint i = 0; i < lst.count(); ++i ) {
	TQVariant var = w->property( lst.at( i ) );
	if ( !var.isValid() && qstrcmp( "pixmap", lst.at( i ) ) == 0 )
	    var = TQVariant( TQPixmap() );
	else if ( !var.isValid() && qstrcmp( "iconSet", lst.at( i ) ) == 0 )
	    var = TQVariant( TQIconSet() );
	propMap.replace( lst.at( i ), var );
    }
    defaultProperties->replace( id, propMap );
}

void WidgetFactory::saveChangedProperties( TQObject *w, int id )
{
    TQStringList l = MetaDataBase::changedProperties( w );
    changedProperties->insert( id, l );
}

/*!  Creates a widget of the type which is registered as \a id as
  child of \a parent. The \a name is optional. If \a init is TRUE, the
  widget is initialized with some defaults, else the plain widget is
  created.
*/

TQWidget *WidgetFactory::create( int id, TQWidget *parent, const char *name, bool init, const TQRect *r, Qt::Orientation orient )
{
    TQString n = WidgetDatabase::className( id );
    if ( n.isEmpty() )
	return 0;

    if ( !defaultProperties ) {
	defaultProperties = new TQMap< int, TQMap< TQString, TQVariant> >();
	changedProperties = new TQMap< int, TQStringList >();
    }

    TQWidget *w = 0;
    TQString str = WidgetDatabase::createWidgetName( id );
    const char *s = str.latin1();
    w = createWidget( n, parent, name ? name : s, init, r, orient );
    if ( ::tqqt_cast<TQScrollView*>(w) )
	( (TQScrollView*)w )->disableSizeHintCaching();
    if ( !w && WidgetDatabase::isCustomWidget( id ) )
	w = createCustomWidget( parent, name ? name : s, MetaDataBase::customWidget( id ) );
    if ( !w )
	return 0;
    MetaDataBase::addEntry( TQT_TQOBJECT(w) );

    if ( !defaultProperties->contains( id ) )
	saveDefaultProperties( TQT_TQOBJECT(w), id );
    if ( !changedProperties->contains( id ) )
	saveChangedProperties( TQT_TQOBJECT(w), id );

    return w;
}

/*!  Creates a layout on the widget \a widget of the type \a type
  which can be \c HBox, \c VBox or \c Grid.
*/

TQLayout *WidgetFactory::createLayout( TQWidget *widget, TQLayout *layout, LayoutType type )
{
    int spacing = MainWindow::self->currentLayoutDefaultSpacing();
    int margin = 0;

    int metaspacing = MetaDataBase::spacing( TQT_TQOBJECT(widget) );
    int metamargin = MetaDataBase::margin( TQT_TQOBJECT(widget) );

    if ( ::tqqt_cast<TQLayoutWidget*>(widget) &&
	 ( WidgetDatabase::isContainer( WidgetDatabase::idFromClassName( WidgetFactory::classNameOf( TQT_TQOBJECT(widget) ) ) ) ||
	   widget && ::tqqt_cast<FormWindow*>(widget->parentWidget()) ) )
	margin = MainWindow::self->currentLayoutDefaultMargin();

    if ( !layout && ::tqqt_cast<TQTabWidget*>(widget) )
	widget = ((TQTabWidget*)widget)->currentPage();

    if ( !layout && ::tqqt_cast<TQWizard*>(widget) )
	widget = ((TQWizard*)widget)->currentPage();

    if ( !layout && ::tqqt_cast<TQMainWindow*>(widget) )
	widget = ((TQMainWindow*)widget)->centralWidget();

    if ( !layout && ::tqqt_cast<TQWidgetStack*>(widget) )
	widget = ((TQWidgetStack*)widget)->visibleWidget();

    if ( !layout && ::tqqt_cast<TQToolBox*>(widget) )
	widget = ((TQToolBox*)widget)->currentItem();

    MetaDataBase::addEntry( TQT_TQOBJECT(widget) );

    TQLayout *l = 0;
    int align = 0;
    if ( !layout && ::tqqt_cast<TQGroupBox*>(widget) ) {
	TQGroupBox *gb = (TQGroupBox*)widget;
	gb->setColumnLayout( 0, Qt::Vertical );
	layout = gb->layout();
	layout->setMargin( 0 );
	layout->setSpacing( 0 );
	switch ( type ) {
	case HBox:
	    l = new TQHBoxLayout( layout );
	    break;
	case VBox:
	    l = new TQVBoxLayout( layout );
	    break;
	case Grid:
	    l = new QDesignerGridLayout( layout );
	    break;
	default:
	    return 0;
	}
	align = TQt::AlignTop;
	MetaDataBase::setMargin( TQT_TQOBJECT(gb), metamargin );
	MetaDataBase::setSpacing( TQT_TQOBJECT(gb), metaspacing );
    } else {
	if ( layout ) {
	    switch ( type ) {
	    case HBox:
		l = new TQHBoxLayout( layout );
		break;
	    case VBox:
		l = new TQVBoxLayout( layout );
		break;
	    case Grid:
		l = new QDesignerGridLayout( layout );
		break;
	    default:
		return 0;
	    }
	    MetaDataBase::addEntry( TQT_TQOBJECT(l) );
	    l->setSpacing( spacing );
	    l->setMargin( margin );
	} else {
	    switch ( type ) {
	    case HBox:
		l = new TQHBoxLayout( widget );
		break;
	    case VBox:
		l = new TQVBoxLayout( widget );
		break;
	    case Grid:
		l = new QDesignerGridLayout( widget );
		break;
	    default:
		return 0;
	    }
	    MetaDataBase::addEntry( TQT_TQOBJECT(l) );
	    if ( widget ) {
		MetaDataBase::setMargin( TQT_TQOBJECT(widget), metamargin );
		MetaDataBase::setSpacing( TQT_TQOBJECT(widget), metaspacing );
	    } else {
		l->setMargin( margin );
		l->setSpacing( spacing );
	    }
	}
    }
    l->setAlignment( align );
    MetaDataBase::addEntry( TQT_TQOBJECT(l) );
    return l;
}

void WidgetFactory::deleteLayout( TQWidget *widget )
{
    if ( !widget )
	return;

    if ( ::tqqt_cast<TQTabWidget*>(widget) )
	widget = ((TQTabWidget*)widget)->currentPage();
    if ( ::tqqt_cast<TQWizard*>(widget) )
	widget = ((TQWizard*)widget)->currentPage();
    if ( ::tqqt_cast<TQMainWindow*>(widget) )
	widget = ((TQMainWindow*)widget)->centralWidget();
    if ( ::tqqt_cast<TQWidgetStack*>(widget) )
	widget = ((TQWidgetStack*)widget)->visibleWidget();
    if ( ::tqqt_cast<TQToolBox*>(widget) )
	widget = ((TQToolBox*)widget)->currentItem();
    delete widget->layout();
}

/*!  Factory functions for creating a widget of the type \a className
  as child of \a parent with the name \a name.

  If \a init is TRUE, some initial default properties are set. This
  has to be in sync with the initChangedProperties() function!
*/

TQWidget *WidgetFactory::createWidget( const TQString &className, TQWidget *parent, const char *name, bool init,
				      const TQRect *r, Qt::Orientation orient )
{
    if ( className == TQPUSHBUTTON_OBJECT_NAME_STRING ) {
	TQPushButton *b = 0;
	if ( init ) {
	    b = new QDesignerPushButton( parent, name );
	    b->setText( TQString::fromLatin1( name ) );
	} else {
	    b = new QDesignerPushButton( parent, name );
	}
	TQWidget *w = find_formwindow( b );
	b->setAutoDefault( w && ::tqqt_cast<TQDialog*>(((FormWindow*)w)->mainContainer()) );
	return b;
    } else if ( className == TQTOOLBUTTON_OBJECT_NAME_STRING ) {
	if ( init ) {
	    QDesignerToolButton *tb = new QDesignerToolButton( parent, name );
	    if ( ::tqqt_cast<TQToolBox*>(widgetOfContainer(parent))) {
		tb->setUsesTextLabel(TRUE);
		tb->setTextLabel("...");
		tb->setAutoRaise(TRUE);
		tb->setTextPosition(TQToolButton::BesideIcon);
	    } else {
		tb->setText( "..." );
	    }
	    return tb;
	}
	return new QDesignerToolButton( parent, name );
    } else if ( className == TQCHECKBOX_OBJECT_NAME_STRING ) {
	if ( init ) {
	    QDesignerCheckBox *cb = new QDesignerCheckBox( parent, name );
	    cb->setText( TQString::fromLatin1( name ) );
	    return cb;
	}
	return new QDesignerCheckBox( parent, name );
    } else if ( className == TQRADIOBUTTON_OBJECT_NAME_STRING ) {
	if ( init ) {
	    QDesignerRadioButton *rb = new QDesignerRadioButton( parent, name );
	    rb->setText( TQString::fromLatin1( name ) );
	    return rb;
	}
	return new QDesignerRadioButton( parent, name );
    } else if ( className == TQGROUPBOX_OBJECT_NAME_STRING ) {
	if ( init )
	    return new TQGroupBox( TQString::fromLatin1( name ), parent, name );
	return new TQGroupBox( parent, name );
    } else if ( className == TQBUTTONGROUP_OBJECT_NAME_STRING ) {
	if ( init )
	    return new TQButtonGroup( TQString::fromLatin1( name ), parent, name );
	return new TQButtonGroup( parent, name );
    } else if ( className == TQICONVIEW_OBJECT_NAME_STRING ) {
#if !defined(TQT_NO_ICONVIEW)
	TQIconView* iv = new TQIconView( parent, name );
	if ( init )
	    (void) new TQIconViewItem( iv, i18n( "New Item" ) );
	return iv;
#else
	return 0;
#endif
    } else if ( className == TQTABLE_OBJECT_NAME_STRING ) {
#if !defined(TQT_NO_TABLE)
	if ( init )
	    return new TQTable( 3, 3, parent, name );
	return new TQTable( parent, name );
#else
	return 0;
#endif
#ifndef TQT_NO_SQL
    } else if ( className == TQDATATABLE_OBJECT_NAME_STRING ) {
	return new TQDataTable( parent, name );
#endif //TQT_NO_SQL
    } else if ( className == TQDATEEDIT_OBJECT_NAME_STRING ) {
	return new TQDateEdit( parent, name );
    } else if ( className == TQTIMEEDIT_OBJECT_NAME_STRING ) {
	return new TQTimeEdit( parent, name );
    } else if ( className == TQDATETIMEEDIT_OBJECT_NAME_STRING ) {
	return new TQDateTimeEdit( parent, name );
    }
    else if ( className == TQLISTBOX_OBJECT_NAME_STRING ) {
	TQListBox* lb = new TQListBox( parent, name );
	if ( init ) {
	    lb->insertItem( i18n( "New Item" ) );
	    lb->setCurrentItem( 0 );
	}
	return lb;
    } else if ( className == TQLISTVIEW_OBJECT_NAME_STRING ) {
	TQListView *lv = new TQListView( parent, name );
	lv->setSorting( -1 );
	if ( init ) {
	    lv->addColumn( i18n( "Column 1" ) );
	    lv->setCurrentItem( new TQListViewItem( lv, i18n( "New Item" ) ) );
	}
	return lv;
    } else if ( className == TQLINEEDIT_OBJECT_NAME_STRING )
	return new TQLineEdit( parent, name );
    else if ( className == TQSPINBOX_OBJECT_NAME_STRING )
	return new TQSpinBox( parent, name );
    else if ( className == TQSPLITTER_OBJECT_NAME_STRING )
	return new TQSplitter( parent, name );
    else if ( className == TQMULTILINEEDIT_OBJECT_NAME_STRING )
	return new TQMultiLineEdit( parent, name );
    else if ( className == TQTEXTEDIT_OBJECT_NAME_STRING )
	return new TQTextEdit( parent, name );
    else if ( className == TQLABEL_OBJECT_NAME_STRING  || className == "TextLabel" ) {
	QDesignerLabel *l = new QDesignerLabel( parent, name );
	if ( init ) {
	    l->setText( TQString::fromLatin1( name ) );
	    MetaDataBase::addEntry( TQT_TQOBJECT(l) );
	    MetaDataBase::setPropertyChanged( TQT_TQOBJECT(l), "text", TRUE );
	}
	return l;
    } else if ( className == "PixmapLabel" ) {
	QDesignerLabel *l = new QDesignerLabel( parent, name );
	if ( init ) {
	    l->setPixmap( BarIcon( "designer_qtlogo.png", KDevDesignerPartFactory::instance() ) );
	    l->setScaledContents( TRUE );
	    MetaDataBase::addEntry( TQT_TQOBJECT(l) );
	    MetaDataBase::setPropertyChanged( TQT_TQOBJECT(l), "pixmap", TRUE );
	    MetaDataBase::setPropertyChanged( TQT_TQOBJECT(l), "scaledContents", TRUE );
	}
	return l;
    } else if ( className == TQLAYOUTWIDGET_OBJECT_NAME_STRING )
	return new TQLayoutWidget( parent, name );
    else if ( className == TQTABWIDGET_OBJECT_NAME_STRING ) {
	TQTabWidget *tw = new QDesignerTabWidget( parent, name );
	if ( init ) {
	    FormWindow *fw = find_formwindow( parent );
	    TQWidget *w = fw ? new QDesignerWidget( fw, tw, "tab" ) : new TQWidget( tw, "tab" );
	    tw->addTab( w, i18n("Tab 1") );
	    MetaDataBase::addEntry( TQT_TQOBJECT(w) );
	    w = fw ? new QDesignerWidget( fw, tw, "tab" ) : new TQWidget( tw, "tab" );
	    tw->addTab( w, i18n("Tab 2") );
	    MetaDataBase::addEntry( TQT_TQOBJECT(tw) );
	    MetaDataBase::addEntry( TQT_TQOBJECT(w) );
	}
	return tw;
    } else if ( className == TQWIDGETSTACK_OBJECT_NAME_STRING ) {
	QDesignerWidgetStack *ws = new QDesignerWidgetStack( parent, name );
	if ( init ) {
	    FormWindow *fw = find_formwindow( parent );
	    TQWidget *w = fw ? new QDesignerWidget( fw, ws, "WStackPage" ) : new TQWidget( ws, "WStackPage" );
	    ws->insertPage( w );
	    MetaDataBase::addEntry( TQT_TQOBJECT(w) );
	    MetaDataBase::addEntry( TQT_TQOBJECT(ws) );
	}
	return ws;
    } else if ( className == TQCOMBOBOX_OBJECT_NAME_STRING ) {
	return new TQComboBox( FALSE, parent, name );
    } else if ( className == TQWIDGET_OBJECT_NAME_STRING ) {
	if ( parent &&
	     ( ::tqqt_cast<FormWindow*>(parent) ||
	       ::tqqt_cast<TQWizard*>(parent) ||
	       ::tqqt_cast<TQTabWidget*>(parent) ||
	       ::tqqt_cast<TQWidgetStack*>(parent) ||
	       ::tqqt_cast<TQToolBox*>(parent) ||
	       ::tqqt_cast<TQMainWindow*>(parent) ) ) {
	    FormWindow *fw = find_formwindow( parent );
	    if ( fw ) {
		QDesignerWidget *dw = new QDesignerWidget( fw, parent, name );
		MetaDataBase::addEntry( TQT_TQOBJECT(dw) );
		return dw;
	    }
	}
	return new TQWidget( parent, name );
    } else if ( className == TQDIALOG_OBJECT_NAME_STRING ) {
	TQDialog *dia = 0;
	if ( ::tqqt_cast<FormWindow*>(parent) )
	    dia = new QDesignerDialog( (FormWindow*)parent, parent, name );
	else
	    dia = new TQDialog( parent, name );
	if ( parent )
	    dia->reparent( parent, TQPoint( 0, 0 ), TRUE );
	return dia;
    } else if ( className == TQWIZARD_OBJECT_NAME_STRING ) {
	TQWizard *wiz = new QDesignerWizard( parent, name );
	if ( parent )
	    wiz->reparent( parent, TQPoint( 0, 0 ), TRUE );
	if ( init && ::tqqt_cast<FormWindow*>(parent) ) {
	    QDesignerWidget *dw = new QDesignerWidget( (FormWindow*)parent, wiz, "WizardPage" );
	    MetaDataBase::addEntry( TQT_TQOBJECT(dw) );
	    wiz->addPage( dw, i18n( "Page" ) );
	    TQTimer::singleShot( 0, wiz, TQT_SLOT( next() ) );
	}
	return wiz;
    } else if ( className == "Spacer" ) {
	Spacer *s = new Spacer( parent, name );
	MetaDataBase::addEntry( TQT_TQOBJECT(s) );
	MetaDataBase::setPropertyChanged( TQT_TQOBJECT(s), "orientation", TRUE );
	MetaDataBase::setPropertyChanged( TQT_TQOBJECT(s), "sizeType", TRUE );
	if ( !r )
	    return s;
	if ( !r->isValid() || r->width() < 2 && r->height() < 2 )
	    s->setOrientation( orient );
	else if ( r->width() < r->height() )
	    s->setOrientation( Qt::Vertical );
	else
	    s->setOrientation( Qt::Horizontal );
	return s;
    } else if ( className == TQLCDNUMBER_OBJECT_NAME_STRING )
	return new TQLCDNumber( parent, name );
    else if ( className == TQPROGRESSBAR_OBJECT_NAME_STRING )
	return new TQProgressBar( parent, name );
    else if ( className == TQTEXTVIEW_OBJECT_NAME_STRING )
	return new TQTextView( parent, name );
    else if ( className == TQTEXTBROWSER_OBJECT_NAME_STRING )
	return new TQTextBrowser( parent, name );
    else if ( className == TQDIAL_OBJECT_NAME_STRING )
	return new TQDial( parent, name );
    else if ( className == TQSLIDER_OBJECT_NAME_STRING ) {
	TQSlider *s = new TQSlider( parent, name );
	if ( !r )
	    return s;
	if ( !r->isValid() || r->width() < 2 && r->height() < 2 )
	    s->setOrientation( orient );
	else if ( r->width() > r->height() )
	    s->setOrientation( Qt::Horizontal );
	MetaDataBase::addEntry( TQT_TQOBJECT(s) );
	MetaDataBase::setPropertyChanged( TQT_TQOBJECT(s), "orientation", TRUE );
	return s;
    } else if ( className == TQSCROLLBAR_OBJECT_NAME_STRING ) {
	TQScrollBar *s = new TQScrollBar( parent, name );
	if ( !r )
	    return s;
	if ( !r->isValid() || r->width() < 2 && r->height() < 2 )
	    s->setOrientation( orient );
	else if ( r->width() > r->height() )
	    s->setOrientation( Qt::Horizontal );
	MetaDataBase::addEntry( TQT_TQOBJECT(s) );
	MetaDataBase::setPropertyChanged( TQT_TQOBJECT(s), "orientation", TRUE );
	return s;
    } else if ( className == TQFRAME_OBJECT_NAME_STRING ) {
	if ( !init )
	    return new TQFrame( parent, name );
	TQFrame *f = new TQFrame( parent, name );
	f->setFrameStyle( TQFrame::StyledPanel | TQFrame::Raised );
	return f;
    } else if ( className == "Line" ) {
	Line *l = new Line( parent, name );
	MetaDataBase::addEntry( TQT_TQOBJECT(l) );
	MetaDataBase::setPropertyChanged( TQT_TQOBJECT(l), "orientation", TRUE );
	MetaDataBase::setPropertyChanged( TQT_TQOBJECT(l), "frameShadow", TRUE );
	MetaDataBase::setPropertyChanged( TQT_TQOBJECT(l), "frameShape", TRUE );
	if ( !r )
	    return l;
	if ( !r->isValid() || r->width() < 2 && r->height() < 2 )
	    l->setOrientation( orient );
	else if ( r->width() < r->height() )
	    l->setOrientation( Qt::Vertical );
	return l;
    } else if ( className == TQMAINWINDOW_OBJECT_NAME_STRING ) {
	TQMainWindow *mw = new TQMainWindow( parent, name, 0 );
	mw->setDockEnabled( TQt::DockMinimized, FALSE );
	QDesignerWidget *dw = new QDesignerWidget( (FormWindow*)parent, mw, "central widget" );
	mw->setDockMenuEnabled( FALSE );
	MetaDataBase::addEntry( TQT_TQOBJECT(dw) );
	mw->setCentralWidget( dw );
	(void)mw->statusBar();
	dw->show();
	return mw;
    } else if ( className == TQTOOLBOX_OBJECT_NAME_STRING ) {
	if ( !init )
	    return new QDesignerToolBox( parent, name );
	TQToolBox *tb = new QDesignerToolBox( parent, name );
	FormWindow *fw = find_formwindow( parent );
	TQWidget *w = fw ? new QDesignerWidget( fw, tb, "page1" ) :
		     new TQWidget( tb, "page1" );
	tb->addItem( w, i18n("Page 1") );
	MetaDataBase::addEntry( TQT_TQOBJECT(w) );
	w = fw ? new QDesignerWidget( fw, tb, "page2" ) : new TQWidget( tb, "page2" );
	tb->addItem( w, i18n("Page 2") );
	MetaDataBase::addEntry( TQT_TQOBJECT(tb) );
	MetaDataBase::addEntry( TQT_TQOBJECT(w) );
	return tb;
    }
#ifndef TQT_NO_SQL
    else if ( className == TQDATABROWSER_OBJECT_NAME_STRING ) {
	TQWidget *w = new QDesignerDataBrowser( parent, name );
	if ( parent )
	    w->reparent( parent, TQPoint( 0, 0 ), TRUE );
	return w;
    } else if ( className == TQDATAVIEW_OBJECT_NAME_STRING ) {
	TQWidget *w = new QDesignerDataView( parent, name );
	if ( parent )
	    w->reparent( parent, TQPoint( 0, 0 ), TRUE );
	return w;
    }
#endif

    WidgetInterface *iface = 0;
    widgetManager()->queryInterface( className, &iface );
    if ( !iface )
	return 0;

    TQWidget *w = iface->create( className, parent, name );
#ifdef TQT_CONTAINER_CUSTOM_WIDGETS
    if ( init && WidgetDatabase::isCustomPluginWidget( WidgetDatabase::idFromClassName( className ) ) ) {
	TQWidgetContainerInterfacePrivate *iface2 = 0;
	iface->queryInterface( IID_TQWidgetContainer, (TQUnknownInterface**)&iface2 );
	if ( iface2 ) {
	    iface2->addPage( className, w, "Page", -1 );
	    iface2->release();
	}
    }
#endif // TQT_CONTAINER_CUSTOM_WIDGETS
    iface->release();
    return w;
}



/*!  Find out which type the layout of the widget is. Returns \c HBox,
  \c VBox, \c Grid or \c NoLayout.  \a layout points to this
  TQWidget::layout() of \a w or to 0 after the function call.
*/

WidgetFactory::LayoutType WidgetFactory::layoutType( TQWidget *w, TQLayout *&layout )
{
    layout = 0;

    if ( ::tqqt_cast<TQTabWidget*>(w) )
	w = ((TQTabWidget*)w)->currentPage();
    if ( ::tqqt_cast<TQWizard*>(w) )
	w = ((TQWizard*)w)->currentPage();
    if ( ::tqqt_cast<TQMainWindow*>(w) )
	w = ((TQMainWindow*)w)->centralWidget();
    if ( ::tqqt_cast<TQWidgetStack*>(w) )
	w = ((TQWidgetStack*)w)->visibleWidget();
    if ( ::tqqt_cast<TQToolBox*>(w) )
	w = ((TQToolBox*)w)->currentItem();

    if ( ::tqqt_cast<TQSplitter*>(w) )
	return ( (TQSplitter*)w )->orientation() ==Qt::Horizontal ? HBox : VBox;

    if ( !w || !w->layout() )
	return NoLayout;
    TQLayout *lay = w->layout();

    if ( ::tqqt_cast<TQGroupBox*>(w) ) {
	TQObjectList *l = TQT_TQOBJECT(lay)->queryList( TQLAYOUT_OBJECT_NAME_STRING );
	if ( l && l->first() )
	    lay = (TQLayout*)l->first();
	delete l;
    }
    layout = lay;

    if ( ::tqqt_cast<TQHBoxLayout*>(lay) )
	return HBox;
    else if ( ::tqqt_cast<TQVBoxLayout*>(lay) )
	return VBox;
    else if ( ::tqqt_cast<TQGridLayout*>(lay) )
	return Grid;
    return NoLayout;
}

/*!
  \overload
*/
WidgetFactory::LayoutType WidgetFactory::layoutType( TQLayout *layout )
{
    if ( ::tqqt_cast<TQHBoxLayout*>(layout) )
	return HBox;
    else if ( ::tqqt_cast<TQVBoxLayout*>(layout) )
	return VBox;
    else if ( ::tqqt_cast<TQGridLayout*>(layout) )
	return Grid;
    return NoLayout;
}

/*!
  \overload
*/
WidgetFactory::LayoutType WidgetFactory::layoutType( TQWidget *w )
{
    TQLayout *l = 0;
    return layoutType( w, l );
}


TQWidget *WidgetFactory::layoutParent( TQLayout *layout )
{
    TQObject *o = TQT_TQOBJECT(layout);
    while ( o ) {
	if ( o->isWidgetType() )
	    return (TQWidget*)o;
	o = o->parent();
    }
    return 0;
}

/*!  Returns the widget into which children should be inserted when \a
  w is a container known to the designer.

  Usually that is \a w itself, sometimes it is different (e.g. a
  tabwidget is known to the designer as a container but the child
  widgets should be inserted into the current page of the
  tabwidget. So in this case this function returns the current page of
  the tabwidget.)
 */
TQWidget* WidgetFactory::containerOfWidget( TQWidget *w )
{
    if ( !w )
	return w;
    if ( ::tqqt_cast<TQTabWidget*>(w) )
	return ((TQTabWidget*)w)->currentPage();
    if ( ::tqqt_cast<TQWizard*>(w) )
	return ((TQWizard*)w)->currentPage();
    if ( ::tqqt_cast<TQWidgetStack*>(w) )
	return ((TQWidgetStack*)w)->visibleWidget();
    if ( ::tqqt_cast<TQToolBox*>(w) )
	return ((TQToolBox*)w)->currentItem();
    if ( ::tqqt_cast<TQMainWindow*>(w) )
	return ((TQMainWindow*)w)->centralWidget();
#ifdef TQT_CONTAINER_CUSTOM_WIDGETS
    if ( !WidgetDatabase::isCustomPluginWidget( WidgetDatabase::idFromClassName( classNameOf( w ) ) ) )
	return w;
    WidgetInterface *iface = 0;
    widgetManager()->queryInterface( classNameOf( w ), &iface );
    if ( !iface )
	return w;
    TQWidgetContainerInterfacePrivate *iface2 = 0;
    iface->queryInterface( IID_TQWidgetContainer, (TQUnknownInterface**)&iface2 );
    if ( !iface2 )
	return w;
    TQWidget *c = iface2->containerOfWidget( w->className(), w );
    iface2->release();
    iface->release();
    if ( c )
	return c;
#endif // TQT_CONTAINER_CUSTOM_WIDGETS
    return w;
}

/*!  Returns the actual designer widget of the container \a w. This is
  normally \a w itself, but might be a parent or grand parent of \a w
  (e.g. when working with a tabwidget and \a w is the container which
  contains and layouts childs, but the actual widget known to the
  designer is the tabwidget which is the parent of \a w. So this
  function returns the tabwidget then.)
*/

TQWidget* WidgetFactory::widgetOfContainer( TQWidget *w )
{
    if ( w && ::tqqt_cast<TQWidgetStack*>(w->parentWidget()) )
	w = w->parentWidget();
    if ( w->parentWidget() && w->parentWidget()->parentWidget() &&
	 w->parentWidget()->parentWidget()->parentWidget() &&
	 ::tqqt_cast<TQToolBox*>(w->parentWidget()->parentWidget()->parentWidget()) )
	return w->parentWidget()->parentWidget()->parentWidget();
    while ( w ) {
	int id = WidgetDatabase::idFromClassName( WidgetFactory::classNameOf( TQT_TQOBJECT(w) ) );
	if ( WidgetDatabase::isContainer( id ) ||
	     w && ::tqqt_cast<FormWindow*>(w->parentWidget()) )
	    return w;
	w = w->parentWidget();
    }
    return w;
}

/*!
  Returns whether \a o is a passive interactor or not.
 */

bool WidgetFactory::lastWasAPassiveInteractor = FALSE;
TQGuardedPtr<TQObject> *WidgetFactory::lastPassiveInteractor = new TQGuardedPtr<TQObject>();

bool WidgetFactory::isPassiveInteractor( TQObject* o )
{
    if ( lastPassiveInteractor && *lastPassiveInteractor && (TQObject*)(*lastPassiveInteractor) == o )
	return lastWasAPassiveInteractor;
    lastWasAPassiveInteractor = FALSE;
    (*lastPassiveInteractor) = o;
    if ( TQApplication::activePopupWidget() ) // if a popup is open, we have to make sure that this one is closed, else X might do funny things
	return ( lastWasAPassiveInteractor = TRUE );

    if ( ::tqqt_cast<TQTabBar*>(o) )
	return ( lastWasAPassiveInteractor = TRUE );
    else if ( ::tqqt_cast<TQSizeGrip*>(o) )
	return ( lastWasAPassiveInteractor = TRUE );
    else if ( ::tqqt_cast<TQButton*>(o) &&
	      ( ::tqqt_cast<TQTabBar*>(o->parent()) || ::tqqt_cast<TQToolBox*>(o->parent()) ) )
	return ( lastWasAPassiveInteractor = TRUE );
    else if ( ::tqqt_cast<TQPushButton*>(o) && ::tqqt_cast<TQWizard*>(o->parent()) )
	return ( lastWasAPassiveInteractor = TRUE );
    else if ( ::tqqt_cast<TQMenuBar*>(o) && ::tqqt_cast<TQMainWindow*>(o->parent()) )
	return ( lastWasAPassiveInteractor = TRUE );
//    else if ( ::tqqt_cast<TQDockWindowHandle*>(o) )
    else if ( o->inherits( TQDOCKWINDOWHANDLE_OBJECT_NAME_STRING ) )
	return ( lastWasAPassiveInteractor = TRUE );
//    else if ( ::tqqt_cast<TQHideDock*>(o) )
    else if ( o->inherits( TQHIDEDOCK_OBJECT_NAME_STRING ) )
	return ( lastWasAPassiveInteractor = TRUE );
    else if ( qstrcmp( o->name(), "designer_wizardstack_button" ) == 0 )
	return ( lastWasAPassiveInteractor = TRUE );
#ifdef TQT_CONTAINER_CUSTOM_WIDGETS
    if ( !o->isWidgetType() )
	return ( lastWasAPassiveInteractor = FALSE );
    WidgetInterface *iface = 0;
    TQWidget *w = (TQWidget*)o;
    while ( !iface && w && !::tqqt_cast<FormWindow*>(w) ) {
	widgetManager()->queryInterface( classNameOf( w ), &iface );
	w = w->parentWidget();
    }
    if ( !iface )
	return ( lastWasAPassiveInteractor = FALSE );
    TQWidgetContainerInterfacePrivate *iface2 = 0;
    iface->queryInterface( IID_TQWidgetContainer, (TQUnknownInterface**)&iface2 );
    if ( !iface2 )
	return ( lastWasAPassiveInteractor = FALSE );
    TQWidget *fw = MainWindow::self->isAFormWindowChild( (TQWidget*)o );
    if ( !fw )
	return ( lastWasAPassiveInteractor = FALSE );
    TQWidget *dw = ( (FormWindow*)fw )->designerWidget( (TQWidget*)o );
    if ( !dw )
	return ( lastWasAPassiveInteractor = FALSE );
    lastWasAPassiveInteractor = iface2->isPassiveInteractor( dw->className(), (TQWidget*)o );
    iface2->release();
    iface->release();
#endif // TQT_CONTAINER_CUSTOM_WIDGETS
    return lastWasAPassiveInteractor;
}


/*!
  Returns the class name of object \a o that should be used for externally (i.e. for saving)
 */
const char* WidgetFactory::classNameOf( TQObject* o )
{
    if ( o->isA( "PropertyObject" ) )
	return o->className();
    if (WidgetDatabase::isCustomPluginWidget(WidgetDatabase::idFromClassName(o->className())))
        return o->className();
    else if ( ::tqqt_cast<QDesignerTabWidget*>(o) )
	return TQTABWIDGET_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<QDesignerWidgetStack*>(o) )
	return TQWIDGETSTACK_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<TQWidgetStack*>(o) )
	return "TQWeDoNotWantToBreakTabWidget";
    else if ( ::tqqt_cast<QDesignerDialog*>(o) )
	return TQDIALOG_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<QDesignerWidget*>(o) )
	return TQWIDGET_OBJECT_NAME_STRING;
    else if ( o->inherits( "CustomWidget" ) )
	return ( (CustomWidget*)o )->realClassName().latin1();
    else if ( ::tqqt_cast<QDesignerLabel*>(o) )
	return TQLABEL_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<QDesignerWizard*>(o) )
	return TQWIZARD_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<QDesignerPushButton*>(o) )
	return TQPUSHBUTTON_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<QDesignerToolButton*>(o) )
	return TQTOOLBUTTON_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<QDesignerRadioButton*>(o) )
	return TQRADIOBUTTON_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<QDesignerCheckBox*>(o) )
	return TQCHECKBOX_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<MenuBarEditor*>(o) )
	return TQMENUBAR_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<QDesignerToolBar*>(o) )
	return TQTOOLBAR_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<QDesignerAction*>(o) )
	return TQACTION_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<QDesignerActionGroup*>(o) )
	return TQACTIONGROUP_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<PopupMenuEditor*>(o) )
	return TQPOPUPMENU_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<QDesignerToolBox*>(o) )
	return TQTOOLBOX_OBJECT_NAME_STRING;
#ifndef TQT_NO_SQL
    else if ( ::tqqt_cast<QDesignerDataBrowser*>(o) )
	return TQDATABROWSER_OBJECT_NAME_STRING;
    else if ( ::tqqt_cast<QDesignerDataView*>(o) )
	return TQDATAVIEW_OBJECT_NAME_STRING;
#endif
    return o->className();
}

TQString WidgetFactory::defaultSignal( TQObject *w )
{
    if ( ::tqqt_cast<TQRadioButton*>(w) || ::tqqt_cast<TQCheckBox*>(w) )
	return "toggled";
    else if ( ::tqqt_cast<TQButton*>(w) || ::tqqt_cast<TQButtonGroup*>(w) )
	return "clicked";
    else if ( ::tqqt_cast<TQTextBrowser*>(w) )
	return "linkClicked";
    else if ( ::tqqt_cast<TQLineEdit*>(w) || ::tqqt_cast<TQTextEdit*>(w) )
	return "textChanged";
    else if ( ::tqqt_cast<TQListView*>(w) || ::tqqt_cast<TQIconView*>(w) ||
	      ::tqqt_cast<TQListBox*>(w) || ::tqqt_cast<TQTable*>(w) )
	return "selectionChanged";
    else if ( ::tqqt_cast<TQTabWidget*>(w) )
	return "selected";
    else if ( ::tqqt_cast<TQToolBox*>(w) )
	return "currentChanged";
    else if ( ::tqqt_cast<TQWidgetStack*>(w) )
	return "aboutToShow";
    else if ( ::tqqt_cast<TQSpinBox*>(w) || ::tqqt_cast<TQSlider*>(w) ||
	      ::tqqt_cast<TQScrollBar*>(w) || ::tqqt_cast<TQDateEdit*>(w) ||
	      ::tqqt_cast<TQTimeEdit*>(w) || ::tqqt_cast<TQDateTimeEdit*>(w) ||
	      ::tqqt_cast<TQDial*>(w) )
	return "valueChanged";
    else if ( ::tqqt_cast<TQComboBox*>(w) )
	return "activated";
    return TQString();
}

/*!  As some properties are set by default when creating a widget this
  functions markes this properties as changed. Has to be in sync with
  createWidget()!
*/

void WidgetFactory::initChangedProperties( TQObject *o )
{
    if ( MainWindow::self && MainWindow::self->currProject() &&
	 MainWindow::self->currProject()->fakeFormFileFor( o ) )
	return;
    MetaDataBase::setPropertyChanged( o, "name", TRUE );
    if ( !::tqqt_cast<QDesignerToolBar*>(o) && !::tqqt_cast<MenuBarEditor*>(o) )
	MetaDataBase::setPropertyChanged( o, "geometry", TRUE );

    if ( ::tqqt_cast<TQPushButton*>(o) ||
	 ::tqqt_cast<TQRadioButton*>(o) ||
	 ::tqqt_cast<TQCheckBox*>(o) ||
	 ::tqqt_cast<TQToolButton*>(o) ) {
	if (::tqqt_cast<TQToolButton*>(o) && ::tqqt_cast<TQToolBox*>(widgetOfContainer((TQWidget*)o->parent()))) {
	    MetaDataBase::setPropertyChanged( o, "usesTextLabel", TRUE );
	    MetaDataBase::setPropertyChanged( o, "textLabel", TRUE );
	    MetaDataBase::setPropertyChanged( o, "autoRaise", TRUE );
	    MetaDataBase::setPropertyChanged( o, "textPosition", TRUE );
	} else {
	    MetaDataBase::setPropertyChanged( o, "text", TRUE );
	}
    }
    else if ( ::tqqt_cast<TQGroupBox*>(o) )
	MetaDataBase::setPropertyChanged( o, "title", TRUE );
    else if ( o->isA( TQFRAME_OBJECT_NAME_STRING ) ) {
	MetaDataBase::setPropertyChanged( o, "frameShadow", TRUE );
	MetaDataBase::setPropertyChanged( o, "frameShape", TRUE );
    } else if ( ::tqqt_cast<TQTabWidget*>(o) || ::tqqt_cast<TQWizard*>(o) ) {
	MetaDataBase::setPropertyChanged( o, "pageTitle", TRUE );
	MetaDataBase::setPropertyChanged( o, "pageName", TRUE );
	MetaDataBase::setPropertyChanged( o, "currentPage", TRUE );
    } else if ( ::tqqt_cast<TQWidgetStack*>(o) ) {
	MetaDataBase::setPropertyChanged( o, "currentPage", TRUE );
	MetaDataBase::setPropertyChanged( o, "pageName", TRUE );
    } else if ( ::tqqt_cast<TQToolBox*>(o) ) {
	MetaDataBase::setPropertyChanged( o, "currentIndex", TRUE );
	MetaDataBase::setPropertyChanged( o, "itemName", TRUE );
	MetaDataBase::setPropertyChanged( o, "itemLabel", TRUE );
	MetaDataBase::setPropertyChanged( o, "itemIconSet", TRUE );
	MetaDataBase::setPropertyChanged( o, "itemToolTip", TRUE );
	MetaDataBase::setPropertyChanged( o, "itemBackgroundMode", TRUE );
#ifndef TQT_NO_TABLE
    } else if ( ::tqqt_cast<TQTable*>(o) ) {
#  ifndef TQT_NO_SQL
        if (!::tqqt_cast<TQDataTable*>(o) )
#  endif
        {
            MetaDataBase::setPropertyChanged( o, "numRows", TRUE );
            MetaDataBase::setPropertyChanged( o, "numCols", TRUE );
            TQTable *t = (TQTable*)o;
            for ( int i = 0; i < 3; ++i ) {
                t->horizontalHeader()->setLabel( i, TQString::number( i + 1 ) );
                t->verticalHeader()->setLabel( i, TQString::number( i + 1 ) );
            }
        }
#endif
    } else if ( ::tqqt_cast<TQSplitter*>(o)  ) {
	MetaDataBase::setPropertyChanged( o, "orientation", TRUE );
    } else if ( ::tqqt_cast<QDesignerToolBar*>(o)  ) {
	MetaDataBase::setPropertyChanged( o, "label", TRUE );
    } else if ( ::tqqt_cast<MenuBarEditor*>(o)  ) {
	MetaDataBase::setPropertyChanged( o, "itemName", TRUE );
	MetaDataBase::setPropertyChanged( o, "itemNumber", TRUE );
	MetaDataBase::setPropertyChanged( o, "itemText", TRUE );
    }
}

bool WidgetFactory::hasSpecialEditor( int id, TQObject *editorWidget )
{
    TQString className = WidgetDatabase::className( id );

    if ( className.contains( "ListBox" ) )
	return TRUE;
    if ( className.contains( "ComboBox" ) )
	return TRUE;
    if ( className.contains( "ListView" ) )
	return TRUE;
    if ( className.contains( "IconView" ) )
	return TRUE;
    if ( className == TQTEXTEDIT_OBJECT_NAME_STRING || className == TQMULTILINEEDIT_OBJECT_NAME_STRING )
	return TRUE;
    if ( ::tqqt_cast<TQTable*>(editorWidget) != 0 )
	return TRUE;

    return FALSE;
}

bool WidgetFactory::hasItems( int id, TQObject *editorWidget )
{
    TQString className = WidgetDatabase::className( id );

    if ( className.contains( "ListBox" ) || className.contains( "ListView" ) ||
	 className.contains( "IconView" )  || className.contains( "ComboBox" ) ||
	 ::tqqt_cast<TQTable*>(editorWidget) != 0 )
	return TRUE;

    return FALSE;
}

void WidgetFactory::editWidget( int id, TQWidget *parent, TQWidget *editWidget, FormWindow *fw )
{
    TQString className = WidgetDatabase::className( id );

    if ( className.contains( "ListBox" ) ) {
	if ( !::tqqt_cast<TQListBox*>(editWidget) )
	    return;
	ListBoxEditor *e = new ListBoxEditor( parent, editWidget, fw );
	e->exec();
	delete e;
	return;
    }

    if ( className.contains( "ComboBox" ) ) {
	if ( !::tqqt_cast<TQComboBox*>(editWidget) )
	    return;
	TQComboBox *cb = (TQComboBox*)editWidget;
	ListBoxEditor *e = new ListBoxEditor( parent, cb->listBox(), fw );
	e->exec();
	delete e;
	cb->update();
	return;
    }

    if ( className.contains( "ListView" ) ) {
	if ( !::tqqt_cast<TQListView*>(editWidget) )
	    return;
	TQListView *lv = (TQListView*)editWidget;
	ListViewEditor *e = new ListViewEditor( parent, lv, fw );
	e->exec();
	delete e;
	return;
    }

    if ( className.contains( "IconView" ) ) {
	if ( !::tqqt_cast<TQIconView*>(editWidget) )
	    return;
	IconViewEditor *e = new IconViewEditor( parent, editWidget, fw );
	e->exec();
	delete e;
	return;
    }

    if ( className == TQMULTILINEEDIT_OBJECT_NAME_STRING || className == TQTEXTEDIT_OBJECT_NAME_STRING ) {
	MultiLineEditor *e = new MultiLineEditor( FALSE, TRUE, parent, editWidget, fw );
	e->exec();
	delete e;
	return;
    }
#ifndef TQT_NO_TABLE
    if (::tqqt_cast<TQTable*>(editWidget) != 0) {
	TableEditor *e = new TableEditor( parent, editWidget, fw );
	e->exec();
	delete e;
	return;
    }
#endif
}

bool WidgetFactory::canResetProperty( TQObject *w, const TQString &propName )
{
    if ( propName == "name" || propName == "geometry" )
	return FALSE;
    TQStringList l = *changedProperties->find( WidgetDatabase::idFromClassName( WidgetFactory::classNameOf( w ) ) );
    return l.findIndex( propName ) == -1;
}

bool WidgetFactory::resetProperty( TQObject *w, const TQString &propName )
{
    const TQMetaProperty *p = w->metaObject()->property( w->metaObject()->
							findProperty( propName, TRUE ), TRUE );
    if (!p )
	return FALSE;
    return p->reset( w );
}

TQVariant WidgetFactory::defaultValue( TQObject *w, const TQString &propName )
{
    if ( propName == "wordwrap" ) {
	int v = defaultValue( w, "alignment" ).toInt();
	return TQVariant( ( v & WordBreak ) == WordBreak, 0 );
    } else if ( propName == "toolTip" || propName == "whatsThis" ) {
	return TQVariant( TQString::fromLatin1( "" ) );
    } else if ( w->inherits( "CustomWidget" ) ) {
	return TQVariant();
    } else if ( propName == "frameworkCode" ) {
	return TQVariant( TRUE, 0 );
    } else if ( propName == "layoutMargin" || propName == "layoutSpacing" ) {
	return TQVariant( -1 );
    }

    return *( *defaultProperties->find( WidgetDatabase::idFromClassName( classNameOf( w ) ) ) ).find( propName );
}

TQString WidgetFactory::defaultCurrentItem( TQObject *w, const TQString &propName )
{
    const TQMetaProperty *p = w->metaObject()->
			     property( w->metaObject()->findProperty( propName, TRUE ), TRUE );
    if ( !p ) {
	int v = defaultValue( w, "alignment" ).toInt();
	if ( propName == "hAlign" ) {
	    if ( ( v & AlignAuto ) == AlignAuto )
		return "AlignAuto";
	    if ( ( v & AlignLeft ) == AlignLeft )
		return "AlignLeft";
	    if ( ( v & AlignCenter ) == AlignCenter || ( v & AlignHCenter ) == AlignHCenter )
		return "AlignHCenter";
	    if ( ( v & AlignRight ) == AlignRight )
		return "AlignRight";
	    if ( ( v & AlignJustify ) == AlignJustify )
		return "AlignJustify";
	} else if ( propName == "vAlign" ) {
	    if ( ( v & AlignTop ) == AlignTop )
		return "AlignTop";
	    if ( ( v & AlignCenter ) == AlignCenter || ( v & AlignVCenter ) == AlignVCenter )
		return "AlignVCenter";
	    if ( ( v & AlignBottom ) == AlignBottom )
		return "AlignBottom";
	} else if ( propName == "resizeMode" ) {
	    return "Auto";
	}
	return TQString();

    }
    return p->valueToKey( defaultValue( w, propName ).toInt() );
}

TQWidget *WidgetFactory::createCustomWidget( TQWidget *parent, const char *name, MetaDataBase::CustomWidget *w )
{
    if ( !w )
	return 0;
    return new CustomWidget( parent, name, w );
}

TQVariant WidgetFactory::property( TQObject *w, const char *name )
{
    int id = w->metaObject()->findProperty( name, TRUE );
    const TQMetaProperty* p = w->metaObject()->property( id, TRUE );
    if ( !p || !p->isValid() )
	return MetaDataBase::fakeProperty( w, name );
    return w->property( name );
}

void QDesignerLabel::updateBuddy()
{

    if ( myBuddy.isEmpty() )
	return;

    TQObjectList *l = topLevelWidget()->queryList( TQWIDGET_OBJECT_NAME_STRING, myBuddy, FALSE, TRUE );
    if ( !l || !l->first() ) {
	delete l;
	return;
    }

    TQLabel::setBuddy( (TQWidget*)l->first() );
    delete l;
}


void QDesignerWidget::resizeEvent( TQResizeEvent* e)
{
    if ( need_frame ) {
	TQPainter p(this);
	p.setPen( backgroundColor() );
	p.drawRect( TQRect( TQPoint(0,0), e->oldSize() ) );
    }
}

void QDesignerWidget::paintEvent( TQPaintEvent *e )
{
    if ( need_frame ) {
	TQPainter p(this);
	p.setPen( backgroundColor().dark() );
	p.drawRect( rect() );
    }
    formwindow->paintGrid( this, e );
}

void QDesignerDialog::paintEvent( TQPaintEvent *e )
{
    formwindow->paintGrid( this, e );
}

TQSizePolicy TQLayoutWidget::sizePolicy() const
{
    return sp;
}

bool TQLayoutWidget::event( TQEvent *e )
{
    if ( e && ( e->type() == TQEvent::ChildInserted ||
		e->type() == TQEvent::ChildRemoved ||
		e->type() == TQEvent::LayoutHint ||
		e->type() == TQEvent::Reparent ) )
	updateSizePolicy();
    return TQWidget::event( e );
}

/*
  This function must be called on TQLayoutWidget creation and whenever
  the TQLayoutWidget's parent layout changes (e.g., from a TQHBoxLayout
  to a TQVBoxLayout), because of the (illogical) way layouting works.
*/
void TQLayoutWidget::updateSizePolicy()
{
    if ( childrenListObject().isEmpty() || childrenListObject().count() == 0 ) {
	sp = TQWidget::sizePolicy();
	return;
    }

    /*
      TQSizePolicy::MayShrink & friends are private. Here we assume the
      following:

	  Fixed = 0
	  Maximum = MayShrink
	  Minimum = MayGrow
	  Preferred = MayShrink | MayGrow
    */

    int ht = (int) TQSizePolicy::Preferred;
    int vt = (int) TQSizePolicy::Preferred;

    if ( layout() ) {
	/*
	  parentLayout is set to the parent layout if there is one and if it is
	  top level, in which case layouting is illogical.
	*/
	TQLayout *parentLayout = 0;
	if ( parent() && parent()->isWidgetType() ) {
	    parentLayout = ((TQWidget *)parent())->layout();
	    if ( parentLayout &&
		 ::tqqt_cast<TQLayoutWidget*>(parentLayout->mainWidget()) )
		parentLayout = 0;
	}

	TQObjectListIt it( childrenListObject() );
	TQObject *o;

	if ( ::tqqt_cast<TQVBoxLayout*>(layout()) ) {
	    if ( ::tqqt_cast<TQHBoxLayout*>(parentLayout) )
		vt = TQSizePolicy::Minimum;
	    else
		vt = TQSizePolicy::Fixed;

	    while ( ( o = it.current() ) ) {
		++it;
		if ( !o->isWidgetType() || ( (TQWidget*)o )->testWState( WState_ForceHide ) )
		    continue;
		TQWidget *w = (TQWidget*)o;

		if ( !w->sizePolicy().mayGrowHorizontally() )
		    ht &= ~TQSizePolicy::Minimum;
		if ( !w->sizePolicy().mayShrinkHorizontally() )
		    ht &= ~TQSizePolicy::Maximum;
		if ( w->sizePolicy().mayGrowVertically() )
		    vt |= TQSizePolicy::Minimum;
		if ( w->sizePolicy().mayShrinkVertically() )
		    vt |= TQSizePolicy::Maximum;
	    }
	} else if ( ::tqqt_cast<TQHBoxLayout*>(layout()) ) {
	    if ( ::tqqt_cast<TQVBoxLayout*>(parentLayout) )
		ht = TQSizePolicy::Minimum;
	    else
		ht = TQSizePolicy::Fixed;

	    while ( ( o = it.current() ) ) {
		++it;
		if ( !o->isWidgetType() || ( (TQWidget*)o )->testWState( WState_ForceHide ) )
		    continue;
		TQWidget *w = (TQWidget*)o;

		if ( w->sizePolicy().mayGrowHorizontally() )
		    ht |= TQSizePolicy::Minimum;
		if ( w->sizePolicy().mayShrinkHorizontally() )
		    ht |= TQSizePolicy::Maximum;
		if ( !w->sizePolicy().mayGrowVertically() )
		    vt &= ~TQSizePolicy::Minimum;
		if ( !w->sizePolicy().mayShrinkVertically() )
		    vt &= ~TQSizePolicy::Maximum;
	    }
	} else if ( ::tqqt_cast<TQGridLayout*>(layout()) ) {
	    ht = TQSizePolicy::Fixed;
	    vt = TQSizePolicy::Fixed;
	    if ( parentLayout ) {
		if ( ::tqqt_cast<TQVBoxLayout*>(parentLayout) )
		    ht = TQSizePolicy::Minimum;
		else if ( ::tqqt_cast<TQHBoxLayout*>(parentLayout) )
		    vt = TQSizePolicy::Minimum;
	    }

	    while ( ( o = it.current() ) ) {
		++it;
		if ( !o->isWidgetType() || ( (TQWidget*)o )->testWState( WState_ForceHide ) )
		    continue;
		TQWidget *w = (TQWidget*)o;

		if ( w->sizePolicy().mayGrowHorizontally() )
		    ht |= TQSizePolicy::Minimum;
		if ( w->sizePolicy().mayShrinkHorizontally() )
		    ht |= TQSizePolicy::Maximum;
		if ( w->sizePolicy().mayGrowVertically() )
		    vt |= TQSizePolicy::Minimum;
		if ( w->sizePolicy().mayShrinkVertically() )
		    vt |= TQSizePolicy::Maximum;
	    }
	}
#ifdef USE_QT4
	if ( layout()->expanding() & TQSizePolicy::Horizontally )
#else // USE_QT4
	if ( layout()->expanding() & TQSizePolicy::Horizontally )
#endif // USE_QT4
	    ht = TQSizePolicy::Expanding;
#ifdef USE_QT4
	if ( layout()->expanding() & TQSizePolicy::Vertically )
#else // USE_QT4
	if ( layout()->expanding() & TQSizePolicy::Vertically )
#endif // USE_QT4
	    vt = TQSizePolicy::Expanding;

	layout()->invalidate();
    }

    sp = TQSizePolicy( (TQSizePolicy::SizeType) ht, (TQSizePolicy::SizeType) vt );
    updateGeometry();
}

void CustomWidget::paintEvent( TQPaintEvent *e )
{
    if ( ::tqqt_cast<FormWindow*>(parentWidget()) ) {
	( (FormWindow*)parentWidget() )->paintGrid( this, e );
    } else {
	TQPainter p( this );
	p.fillRect( rect(), colorGroup().dark() );
	p.setPen( colorGroup().light() );
	p.drawText( 2, 2, width() - 4, height() - 4, TQt::AlignAuto | TQt::AlignTop, cusw->className );
	p.drawPixmap( ( width() - cusw->pixmap->width() ) / 2,
		      ( height() - cusw->pixmap->height() ) / 2,
		      *cusw->pixmap );
    }
}


CustomWidgetFactory::CustomWidgetFactory()
{
}

TQWidget *CustomWidgetFactory::createWidget( const TQString &className, TQWidget *parent, const char *name ) const
{
    MetaDataBase::CustomWidget *w = MetaDataBase::customWidget( WidgetDatabase::idFromClassName( className ) );
    if ( !w )
	return 0;
    return WidgetFactory::createCustomWidget( parent, name, w );
}


QDesignerToolBox::QDesignerToolBox( TQWidget *parent, const char *name )
    : TQToolBox( parent, name )
{
}

TQString QDesignerToolBox::itemLabel() const
{
    return TQToolBox::itemLabel( currentIndex() );
}

void QDesignerToolBox::setItemLabel( const TQString &l )
{
    TQToolBox::setItemLabel( currentIndex(), l );
}

TQCString QDesignerToolBox::itemName() const
{
    return currentItem() ? currentItem()->name() : 0;
}

void QDesignerToolBox::setItemName( const TQCString &n )
{
    if (currentItem())
	currentItem()->setName( n );
}


TQt::BackgroundMode QDesignerToolBox::itemBackgroundMode() const
{
    return (item(0) ? item(0)->backgroundMode() : PaletteBackground);
}

void QDesignerToolBox::setItemBackgroundMode( BackgroundMode bmode )
{
    for (int i = 0; i < count(); ++i) {
	TQWidget *w = item(i);
	w->setBackgroundMode( bmode );
	w->update();
    }
}

void QDesignerToolBox::itemInserted( int index )
{
    if (count() > 1)
	item(index)->setBackgroundMode(item(index>0?0:1)->backgroundMode());
}
