/* This file is part of the KDE project
   Copyright (C) 2002 Heinrich Kuettler <heinrich.kuettler@gmx.de>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include <tqlistbox.h>
#include <tqpainter.h>

#include <tdeapplication.h>
#include <kcombobox.h>
#include <tdeglobalsettings.h>
#include <tdetoolbar.h>
#include <kdebug.h>

#include "symbolaction.h"

/*
 * The items for the SymbolCombos. *
 */

KFORMULA_NAMESPACE_BEGIN

class SymbolComboItem : public TQListBoxItem
{
public:
    SymbolComboItem( const TQString&, const TQFont& , TQChar, TQComboBox* combo );
    virtual ~SymbolComboItem();

    virtual int width( const TQListBox* ) const;
    virtual int height( const TQListBox* ) const;

protected:
    virtual void paint( TQPainter *p );

private:
    TQComboBox *m_combo;
    TQString m_name;
    TQFont m_font;
    TQChar m_symbol;

    static int widest;
};

int SymbolComboItem::widest = 0;

SymbolComboItem::SymbolComboItem( const TQString &name, const TQFont& font, 
                                  TQChar symbol, TQComboBox *combo )
    : TQListBoxItem( combo->listBox() ),
      m_combo( combo ),
      m_name( name ),
      m_font( font ),
      m_symbol( symbol )
{
    setText( name );
    int charWidth = TQFontMetrics( m_font ).width( TQChar( m_symbol ) );
    widest = TQMAX( widest, charWidth );
}

SymbolComboItem::~SymbolComboItem()
{
}

int SymbolComboItem::width( const TQListBox * /*lb*/ ) const
{
    return widest + TQFontMetrics( TDEGlobalSettings::generalFont() ).width( text() ) + 12;
}

int SymbolComboItem::height( const TQListBox * /*lb*/ ) const
{
    int generalHeight = TQFontMetrics( TDEGlobalSettings::generalFont() ).lineSpacing();
    int fontHeight = TQFontMetrics( m_font ).lineSpacing();
    return TQMAX( generalHeight, fontHeight ) + 2;
}

void SymbolComboItem::paint( TQPainter *p )
{
    p->setFont( m_font );
    TQFontMetrics fm( p->fontMetrics() );
    p->drawText( 3, fm.ascent() + fm.leading() / 2,
                 TQString( "%1" ).arg( TQChar( m_symbol ) ) );

    p->setFont( TDEGlobalSettings::generalFont() );
    fm = p->fontMetrics();
    p->drawText( widest + 6, height( m_combo->listBox() ) / 2 + fm.strikeOutPos(), m_name );
}

/*
 * The symbol action *
 */
SymbolAction::SymbolAction( TQObject* parent, const char* name )
    : TDESelectAction( parent, name )
{
    setEditable( FALSE );
}

SymbolAction::SymbolAction( const TQString& text, const TDEShortcut& cut,
                            const TQObject* receiver, const char* slot,
                            TQObject* parent, const char* name )
    : TDESelectAction( text, cut, receiver, slot, parent, name )
{
    setEditable( FALSE );
}

int SymbolAction::plug( TQWidget* w, int index )
{
    if (kapp && !kapp->authorizeTDEAction(name()))
        return -1;
    if ( w->inherits( "TDEToolBar" ) )
    {
        TDEToolBar* bar = static_cast<TDEToolBar*>( w );
        int id_ = TDEAction::getToolButtonID();
        KComboBox *cb = new KComboBox( bar );
        connect( cb, TQT_SIGNAL( activated( const TQString & ) ),
                 TQT_SLOT( slotActivated( const TQString & ) ) );
        cb->setEnabled( isEnabled() );
        bar->insertWidget( id_, comboWidth(), cb, index );
        cb->setMinimumWidth( cb->sizeHint().width() );

        addContainer( bar, id_ );

        connect( bar, TQT_SIGNAL( destroyed() ), this, TQT_SLOT( slotDestroyed() ) );

        updateItems( containerCount() - 1 );

        return containerCount() - 1;
    }
    else return TDESelectAction::plug( w, index );
}

void SymbolAction::setSymbols( const TQStringList &names,
                               const TQFont& font,
                               const TQMemArray<TQChar>& chars )
{
    m_chars = chars;
    m_font = font;
    setItems( names );

    int len = containerCount();
    for ( int i = 0; i < len; ++i )
        updateItems( i );
}

void SymbolAction::updateItems( int id )
{
    TQWidget *w = container( id );
    if ( w->inherits( "TDEToolBar" ) ) {
        TQWidget *r = static_cast<TDEToolBar*>( w )->getWidget( itemId( id ) );
        if ( r->inherits( TQCOMBOBOX_OBJECT_NAME_STRING ) ) {
            TQComboBox *cb = static_cast<TQComboBox*>( r );
            cb->clear();

            for( uint i = 0; i < items().count(); ++i ) {
                new SymbolComboItem( *items().at( i ), m_font, m_chars.at( i ), cb );
            }
            cb->setMinimumWidth( cb->sizeHint().width() );
        }
    }
}

KFORMULA_NAMESPACE_END
