/* This file is part of the KDE project
   Copyright (C) 2001 Laurent Montel <lmontel@mandrakesoft.com>
   Copyright (C) 2005 Thorsten Zachmann <zachmann@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef kprcommand_h
#define kprcommand_h

#include <kcommand.h>
#include <tqptrlist.h>
#include <tqpoint.h>
#include <tqcolor.h>
#include <tqsize.h>
#include "global.h"
#include <KoPictureCollection.h>
#include <tqvaluelist.h>
#include <tqpen.h>
#include <tqbrush.h>
#include <KoPageLayoutDia.h>
#include <KoParagLayout.h>
#include <KoTextCommand.h>
#include <KoPoint.h>
#include <KoSize.h>
#include <tqvariant.h>
#include <tqvaluevector.h>
#include "KPrBackground.h"
#include "KPrVariableCollection.h"
#include <KoPen.h>

class KPrDocument;
class KPrTextObject;
class KPrObject;
class KPr2DObject;
class KPrPixmapObject;
class KPrGroupObject;
class KPrView;
class KoParagLayout;
class KPrPage;
class KoCustomVariable;
class KoLinkVariable;
class KPrPolylineObject;
class KPrFieldVariable;
class KPrTimeVariable;
class KPrDateVariable;
class KPrPgNumVariable;

class KPrShadowCmd : public KNamedCommand
{
public:
    struct ShadowValues
    {
        int shadowDistance;
        ShadowDirection shadowDirection;
        TQColor shadowColor;
    };

    KPrShadowCmd( const TQString &_name, TQPtrList<ShadowValues> &_oldShadow, ShadowValues _newShadow,
               TQPtrList<KPrObject> &_objects, KPrDocument *_doc );
    ~KPrShadowCmd();

    virtual void execute();
    virtual void unexecute();

protected:
    KPrDocument *doc;
    TQPtrList<ShadowValues> oldShadow;
    TQPtrList<KPrObject> objects;
    ShadowValues newShadow;
    KPrPage *m_page;
};

class KPrSetOptionsCmd : public KNamedCommand
{
public:
    KPrSetOptionsCmd( const TQString &_name, TQValueList<KoPoint> &_diffs, TQPtrList<KPrObject> &_objects,
                   double _rastX, double _rastY, double _orastX, double _orastY,
                   const TQColor &_txtBackCol, const TQColor &_otxtBackCol, KPrDocument *_doc );
    ~KPrSetOptionsCmd();

    virtual void execute();
    virtual void unexecute();

protected:

    TQValueList<KoPoint> diffs;
    TQPtrList<KPrObject> objects;
    double gridX;
    double gridY;
    double oldGridX;
    double oldGridY;
    KPrDocument *doc;
    TQColor txtBackCol;
    TQColor otxtBackCol;
};

class KPrSetBackCmd : public KNamedCommand
{
public:
    KPrSetBackCmd( const TQString &name, const KPrBackGround::Settings &settings,
                const KPrBackGround::Settings &oldSettings,
                bool useMasterBackground,
                bool takeGlobal, KPrDocument *doc, KPrPage *page );

    virtual void execute();
    virtual void unexecute();

protected:

    KPrBackGround::Settings m_settings;
    KPrBackGround::Settings m_oldSettings;
    bool m_useMasterBackground;
    bool m_oldUseMasterBackground;
    bool m_takeGlobal;
    KPrDocument *m_doc;
    KPrPage *m_page;
};

class KPrRotateCmd : public KNamedCommand
{
public:
    struct RotateValues
    {
        float angle;
    };

    KPrRotateCmd( const TQString &_name, float newAngle, TQPtrList<KPrObject> &objects,
               KPrDocument *doc, bool addAngle = false );
    ~KPrRotateCmd();

    virtual void execute();
    virtual void unexecute();

protected:
    KPrDocument *m_doc;
    TQPtrList<RotateValues> m_oldAngles;
    TQPtrList<KPrObject> m_objects;
    float m_newAngle;
    //necessary for duplicate object, we can duplicated and add angle.
    bool m_addAngle;
    KPrPage *m_page;
};

class KPrResizeCmd : public KNamedCommand
{
public:
    KPrResizeCmd( const TQString &_name, const KoPoint &_m_diff, const KoSize &_r_diff,
               KPrObject *_object, KPrDocument *_doc );
    ~KPrResizeCmd();

    virtual void execute();
    virtual void unexecute();

protected:

    KoPoint m_diff;
    KoSize r_diff;
    KPrObject *object;
    KPrDocument *doc;
    KPrPage *m_page;
};

class KPrChgPixCmd : public KNamedCommand
{
public:
    KPrChgPixCmd( const TQString &_name, KPrPixmapObject *_oldObject, KPrPixmapObject *_newObject,
               KPrDocument *_doc, KPrPage *_page );
    ~KPrChgPixCmd();

    virtual void execute();
    virtual void unexecute();

protected:

    KPrPixmapObject *oldObject, *newObject;
    KPrDocument *doc;
    KPrPage *m_page;
};

class KPrDeleteCmd : public KNamedCommand
{
public:
    KPrDeleteCmd( const TQString &_name, TQPtrList<KPrObject> &_objects, KPrDocument *_doc,
               KPrPage *_page);
    ~KPrDeleteCmd();

    virtual void execute();
    virtual void unexecute();

protected:
    TQPtrList<KPrObject> m_oldObjectList;
    TQPtrList<KPrObject> m_objectsToDelete;
    KPrDocument *m_doc;
    KPrPage *m_page;
};

class KPrEffectCmd : public KNamedCommand
{
public:
    struct EffectStruct {
        int appearStep, disappearStep;
        Effect effect;
        Effect2 effect2;
        Effect3 effect3;
        EffectSpeed m_appearSpeed;
        EffectSpeed m_disappearSpeed;
        bool disappear;
        int appearTimer, disappearTimer;
        bool appearSoundEffect, disappearSoundEffect;
        TQString a_fileName, d_fileName;
    };

    KPrEffectCmd( const TQString &_name, const TQPtrList<KPrObject> &_objs,
               const TQValueList<EffectStruct> &_oldEffects, EffectStruct _newEffect );
    ~KPrEffectCmd();

    virtual void execute();
    virtual void unexecute();

protected:

    TQValueList<EffectStruct> oldEffects;
    EffectStruct newEffect;
    TQPtrList<KPrObject> objs;
};

class KPrGroupObjCmd : public KNamedCommand
{
public:
    KPrGroupObjCmd( const TQString &_name,
                 const TQPtrList<KPrObject> &_objects,
                 KPrDocument *_doc, KPrPage *_page );
    ~KPrGroupObjCmd();

    virtual void execute();
    virtual void unexecute();

protected:

    TQPtrList<KPrObject> m_objectsToGroup;
    TQPtrList<KPrObject> m_oldObjectList;
    KPrDocument *m_doc;
    KPrGroupObject *m_groupObject;
    KPrPage *m_page;

};

class UnGroupObjCmd : public KNamedCommand
{
public:
    UnGroupObjCmd( const TQString &_name,
                   KPrGroupObject *grpObj_,
                   KPrDocument *_doc, KPrPage *_page );
    ~UnGroupObjCmd();

    virtual void execute();
    virtual void unexecute();

protected:

    TQPtrList<KPrObject> m_groupedObjects;
    KPrGroupObject *m_groupObject;
    KPrDocument *m_doc;
    KPrPage *m_page;
};


class KPrInsertCmd : public KNamedCommand
{
public:
    KPrInsertCmd( const TQString &name, const TQValueList<KPrObject *> objects, KPrDocument *doc, KPrPage *page );
    KPrInsertCmd( const TQString &name, KPrObject *object, KPrDocument *doc, KPrPage *page );
    ~KPrInsertCmd();

    virtual void execute();
    virtual void unexecute();

protected:
    TQValueList<KPrObject *> m_objects;
    KPrObject *m_object;
    KPrDocument *m_doc;
    KPrPage *m_page;
};


class KPrLowerRaiseCmd : public KNamedCommand
{
public:
    KPrLowerRaiseCmd( const TQString &_name, const TQPtrList<KPrObject>& _oldList,
                   const TQPtrList<KPrObject>& _newList,
                   KPrDocument *_doc, KPrPage *_page );
    ~KPrLowerRaiseCmd();

    virtual void execute();
    virtual void unexecute();

protected:

    TQPtrList<KPrObject> oldList, newList;
    KPrDocument *doc;
    KPrPage *m_page;
};

class KPrMoveByCmd : public KNamedCommand
{
public:
    KPrMoveByCmd( const TQString &_name, const KoPoint &_diff, TQPtrList<KPrObject> &_objects,
               KPrDocument *_doc, KPrPage *_page );
    ~KPrMoveByCmd();

    virtual void execute();
    virtual void unexecute();

protected:

    KoPoint diff;
    TQPtrList<KPrObject> objects;
    KPrDocument *doc;
    KPrPage *m_page;
};

/**
 * Command to align objects
 */
class KPrAlignCmd : public KNamedCommand
{
public:
    KPrAlignCmd( const TQString &_name, TQPtrList<KPrObject> &_objects, AlignType _at, KPrDocument *_doc );
    ~KPrAlignCmd();

    virtual void execute();
    virtual void unexecute();

protected:
    TQPtrList<KoPoint> diffs;
    TQPtrList<KPrObject> objects;
    KPrDocument *doc;
    KPrPage *m_page;
};

class KoPenCmd : public KNamedCommand
{
public:
    struct Pen {
        Pen( KoPen p, LineEnd lb, LineEnd le )
        : pen( p )
        , lineBegin( lb )
        , lineEnd( le )
        {}

        KoPen pen;
        LineEnd lineBegin, lineEnd;

        Pen &operator=( const Pen &_pen ) {
            pen  = _pen.pen;
            lineBegin = _pen.lineBegin;
            lineEnd = _pen.lineEnd;
            return *this;
        }
    };

    // the flags indicate what has changed
    enum PenConfigChange {
        LineBegin = 1,
        LineEnd = 2,
        Color = 4,
        Width = 8,
        Style = 16,
        All = LineBegin | LineEnd | Color | Width | Style
    };

    KoPenCmd( const TQString &_name, TQPtrList<KPrObject> &_objects, Pen _newPen,
            KPrDocument *_doc, KPrPage *_page, int _flags = All);
    ~KoPenCmd();

    virtual void execute();
    virtual void unexecute();

protected:
    void addObjects( const TQPtrList<KPrObject> &_objects );
    void applyPen( KPrObject *object, Pen *tmpPen );

    KPrDocument *doc;
    KPrPage *m_page;
    TQPtrList<Pen> oldPen;
    TQPtrList<KPrObject> objects;
    Pen newPen;
    int flags;
};

class KPrBrushCmd : public KNamedCommand
{
public:
    struct Brush {
        Brush() {}
        Brush( TQBrush _brush, TQColor _gColor1, TQColor _gColor2,
               BCType _gType, FillType _fillType, bool _unbalanced,
               int _xfactor, int _yfactor )
            : brush( _brush )
            , gColor1( _gColor1 )
            , gColor2( _gColor2 )
            , gType( _gType )
            , fillType( _fillType )
            , unbalanced( _unbalanced )
            , xfactor( _xfactor )
            , yfactor( _yfactor )
            {}

        TQBrush brush;
        TQColor gColor1;
        TQColor gColor2;
        BCType gType;
        FillType fillType;
        bool unbalanced;
        int xfactor, yfactor;

        Brush &operator=( const Brush &_brush ) {
            brush = _brush.brush;
            gColor1 = _brush.gColor1;
            gColor2 = _brush.gColor2;
            gType = _brush.gType;
            fillType = _brush.fillType;
            unbalanced = _brush.unbalanced;
            xfactor = _brush.xfactor;
            yfactor = _brush.yfactor;
            return *this;
        }
    };

    // the flags indicate what has changed
    enum BrushConfigChange {
        BrushColor = 1,
        BrushStyle = 2,
        BrushGradientSelect = 4,
        GradientColor1 = 8,
        GradientColor2 = 16,
        GradientType = 32,
        GradientBalanced = 64,
        GradientXFactor = 128,
        GradientYFactor = 256,
        All = BrushColor | BrushStyle | BrushGradientSelect | GradientColor1 | GradientColor2 | GradientType | GradientBalanced | GradientXFactor | GradientYFactor
    };

    KPrBrushCmd( const TQString &_name, TQPtrList<KPrObject> &_objects, Brush _newBrush,
              KPrDocument *_doc, KPrPage *_page, int _flags = All );
    ~KPrBrushCmd();

    virtual void execute();
    virtual void unexecute();

protected:
    void addObjects( const TQPtrList<KPrObject> &_objects );
    void applyBrush( KPr2DObject *object, Brush *tmpBrush );

    KPrDocument *doc;
    TQPtrList<Brush> oldBrush;
    TQPtrList<KPr2DObject> objects;
    Brush newBrush;
    KPrPage *m_page;
    int flags;
};

class KPrPgConfCmd : public KNamedCommand
{
public:
    KPrPgConfCmd( const TQString &_name, bool _manualSwitch, bool _infiniteLoop,
               bool _showEndOfPresentationSlide, bool _showPresentationDuration, TQPen _pen,
               TQValueList<bool> _selectedSlides, const TQString & _presentationName,
               bool _oldManualSwitch, bool _oldInfiniteLoop,
               bool _oldShowEndOfPresentationSlide, bool _oldShowPresentationDuration, TQPen _oldPen,
               TQValueList<bool> _oldSelectedSlides, const TQString & _oldPresentationName,
               KPrDocument *_doc );

    virtual void execute();
    virtual void unexecute();

protected:
    bool manualSwitch, oldManualSwitch;
    bool infiniteLoop, oldInfiniteLoop;
    bool showEndOfPresentationSlide, oldShowEndOfPresentationSlide;
    bool showPresentationDuration, oldShowPresentationDuration;
    TQPen pen, oldPen;
    TQValueList<bool> selectedSlides, oldSelectedSlides;
    TQString oldPresentationName, presentationName;

    KPrDocument *doc;
};

class KPrTransEffectCmd : public KCommand
{
public:
    struct PageEffectSettings {
        PageEffect pageEffect;
        EffectSpeed effectSpeed;
        bool soundEffect;
        TQString soundFileName;
        bool autoAdvance;
        int slideTime;
        void applyTo( KPrPage * );
    };
    // TODO also pass a flag to know which settings need to be applied (especially for "all pages")
    KPrTransEffectCmd( TQValueVector<PageEffectSettings> oldSettings,
                    PageEffectSettings newSettings,
                    KPrPage* page, KPrDocument* doc );

    virtual void execute();
    virtual void unexecute();
    virtual TQString name() const;

private:
    void exec( KPrPage* page );
    void unexec( KPrPage* page );

    TQValueVector<PageEffectSettings> m_oldSettings;
    PageEffectSettings m_newSettings;
    KPrPage *m_page;
    KPrDocument *m_doc;
};

class KPrPgLayoutCmd : public KNamedCommand
{
public:
    KPrPgLayoutCmd( const TQString &_name, KoPageLayout _layout, KoPageLayout _oldLayout,
                 KoUnit::Unit _oldUnit, KoUnit::Unit _unit,KPrDocument *_doc );

    virtual void execute();
    virtual void unexecute();

protected:
    KPrDocument *m_doc;
    KoPageLayout layout, oldLayout;
    KoUnit::Unit unit, oldUnit;
};

class KPrPieValueCmd : public KNamedCommand
{
public:
    struct PieValues
    {
        PieType pieType;
        int pieAngle, pieLength;
    };

    // the flags indicate what has changed
    enum PieConfigChange {
        Type = 1,
        Angle = 2,
        Length = 4,
        All = Type | Angle | Length
    };

    KPrPieValueCmd( const TQString &name, PieValues newValues,
                 TQPtrList<KPrObject> &objects, KPrDocument *doc,
                 KPrPage *page, int flags = All );
    KPrPieValueCmd( const TQString &_name, TQPtrList<PieValues> &_oldValues, PieValues _newValues,
                 TQPtrList<KPrObject> &_objects, KPrDocument *_doc, KPrPage *_page, int _flags = All );
    ~KPrPieValueCmd();

    virtual void execute();
    virtual void unexecute();

protected:
    void addObjects( const TQPtrList<KPrObject> &objects );

    KPrDocument *m_doc;
    KPrPage *m_page;
    TQPtrList<PieValues> m_oldValues;
    TQPtrList<KPrObject> m_objects;
    PieValues m_newValues;
    int m_flags;
};

class KPrPolygonSettingCmd : public KNamedCommand
{
public:
    struct PolygonSettings
    {
        bool checkConcavePolygon;
        int cornersValue;
        int sharpnessValue;
    };

    // the flags indicate what has changed
    enum PolygonConfigChange {
        ConcaveConvex = 1,
        Corners = 2,
        Sharpness = 4,
        All = ConcaveConvex | Corners | Sharpness
    };

    KPrPolygonSettingCmd( const TQString &name, PolygonSettings newSettings,
                          TQPtrList<KPrObject> &objects, KPrDocument *doc,
                          KPrPage *page, int flags = All );
    ~KPrPolygonSettingCmd();

    virtual void execute();
    virtual void unexecute();

protected:
    void addObjects( const TQPtrList<KPrObject> &objects );

    KPrDocument *m_doc;
    KPrPage *m_page;
    TQPtrList<PolygonSettings> m_oldSettings;
    TQPtrList<KPrObject> m_objects;
    PolygonSettings m_newSettings;
    int m_flags;
};

class KPrPictureSettingCmd : public KNamedCommand
{
public:
    struct PictureSettings
    {
        PictureMirrorType mirrorType;
        int depth;
        bool swapRGB;
        bool grayscal;
        int bright;
    };

    enum PictureConfigChange {
        MirrorType = 1,
        Depth = 2,
        SwapRGB = 4,
        Grayscal = 8,
        Bright = 16,
        All = MirrorType | Depth | SwapRGB | Grayscal | Bright
    };

    KPrPictureSettingCmd( const TQString &name, PictureSettings newSettings,
                       TQPtrList<KPrObject> &objects, KPrDocument *doc,
                       KPrPage *page, int flags = All );
    KPrPictureSettingCmd( const TQString &_name, TQPtrList<PictureSettings> &_oldSettings,
                       PictureSettings _newSettings, TQPtrList<KPrObject> &_objects,
                       KPrDocument *_doc, int flags = All );
    ~KPrPictureSettingCmd();

    virtual void execute();
    virtual void unexecute();

protected:
    void addObjects( const TQPtrList<KPrObject> &_objects );

    KPrDocument *m_doc;
    TQPtrList<PictureSettings> m_oldValues;
    TQPtrList<KPrObject> m_objects;
    PictureSettings m_newSettings;
    KPrPage *m_page;
    int m_flags;
};

class KPrImageEffectCmd : public KNamedCommand
{
public:
    struct ImageEffectSettings
    {
        ImageEffect effect;
        TQVariant param1;
        TQVariant param2;
        TQVariant param3;
    };

    KPrImageEffectCmd(const TQString &_name, TQPtrList<ImageEffectSettings> &_oldSettings,
                   ImageEffectSettings _newSettings, TQPtrList<KPrObject> &_objects,
                   KPrDocument *_doc );
    ~KPrImageEffectCmd();

    virtual void execute();
    virtual void unexecute();

protected:
    KPrDocument *doc;
    TQPtrList<ImageEffectSettings> oldSettings;
    TQPtrList<KPrObject> objects;
    ImageEffectSettings newSettings;
    KPrPage *m_page;

};

class KPrRectValueCmd : public KNamedCommand
{
public:
    struct RectValues
    {
        int xRnd, yRnd;
    };

    // the flags indicate what has changed
    enum RectangleConfigChange {
        XRnd = 1,
        YRnd = 2,
        All = XRnd | YRnd
    };

    KPrRectValueCmd( const TQString &_name, TQPtrList<RectValues> &_oldValues, RectValues _newValues,
                  TQPtrList<KPrObject> &_objects, KPrDocument *_doc, KPrPage *_page, int _flags = All );
    KPrRectValueCmd( const TQString &_name, TQPtrList<KPrObject> &_objects, RectValues _newValues,
                  KPrDocument *_doc, KPrPage *_page, int _flags = All );
    ~KPrRectValueCmd();

    virtual void execute();
    virtual void unexecute();

protected:
    void addObjects( const TQPtrList<KPrObject> &_objects );

    KPrDocument *m_doc;
    KPrPage *m_page;
    TQPtrList<RectValues> m_oldValues;
    TQPtrList<KPrObject> m_objects;
    RectValues m_newValues;
    int m_flags;
};

class KPrDeletePageCmd : public KNamedCommand
{
public:
    KPrDeletePageCmd( const TQString &name, int pageNum, KPrDocument *doc );
    ~KPrDeletePageCmd();

    virtual void execute();
    virtual void unexecute();

protected:
    KPrDocument *m_doc;
    KPrPage *m_page;
    int m_pageNum;
};

class KPrInsertPageCmd : public KNamedCommand
{
public:
    KPrInsertPageCmd( const TQString &name, int currentPageNum, InsertPos pos,
                      KPrPage *page, KPrDocument *doc );
    ~KPrInsertPageCmd();

    virtual void execute();
    virtual void unexecute();
protected:
    KPrDocument *m_doc;
    KPrPage *m_page;
    int m_currentPageNum;
    int m_insertPageNum;
};

class KPrMovePageCmd : public KNamedCommand
{
public:
    KPrMovePageCmd( const TQString &_name, int from,int to, KPrDocument *_doc );
    ~KPrMovePageCmd();

    virtual void execute();
    virtual void unexecute();
protected:
    KPrDocument *m_doc;
    int m_oldPosition;
    int m_newPosition;
};

/**
 * Command created when pasting oasis-formatted text
 */
class KPrOasisPasteTextCommand : public KoTextDocCommand
{
public:
    KPrOasisPasteTextCommand( KoTextDocument *d, int parag, int idx,
                             const TQByteArray &data );
    KoTextCursor *execute( KoTextCursor *c );
    KoTextCursor *unexecute( KoTextCursor *c );
protected:
    int m_parag;
    int m_idx;
    TQByteArray m_data;
    // filled in by execute(), for unexecute()
    int m_lastParag;
    int m_lastIndex;
    KoParagLayout* m_oldParagLayout;
};

/**
 * Command to change variable setting
 */
class KPrChangeStartingPageCommand : public KNamedCommand
{
public:
    KPrChangeStartingPageCommand( const TQString &name, KPrDocument *_doc,
                                  int _oldStartingPage, int _newStartingPage);
    ~KPrChangeStartingPageCommand(){}

    void execute();
    void unexecute();
protected:
    KPrDocument *m_doc;
    int oldStartingPage;
    int newStartingPage;
};

/**
 * Command to display link setting
 */
class KPrChangeVariableSettingsCommand : public KNamedCommand
{
public:
    enum VariableProperties { VS_DISPLAYLINK, VS_UNDERLINELINK, VS_DISPLAYCOMMENT, VS_DISPLAYFIELDCODE};
    KPrChangeVariableSettingsCommand( const TQString &name, KPrDocument *_doc, bool _oldValue,
                                      bool _newValue, VariableProperties _type);
    ~KPrChangeVariableSettingsCommand(){}

    void execute();
    void unexecute();
protected:
    void changeValue( bool b );
    KPrDocument *m_doc;
    VariableProperties type;
    bool m_bOldValue;
    bool m_bNewValue;
};

/**
 * Command to change title page name
 */
class KPrChangeTitlePageNameCommand : public KNamedCommand
{
public:
    KPrChangeTitlePageNameCommand( const TQString &name, KPrDocument *_doc, const TQString &_oldPageName,
                                   const TQString &_newPageName, KPrPage *_page);
    ~KPrChangeTitlePageNameCommand(){}

    void execute();
    void unexecute();
protected:
    KPrDocument *m_doc;
    TQString oldPageName;
    TQString newPageName;
    KPrPage *m_page;
};

class KPrChangeCustomVariableValue : public KNamedCommand
{
public:
    KPrChangeCustomVariableValue( const TQString &name, KPrDocument *_doc,const TQString & _oldValue,
                                  const TQString & _newValue, KoCustomVariable *var);

    void execute();
    void unexecute();
protected:
    KPrDocument *m_doc;
    TQString newValue;
    TQString oldValue;
    KoCustomVariable *m_var;
};

class KPrChangeLinkVariable : public KNamedCommand
{
public:
    KPrChangeLinkVariable( const TQString &name, KPrDocument *_doc,const TQString & _oldHref,
                           const TQString & _newHref, const TQString & _oldLink,const TQString &_newLink, KoLinkVariable *var);
    ~KPrChangeLinkVariable(){};
    void execute();
    void unexecute();
protected:
    KPrDocument *m_doc;
    TQString oldHref;
    TQString newHref;
    TQString oldLink;
    TQString newLink;
    KoLinkVariable *m_var;
};


class KPrNameObjectCommand : public KNamedCommand
{
public:
    KPrNameObjectCommand( const TQString &_name, const TQString &_objectName, KPrObject *_obj, KPrDocument *_doc );
    ~KPrNameObjectCommand();
    void execute();
    void unexecute();
protected:
    TQString oldObjectName, newObjectName;
    KPrObject *object;
    KPrDocument *doc;
    KPrPage *m_page;
};

class KPrDisplayObjectFromMasterPage : public KNamedCommand
{
public:
    KPrDisplayObjectFromMasterPage( const TQString &name, KPrDocument *_doc, KPrPage *_page, bool _newValue);
    ~KPrDisplayObjectFromMasterPage(){};
    void execute();
    void unexecute();
protected:
    KPrDocument *m_doc;
    KPrPage *m_page;
    bool newValue;
};

class KPrDisplayBackgroundPage : public KNamedCommand
{
public:
    KPrDisplayBackgroundPage( const TQString &name, KPrDocument *_doc, KPrPage *_page, bool _newValue);
    ~KPrDisplayBackgroundPage(){};
    void execute();
    void unexecute();
protected:
    KPrDocument *m_doc;
    KPrPage *m_page;
    bool newValue;
};

class KPrHideShowHeaderFooter : public KNamedCommand
{
public:
    KPrHideShowHeaderFooter( const TQString &name, KPrDocument *_doc, KPrPage *_page, bool _newValue,KPrTextObject *_textObject);
    ~KPrHideShowHeaderFooter(){};
    void execute();
    void unexecute();
protected:
    KPrDocument *m_doc;
    KPrPage *m_page;
    KPrTextObject *m_textObject;
    bool newValue;
};

class KPrFlipObjectCommand : public KNamedCommand
{
public:
    KPrFlipObjectCommand( const TQString &name, KPrDocument *_doc, bool _horizontal ,
                          TQPtrList<KPrObject> &_objects );
    ~KPrFlipObjectCommand();
    void execute();
    void unexecute();
protected:
    void flipObjects();
    KPrDocument *m_doc;
    TQPtrList<KPrObject> objects;
    bool horizontal;
    KPrPage *m_page;
};

class KPrGeometryPropertiesCommand : public KNamedCommand
{
public:
    enum KgpType { ProtectSize, KeepRatio};
    KPrGeometryPropertiesCommand( const TQString &name, TQPtrList<KPrObject> &objects,
                                  bool newValue, KgpType type, KPrDocument *_doc );
    KPrGeometryPropertiesCommand( const TQString &name, TQValueList<bool> &lst, TQPtrList<KPrObject> &objects,
                                  bool newValue, KgpType type, KPrDocument *_doc );
    ~KPrGeometryPropertiesCommand();

    virtual void execute();
    virtual void unexecute();

protected:
    TQValueList<bool> m_oldValue;
    TQPtrList<KPrObject> m_objects;
    bool m_newValue;
    KgpType m_type;
    KPrDocument *m_doc;
};

class KPrProtectContentCommand : public KNamedCommand
{
public:
    KPrProtectContentCommand( const TQString &name, TQPtrList<KPrObject> &objects,
                              bool protectContent, KPrDocument *doc );
    KPrProtectContentCommand( const TQString &name, bool protectContent,
                              KPrTextObject *obj, KPrDocument *doc );

    ~KPrProtectContentCommand();
    virtual void execute();
    virtual void unexecute();

protected:
    void addObjects( const TQPtrList<KPrObject> &objects );

    TQPtrList<KPrTextObject> m_objects;
    bool m_protectContent;
    TQValueList<bool> m_oldValues;
    KPrDocument *m_doc;
};

class KPrCloseObjectCommand : public KNamedCommand
{
public:
    KPrCloseObjectCommand( const TQString &name, TQPtrList<KPrObject> objects, KPrDocument *doc );

    ~KPrCloseObjectCommand();
    virtual void execute();
    virtual void unexecute();

protected:
    void closeObject(bool close);

    TQPtrList<KPrObject> m_openObjects;
    TQPtrList<KPrObject> m_closedObjects;
    KPrDocument * m_doc;
    KPrPage *m_page;
};

struct MarginsStruct {
    MarginsStruct() {}
    MarginsStruct( KPrTextObject *obj );
    MarginsStruct( double _left, double top, double right, double bottom );
    double topMargin;
    double bottomMargin;
    double leftMargin;
    double rightMargin;
};

class KPrChangeMarginCommand : public KNamedCommand
{
public:
    KPrChangeMarginCommand( const TQString &name, TQPtrList<KPrObject> &objects, MarginsStruct newMargins,
                            KPrDocument *doc, KPrPage *page );
    ~KPrChangeMarginCommand();

    virtual void execute();
    virtual void unexecute();
protected:
    void addObjects( const TQPtrList<KPrObject> &objects );
    TQPtrList<MarginsStruct> m_oldMargins;
    TQPtrList<KPrTextObject> m_objects;
    MarginsStruct m_newMargins;
    KPrPage *m_page;
    KPrDocument *m_doc;
};


class KPrChangeVerticalAlignmentCommand : public KNamedCommand
{
public:
    KPrChangeVerticalAlignmentCommand( const TQString &name, KPrTextObject *_obj, VerticalAlignmentType _oldAlign,
                                       VerticalAlignmentType _newAlign, KPrDocument *_doc);
    ~KPrChangeVerticalAlignmentCommand() {}

    virtual void execute();
    virtual void unexecute();
protected:
    KPrTextObject *m_obj;
    VerticalAlignmentType m_oldAlign;
    VerticalAlignmentType m_newAlign;
    KPrPage *m_page;
    KPrDocument *m_doc;
};


class KPrChangeTabStopValueCommand : public KNamedCommand
{
public:
    KPrChangeTabStopValueCommand( const TQString &name, double _oldValue, double _newValue, KPrDocument *_doc);

    ~KPrChangeTabStopValueCommand() {}

    virtual void execute();
    virtual void unexecute();
protected:
    KPrDocument *m_doc;
    double m_oldValue;
    double m_newValue;
};

class KPrChangeVariableNoteText : public KNamedCommand
{
 public:
    KPrChangeVariableNoteText( const TQString &name, KPrDocument *_doc, const TQString &_oldValue,const TQString &_newValue, KoNoteVariable *var);
    ~KPrChangeVariableNoteText();
    void execute();
    void unexecute();
 protected:
    KPrDocument *m_doc;
    TQString newValue;
    TQString oldValue;
    KoNoteVariable *m_var;
};


#endif
