/***************************************************************************
                          radiostation-listview.cpp  -  description
                             -------------------
    begin                : Mi Feb 3 2004
    copyright            : (C) 2003 by Martin Witte
    email                : witte@kawo1.rwth-aachen.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "include/radiostation-listview.h"
#include "include/stationlist.h"
#include "include/radiostation.h"
#include "include/station-drag-object.h"

#include <tdelocale.h>
#include <tqfile.h>
#include <tqimage.h>
#include <tqpixmap.h>

#include <tdeconfig.h>

RadioStationListView::RadioStationListView(TQWidget *parent, const char *name)
  : TDEListView(parent, name)
{
    addColumn(i18n("No."));
    addColumn(i18n("Icon"));
    addColumn(i18n("Station"));
    addColumn(i18n("Description"));
    setAllColumnsShowFocus(true);
    setSorting(-1);

    TQObject::connect(this, TQT_SIGNAL(spacePressed(TQListViewItem*)),
                     this, TQT_SLOT(slotStationActivation(TQListViewItem* )));
    TQObject::connect(this, TQT_SIGNAL(returnPressed(TQListViewItem*)),
                     this, TQT_SLOT(slotStationActivation(TQListViewItem* )));
    TQObject::connect(this, TQT_SIGNAL(doubleClicked(TQListViewItem*)),
                     this, TQT_SLOT(slotStationActivation(TQListViewItem *)));
    TQObject::connect(this, TQT_SIGNAL(currentChanged(TQListViewItem*)),
                     this, TQT_SLOT(slotCurrentStationChanged(TQListViewItem *)));

    setAcceptDrops(true);
}


RadioStationListView::~RadioStationListView()
{
}


TQListViewItem *RadioStationListView::getItemForIndex(int idx) const
{
    TQListViewItem *item = NULL;

    if (idx >= 0 && idx < childCount()) {
        item = firstChild();
        int i = 0;
        while (item && i < idx) {
            item = item->nextSibling();
            ++i;
        }
    }
    return item;
}


int RadioStationListView::getIndexForItem(TQListViewItem *queryItem) const
{
    int idx = -1;

    if (queryItem) {
        TQListViewItem *item = firstChild();
        ++idx;
        while (item && item != queryItem) {
            item = item->nextSibling();
            ++idx;
        }
        if (!item)
            idx = -1;
    }

    return idx;
}


void RadioStationListView::setStation(int idx, const RadioStation &s, int nr)
{
    TQListViewItem *item = getItemForIndex(idx);

    if (idx < 0) {
        item = new TQListViewItem(this, firstChild());
        firstChild()->moveItem(item);
        m_StationIDs.prepend(s.stationID());
        idx = 0;
    } else if (idx >= childCount()) {
        item = new TQListViewItem(this, lastChild());
        m_StationIDs.append(s.stationID());
        idx = childCount() - 1;
    }

    if (item) {
        item->setDragEnabled(true);
        item->setDropEnabled(true);

        item->setText(0, TQString::number(nr > 0 ? nr : idx+1));
        item->setText(2, s.name());
        item->setText(3, s.description());

        m_StationIDs[idx] = s.stationID();

        TQImage  img(s.iconName());
        if (!img.isNull()) {
            int   h = img.height();
            float f = 0.9 * (float)(item->height()) / (h ? (float)h : 1.0);
            item->setPixmap(1, img.smoothScale((int)(img.width()*f), (int)(h * f)));
        } else {
            item->setPixmap(1, TQPixmap());
        }
    }
}


void RadioStationListView::appendStation(const RadioStation &st, int nr)
{
    setStation(childCount(), st, nr);
}


void RadioStationListView::setStations(const StationList &stations)
{
    clear();
    for (RawStationList::Iterator it(stations.all()); it.current(); ++it) {
        setStation(childCount(), *it.current());
    }
}


void RadioStationListView::removeStation(int idx)
{
    TQListViewItem *item = getItemForIndex(idx);
    if (item) {
        delete item;
        m_StationIDs.remove(m_StationIDs.at(idx));
    }
}

void RadioStationListView::takeItem(TQListViewItem *item, int idx)
{
    TQListView::takeItem(item);
    m_StationIDs.remove(m_StationIDs.at(idx));
}

void RadioStationListView::insertItem(TQListViewItem *item, const TQString &stationid, int idx_to)
{
    TQListView::insertItem(item);
    m_StationIDs.insert(m_StationIDs.at(idx_to), stationid);
}

void RadioStationListView::setCurrentStation(int idx)
{
    TQListViewItem *item = getItemForIndex(idx);
    if (item) {
        clearSelection();
        setSelected(item, true);
        setCurrentItem(item);
    }
}


int RadioStationListView::currentStationIndex() const
{
    return getIndexForItem(currentItem());
}


void RadioStationListView::slotStationActivation(TQListViewItem *item)
{
    emit sigStationActivated(getIndexForItem(item));
}


void RadioStationListView::slotCurrentStationChanged(TQListViewItem *item)
{
    emit sigCurrentStationChanged(getIndexForItem(item));
}


void RadioStationListView::saveState (TDEConfig *cfg) const
{
    if (!cfg)
        return;
    for (int i = 0; i < 4; ++i)
        cfg->writeEntry(TQString(name()) + "_radiostation_listview_col_" + TQString::number(i), columnWidth(i));
}


void RadioStationListView::restoreState (TDEConfig *cfg)
{
    if (!cfg)
        return;
    for (int i = 0; i < 4; ++i)
        setColumnWidth(i, cfg->readNumEntry(TQString(name()) + "_radiostation_listview_col_" + TQString::number(i), -1));
}


TQDragObject *RadioStationListView::dragObject()
{
    TQStringList list;
    TQListViewItem *item = firstChild();
    for (int idx = 0; item; ++idx, item = item->nextSibling()) {
        if (item->isSelected()) {
            list.append(m_StationIDs[idx]);
        }
    }
    return new StationDragObject(list, this);
}

void RadioStationListView::dragEnterEvent(TQDragEnterEvent* event)
{
    event->accept(StationDragObject::canDecode(event));
}

void RadioStationListView::contentsDragEnterEvent(TQDragEnterEvent* event)
{
    bool a = StationDragObject::canDecode(event);
    if (a)
        IErrorLogClient::staticLogDebug(i18n("contentsDragEnterEvent accepted"));
    else
        IErrorLogClient::staticLogDebug(i18n("contentsDragEnterEvent rejected"));
    event->accept(a);
}

void RadioStationListView::dropEvent(TQDropEvent* event)
{
    TQStringList list;

    if ( StationDragObject::decode(event, list) ) {
        emit sigStationsReceived(list);
    }
}

void RadioStationListView::contentsDropEvent(TQDropEvent* event)
{
    dropEvent(event);
}

void RadioStationListView::contentsDragMoveEvent(TQDragMoveEvent* event)
{
    event->accept();
}

#include "radiostation-listview.moc"
