#!/usr/bin/env python

# TQt tutorial 11.

import sys
import math
from python_tqt import qt


class LCDRange(qt.TQVBox):
    def __init__(self, parent=None, name=None):
        qt.TQVBox.__init__(self, parent, name)

        lcd = qt.TQLCDNumber(2, self, "lcd")
        self.slider = qt.TQSlider(qt.TQt.Horizontal, self, "slider")
        self.slider.setRange(0, 99)
        self.slider.setValue(0)
        self.connect(self.slider, qt.SIGNAL("valueChanged(int)"), lcd, qt.SLOT("display(int)"))
        self.connect(self.slider, qt.SIGNAL("valueChanged(int)"), self, qt.PYSIGNAL("valueChanged(int)"))

        self.setFocusProxy(self.slider)

    def value(self):
        return self.slider.value()

    def setValue(self, value):
        self.slider.setValue(value)

    def setRange(self, minVal, maxVal):
        if minVal < 0 or maxVal > 99 or minVal > maxVal:
            raise ValueError("LCDRange.setRange(): invalid range")
        self.slider.setRange(minVal, maxVal)


class CannonField(qt.TQWidget):
    def __init__(self, parent=None, name=None):
        qt.TQWidget.__init__(self, parent, name)

        self.ang = 45
        self.f = 0
        self.timerCount = 0

        self.autoShootTimer = qt.TQTimer(self, "movement handler")
        self.connect(self.autoShootTimer, qt.SIGNAL("timeout()"), self.moveShot)

        self.shoot_ang = 0
        self.shoot_f = 0

        self.setPalette(qt.TQPalette(qt.TQColor(250, 250, 200)))

        self.barrelRect = qt.TQRect(33, -4, 15, 8)

    def angle(self):
        return self.ang

    def setAngle(self, degrees):
        if degrees < 5:
            degrees = 5
        if degrees > 70:
            degrees = 70
        if self.ang == degrees:
            return
        self.ang = degrees
        self.repaint(self.cannonRect(), 0)
        self.emit(qt.PYSIGNAL("angleChanged(int)"), (self.ang, ))

    def force(self):
        return self.f

    def setForce(self, newton):
        if newton < 0:
            newton = 0
        if self.f == newton:
            return
        self.f = newton
        self.emit(qt.PYSIGNAL("forceChanged(int)"), (self.f, ))

    def shoot(self):
        if self.autoShootTimer.isActive():
            return

        self.timerCount = 0
        self.shoot_ang = self.ang
        self.shoot_f = self.f
        self.autoShootTimer.start(50)

    def moveShot(self):
        r = qt.TQRegion(self.shotRect())
        self.timerCount = self.timerCount + 1

        shotR = self.shotRect()

        if shotR.x() > self.width() or shotR.y() > self.height():
            self.autoShootTimer.stop()
        else:
            r = r.unite(qt.TQRegion(shotR))

        self.repaint(r)

    def paintEvent(self, ev):
        updateR = ev.rect()
        p = qt.TQPainter(self)

        if updateR.intersects(self.cannonRect()):
            self.paintCannon(p)

        if self.autoShootTimer.isActive() and updateR.intersects(self.shotRect()):
            self.paintShot(p)

    def paintShot(self, p):
        p.setBrush(qt.TQt.black)
        p.setPen(qt.TQt.NoPen)
        p.drawRect(self.shotRect())

    def paintCannon(self, p):
        cr = self.cannonRect()
        pix = qt.TQPixmap(cr.size())
        pix.fill(self, cr.topLeft())

        tmp = qt.TQPainter(pix)
        tmp.setBrush(qt.TQt.blue)
        tmp.setPen(qt.TQt.NoPen)

        tmp.translate(0, pix.height() - 1)
        tmp.drawPie(qt.TQRect(-35, -35, 70, 70), 0, 90 * 16)
        tmp.rotate(-self.ang)
        tmp.drawRect(self.barrelRect)
        tmp.end()

        p.drawPixmap(cr.topLeft(), pix)

    def cannonRect(self):
        r = qt.TQRect(0, 0, 50, 50)
        r.moveBottomLeft(self.rect().bottomLeft())
        return r

    def shotRect(self):
        gravity = 4.0

        time = self.timerCount / 4.0
        velocity = self.shoot_f
        radians = self.shoot_ang * 3.14159265 / 180

        velx = velocity * math.cos(radians)
        vely = velocity * math.sin(radians)
        x0 = (self.barrelRect.right() + 5) * math.cos(radians)
        y0 = (self.barrelRect.right() + 5) * math.sin(radians)
        x = x0 + velx * time
        y = y0 + vely * time - 0.5 * gravity * time * time

        r = qt.TQRect(0, 0, 6, 6)
        r.moveCenter(qt.TQPoint(int(x), int(self.height() - 1 - y)))
        return r

    def sizePolicy(self):
        return qt.TQSizePolicy(qt.TQSizePolicy.Expanding, qt.TQSizePolicy.Expanding)


class MyWidget(qt.TQWidget):
    def __init__(self, parent=None, name=None):
        qt.TQWidget.__init__(self, parent, name)

        quit = qt.TQPushButton("&Quit", self, "quit")
        quit.setFont(qt.TQFont("Times", 18, qt.TQFont.Bold))
        self.connect(quit, qt.SIGNAL("clicked()"), qt.tqApp, qt.SLOT("quit()"))

        self.angle = LCDRange(self, "angle")
        self.angle.setRange(5, 70)

        self.force = LCDRange(self, "force")
        self.force.setRange(10, 50)

        self.cannonField = CannonField(self, "cannonField")

        self.connect(self.angle, qt.PYSIGNAL("valueChanged(int)"), self.cannonField.setAngle)
        self.connect(self.cannonField, qt.PYSIGNAL("angleChanged(int)"), self.angle.setValue)

        self.connect(self.force, qt.PYSIGNAL("valueChanged(int)"), self.cannonField.setForce)
        self.connect(self.cannonField, qt.PYSIGNAL("forceChanged(int)"), self.force.setValue)

        shoot = qt.TQPushButton("&Shoot", self, "shoot")
        shoot.setFont(qt.TQFont("Times", 18, qt.TQFont.Bold))
        self.connect(shoot, qt.SIGNAL("clicked()"), self.cannonField.shoot)

        grid = qt.TQGridLayout(self, 2, 2, 10)

        grid.addWidget(quit, 0, 0)
        grid.addWidget(self.cannonField, 1, 1)
        grid.setColStretch(1, 10)

        leftBox = qt.TQVBoxLayout()
        grid.addLayout(leftBox, 1, 0)
        leftBox.addWidget(self.angle)
        leftBox.addWidget(self.force)

        topBox = qt.TQHBoxLayout()
        grid.addLayout(topBox, 0, 1)
        topBox.addWidget(shoot)
        topBox.addStretch(1)

        self.angle.setValue(60)
        self.force.setValue(25)
        self.angle.setFocus()


qt.TQApplication.setColorSpec(qt.TQApplication.CustomColor)
a = qt.TQApplication(sys.argv)

w = MyWidget()
w.setGeometry(100, 100, 500, 355)
a.setMainWidget(w)
w.show()
sys.exit(a.exec_loop())
