/*
    This file is part of Akregator.

    Copyright (C) 2004 Sashmit Bhaduri <smt@vfemail.net>
                  2005 Frank Osterfeld <frank.osterfeld at kdemail.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

    As a special exception, permission is given to link this program
    with any edition of TQt, and distribute the resulting executable,
    without including the source code for TQt in the source distribution.
*/

#ifndef ARTICLEVIEWER_H
#define ARTICLEVIEWER_H

#include <tdelocale.h>

#include <tqcolor.h>
#include <tqfont.h>

#include "article.h"
#include "articlefilter.h"
#include "viewer.h"

class TQKeyEvent;

namespace Akregator
{
    class Feed;
    class Folder;
    class TreeNode;
    
    /** This HTML viewer is used to display articles. 
    Use the high-level interface provided by the public slots whereever possible (and extend them when necessary instead of using low-level methods).*/
     class ArticleViewer : public Viewer
    {
        Q_OBJECT
  
        public:
            /** Constructor */
            ArticleViewer(TQWidget* parent, const char* name);
            virtual ~ArticleViewer();
            
            virtual bool openURL(const KURL &url);

            /** Repaints the view. */
            void reload();

            void displayAboutPage();
            
            public slots:
	    
            // Commandment: We are your interfaces.
            // You shall not use strange interfaces before us.
	    
            /** Show single article (normal view) 
                @param article the article to render */
            void slotShowArticle(const Article& article);
            
            /** Shows the articles of the tree node @c node (combined view). Changes in the node will update the view automatically. 
            @param node The node to observe */
            void slotShowNode(TreeNode* node);
            
            /** Set filters @c textFilter and @c statusFilter which will be used if the viewer is in combined view mode 
            @param textFilter text filter 
            @param statusFilter status filter */    
            void slotSetFilter(const Akregator::Filters::ArticleMatcher& textFilter, const Akregator::Filters::ArticleMatcher& statusFilter);
            
            /** Update view if combined view mode is set. Has to be called when the displayed node gets modified. */ 
            void slotUpdateCombinedView();
            
            /** Clears the canvas and disconnects from the currently observed node (if in combined view mode). */
            void slotClear();
	    
            void slotShowSummary(TreeNode *node);

            virtual void slotPaletteOrFontChanged();
            
        protected slots:
                    
            void slotArticlesUpdated(TreeNode* node, const TQValueList<Article>& list);
            void slotArticlesAdded(TreeNode* node, const TQValueList<Article>& list);
            void slotArticlesRemoved(TreeNode* node, const TQValueList<Article>& list);
            
        protected:
            
            virtual void keyPressEvent(TQKeyEvent* e);
            virtual void urlSelected (const TQString &url, int button, int state, const TQString &_target, KParts::URLArgs args);

        public:		// compat with KDE-3.x assertions, remove for KDE 4
	// private:

            friend class ShowNodeSummaryVisitor;
            class ShowSummaryVisitor;
            ShowSummaryVisitor* m_showSummaryVisitor;
            
            /** renders @c body. Use this method whereever possible.
            @param body html to render, without header and footer */
            void renderContent(const TQString& body);
            
            /** Takes an article and renders it as HTML with settings for normal view and widescreen view
            @param feed article's feed (used for feed icon atm) -- article.feed() would do. better use a (No)Icon flag. -fo
            @param article The article to render
            @return the rendered article as HTML */
            TQString formatArticleNormalMode(Feed* feed, const Article& article);
            
            /** Takes an article and renders it as HTML with settings for combined view
            @param feed article's feed (used for feed icon atm) -- article.feed() would do. better use a (No)Icon flag. -fo
            @param article The article to render
            @return the rendered article as HTML */
            TQString formatArticleCombinedMode(Feed* feed, const Article& article);
            
            /** Resets the canvas and adds writes the HTML header to it.
             */
            void beginWriting();

            /** Finishes writing to the canvas and completes the HTML (by adding closing tags) */
            void endWriting();
            
            /** generates the CSS used for rendering in single article mode (normal and wide screen view) */
            void generateNormalModeCSS();
            /** generates the CSS for combined view mode */
            void generateCombinedModeCSS();
            void connectToNode(TreeNode* node);
            void disconnectFromNode(TreeNode* node);
                                  
            TQString m_normalModeCSS;
            TQString m_combinedModeCSS;
            TQString m_htmlFooter;
            TQString m_currentText;
            KURL m_imageDir;
            TreeNode* m_node;
            Article m_article;
            KURL m_link;
            Akregator::Filters::ArticleMatcher m_textFilter; 
            Akregator::Filters::ArticleMatcher m_statusFilter;
            enum ViewMode { NormalView, CombinedView, SummaryView };
            ViewMode m_viewMode;
   };
}

#endif // ARTICLEVIEWER_H
