/*
 * This file is part of the PolKit-tqt project
 * Copyright (C) 2009 Radek Novacek <rnovacek@redhat.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB. If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <polkit/polkit.h>

#include "polkit-tqt-temporaryauthorization.h"
#include "polkit-tqt-authority.h"

#include <tqdatetime.h>
#include <tqshared.h>
#include <tqstring.h>



namespace PolkitTQt
{

//--------------------------------------
// TemporaryAuthorization::Data
//--------------------------------------

class TemporaryAuthorization::Data : public TQShared
{
  public:
    Data() : id(TQString::null) , actionId(TQString::null), subject(),
            timeObtained(TQDateTime()), timeExpires(TQDateTime())
    {
    }

    Data(const Data& other) : id(other.id), actionId(other.actionId), subject(other.subject),
            timeObtained(other.timeObtained), timeExpires(other.timeExpires)
    {
    }

    ~Data()
    {
    }

    TQString id;
    TQString actionId;
    Subject subject;
    TQDateTime timeObtained;
    TQDateTime timeExpires;
};

//--------------------------------------
// TemporaryAuthorization
//--------------------------------------

TemporaryAuthorization::TemporaryAuthorization() : d(new Data)
{
}

TemporaryAuthorization::TemporaryAuthorization(PolkitTemporaryAuthorization *pkTemporaryAuthorization) : d(new Data)
{
  d->id = TQString::fromUtf8(polkit_temporary_authorization_get_id(pkTemporaryAuthorization));
  d->actionId = TQString::fromUtf8(polkit_temporary_authorization_get_action_id(pkTemporaryAuthorization));
  d->subject = Subject::fromString(polkit_subject_to_string(polkit_temporary_authorization_get_subject(pkTemporaryAuthorization)));
  d->timeObtained = TQDateTime();
  d->timeObtained.setTime_t(polkit_temporary_authorization_get_time_obtained(pkTemporaryAuthorization));
  d->timeExpires = TQDateTime();
  d->timeExpires.setTime_t(polkit_temporary_authorization_get_time_expires(pkTemporaryAuthorization));
}

TemporaryAuthorization::TemporaryAuthorization(const TemporaryAuthorization &other) : d(other.d)
{
  d->ref();
}

TemporaryAuthorization& TemporaryAuthorization::operator=(const PolkitTQt::TemporaryAuthorization& other)
{
  if (d != other.d)
  {
    if (d->deref())
    {
      delete d;
    }
    d = other.d;
    d->ref();
  }
  return *this;
}

TemporaryAuthorization::~TemporaryAuthorization()
{
  if (d->deref())
  {
    delete d;
  }
}

TQString TemporaryAuthorization::id() const
{
  return d->id;
}

TQString TemporaryAuthorization::actionId() const
{
  return d->actionId;
}

Subject TemporaryAuthorization::subject() const
{
  return d->subject;
}

TQDateTime TemporaryAuthorization::obtainedAt() const
{
  return d->timeObtained;
}

TQDateTime TemporaryAuthorization::expirationTime() const
{
  return d->timeExpires;
}

bool TemporaryAuthorization::revoke()
{
  Authority::instance()->revokeTemporaryAuthorization(id());
  return true;
}

}
