/*
 *  Copyright (c) 2004 Boudewijn Rempt <boud@valdyas.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#include <tqapplication.h>
#include <tqclipboard.h>
#include <tqobject.h>
#include <tqimage.h>
#include <tqmessagebox.h>
#include <tqbuffer.h>
#include <tdemultipledrag.h>
#include <tdelocale.h>

#include "kdebug.h"

#include "KoStore.h"
#include "KoStoreDrag.h"

#include "kis_types.h"
#include "kis_paint_device.h"
#include "kis_config.h"
#include "kis_colorspace_factory_registry.h"
#include "kis_factory.h"
#include <kis_meta_registry.h>
#include "kis_clipboard.h"

KisClipboard *KisClipboard::m_singleton = 0;

KisClipboard::KisClipboard()
{
    Q_ASSERT(KisClipboard::m_singleton == 0);
    KisClipboard::m_singleton = this;

    m_pushedClipboard = false;
    m_hasClip = false;
    m_clip = 0;

    // Check that we don't already have a clip ready
    clipboardDataChanged();

    // Make sure we are notified when clipboard changes
    connect( TQApplication::clipboard(), TQ_SIGNAL( dataChanged() ),
         this, TQ_SLOT( clipboardDataChanged() ) );
}

KisClipboard::~KisClipboard()
{
}

KisClipboard* KisClipboard::instance()
{
    if(KisClipboard::m_singleton == 0)
    {
        KisClipboard::m_singleton = new KisClipboard();
        TQ_CHECK_PTR(KisClipboard::m_singleton);
    }
    return KisClipboard::m_singleton;
}

void KisClipboard::setClip(KisPaintDeviceSP selection)
{
    m_clip = selection;

    if (!selection)
        return;

    m_hasClip = true;

    // We'll create a store (ZIP format) in memory
    TQBuffer buffer;
    TQCString mimeType("application/x-chalk-selection");
    KoStore* store = KoStore::createStore( &buffer, KoStore::Write, mimeType );
    Q_ASSERT( store );
    Q_ASSERT( !store->bad() );

    // Layer data
    if (store->open("layerdata")) {
        if (!selection->write(store)) {
            selection->disconnect();
            store->close();
            return;
        }
        store->close();
    }

    // ColorSpace id of layer data
    if (store->open("colorspace")) {
        TQString csName = selection->colorSpace()->id().id();
        store->write(csName.ascii(), strlen(csName.ascii()));
        store->close();
    }

    if (selection->colorSpace()->getProfile()) {
        KisAnnotationSP annotation = selection->colorSpace()->getProfile()->annotation();
         if (annotation) {
            // save layer profile
             if (store->open("profile.icc")) {
                store->write(annotation->annotation());
                store->close();
            }
        }
    }

    delete store;

    // We also create a TQImage so we can interchange with other applications
    TQImage qimg;
    KisConfig cfg;
    TQString monitorProfileName = cfg.monitorProfile();
    KisProfile *  monitorProfile = KisMetaRegistry::instance()->csRegistry()->getProfileByName(monitorProfileName);
    qimg = selection->convertToTQImage(monitorProfile);

    TQImageDrag *qimgDrag = new TQImageDrag(qimg);
    KMultipleDrag *multiDrag = new KMultipleDrag();
    if ( !qimg.isNull() )
        multiDrag->addDragObject( qimgDrag );
    KoStoreDrag* storeDrag = new KoStoreDrag( mimeType, 0 );
    storeDrag->setEncodedData( buffer.buffer() );
    multiDrag->addDragObject( storeDrag );


    TQClipboard *cb = TQApplication::clipboard();
    cb->setData(multiDrag);
    m_pushedClipboard = true;
}

KisPaintDeviceSP KisClipboard::clip()
{
    TQClipboard *cb = TQApplication::clipboard();
    TQCString mimeType("application/x-chalk-selection");
    TQMimeSource *cbData = cb->data();

    if(cbData && cbData->provides(mimeType))
    {
        TQBuffer buffer(cbData->encodedData(mimeType));
        KoStore* store = KoStore::createStore( &buffer, KoStore::Read, mimeType );
        KisProfile *profile=0;

        if (store->hasFile("profile.icc")) {
            TQByteArray data;
            store->open("profile.icc");
            data = store->read(store->size());
            store->close();
           profile = new KisProfile(data);
        }

        TQString csName;
        // ColorSpace id of layer data
        if (store->hasFile("colorspace")) {
            store->open("colorspace");
            csName = TQString(store->read(store->size()));
            store->close();
        }

        KisColorSpace *cs = KisMetaRegistry::instance()->csRegistry()->getColorSpace(KisID(csName, ""), profile);

        m_clip = new KisPaintDevice(cs, "clip");

        if (store->hasFile("layerdata")) {
            store->open("layerdata");
            m_clip->read(store);
            store->close();
        }
        delete store;
    }
    else
    {
        TQImage qimg = cb->image();

        if (qimg.isNull())
            return 0;

        KisConfig cfg;

        TQ_UINT32 behaviour = cfg.pasteBehaviour();

        if(behaviour==2)
        {
            // Ask user each time
            behaviour = TQMessageBox::question(0,i18n("Pasting data from simple source"),i18n("The image data you are trying to paste has no colour profile information.\n\nOn the web and in simple applications the data are supposed to be in sRGB color format.\nImporting as web will show it as it is supposed to look.\nMost monitors are not perfect though so if you made the image yourself\nyou might want to import it as it looked on you monitor.\n\nHow do you want to interpret these data?"),i18n("As &Web"),i18n("As on &Monitor"));
        }

        KisColorSpace * cs;
        TQString profileName("");
        if(behaviour==1)
            profileName = cfg.monitorProfile();

        cs = KisMetaRegistry::instance()->csRegistry() ->getColorSpace(KisID("RGBA",""), profileName);
        m_clip = new KisPaintDevice(cs, "from paste");
        TQ_CHECK_PTR(m_clip);
        m_clip->convertFromTQImage(qimg, profileName);
    }

    return m_clip;
}

void KisClipboard::clipboardDataChanged()
{
    if (!m_pushedClipboard) {
        m_hasClip = false;
        TQClipboard *cb = TQApplication::clipboard();
        TQImage qimg = cb->image();
        TQMimeSource *cbData = cb->data();
        TQCString mimeType("application/x-chalk-selection");

        if(cbData && cbData->provides(mimeType))
            m_hasClip = true;

        if (!qimg.isNull())
            m_hasClip = true;
    }

    m_pushedClipboard = false;
}


bool KisClipboard::hasClip()
{
    return m_hasClip;
}

TQSize KisClipboard::clipSize()
{

    TQClipboard *cb = TQApplication::clipboard();
    TQCString mimeType("application/x-chalk-selection");
    TQMimeSource *cbData = cb->data();

    KisPaintDeviceSP clip;
    
    if(cbData && cbData->provides(mimeType)) {
        
        TQBuffer buffer(cbData->encodedData(mimeType));
        KoStore* store = KoStore::createStore( &buffer, KoStore::Read, mimeType );
        KisProfile *profile=0;

        if (store->hasFile("profile.icc")) {
            TQByteArray data;
            store->open("profile.icc");
            data = store->read(store->size());
            store->close();
            profile = new KisProfile(data);
        }

        TQString csName;
        // ColorSpace id of layer data
        if (store->hasFile("colorspace")) {
            store->open("colorspace");
            csName = TQString(store->read(store->size()));
            store->close();
        }

        KisColorSpace *cs = KisMetaRegistry::instance()->csRegistry()->getColorSpace(KisID(csName, ""), profile);

        clip = new KisPaintDevice(cs, "clip");

        if (store->hasFile("layerdata")) {
            store->open("layerdata");
            clip->read(store);
            store->close();
        }
        delete store;

        return clip->exactBounds().size();
    }
    else {
        TQImage qimg = cb->image();
        return qimg.size();
    }
;

}

#include "kis_clipboard.moc"
