/***************************************************************************
 *   Copyright (C) 2004-2009 by Thomas Fischer                             *
 *   fischer@unix-ag.uni-kl.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <tqlayout.h>
#include <tqcheckbox.h>
#include <tqtimer.h>
#include <tqvalidator.h>
#include <tqheader.h>
#include <tqlabel.h>

#include <kinputdialog.h>
#include <tdelistview.h>
#include <kiconloader.h>
#include <kpushbutton.h>
#include <tdelocale.h>
#include <kdialog.h>
#include <klineedit.h>

#include "idsuggestions.h"
#include "idsuggestionswidget.h"
#include "settings.h"
#include "fileimporter.h"
#include "fileimporterbibtex.h"
#include "settingsidsuggestions.h"

namespace KBibTeX
{
    TQString SettingsIdSuggestions::exampleBibTeXEntry = "@Article{ dijkstra1983terminationdetect,\nauthor = {Edsger W. Dijkstra and W. H. J. Feijen and A. J. M. {van Gasteren}},\ntitle = {{Derivation of a Termination Detection Algorithm for Distributed Computations}},\njournal = {Information Processing Letters},\nvolume = 16,\nnumber = 5,\npages = {217--219},\nmonth = jun,\nyear = 1983\n}";

    IdSuggestionsListViewItem::IdSuggestionsListViewItem( TDEListView *list, const TQString& plainText, BibTeX::Entry *example ): TDEListViewItem( list ), m_original( plainText ), m_example( example )
    {
        TDEListViewItem::setText( 0, parse( plainText ) );
        TDEListViewItem::setMultiLinesEnabled( true );
    }

    IdSuggestionsListViewItem::IdSuggestionsListViewItem( TDEListView *list, TDEListViewItem *prev, const TQString& plainText, BibTeX::Entry *example ): TDEListViewItem( list, prev ), m_original( plainText ), m_example( example )
    {
        TDEListViewItem::setText( 0, parse( plainText ) );
        TDEListViewItem::setMultiLinesEnabled( true );
    }

    void IdSuggestionsListViewItem::setText( int col, const TQString& text )
    {
        if ( col == 0 )
            TDEListViewItem::setText( 0, parse( text ) );
        else
            TDEListViewItem::setText( col, text );
    }

    int IdSuggestionsListViewItem::width( const TQFontMetrics & fm, const TQListView *, int c ) const
    {
        int max = 100;
        TQStringList lines = TQStringList::split( '\n', text( c ) );
        for ( TQStringList::Iterator it = lines.begin(); it != lines.end(); ++it )
        {
            int w = fm.width( *it );
            if ( w > max ) max = w;
        }

        return max + 36;
    }

    TQString IdSuggestionsListViewItem::originalText() const
    {
        return m_original;
    }

    TQString IdSuggestionsListViewItem::parse( const TQString&plainText )
    {
        m_original = plainText;
        TQString result = IdSuggestions::formatStrToHuman( plainText );
        if ( m_example != NULL )
        {
            result.append( TQString( i18n( "\nExample: %1" ) ).arg( IdSuggestions::formatId( m_example, plainText ) ) );
        }
        return result;
    }

    SettingsIdSuggestions::SettingsIdSuggestions( TQWidget *parent, const char *name )
            : TQWidget( parent, name )
    {
        m_validator = new TQRegExpValidator( TQRegExp( "[^\\s]+" ), this );
        setupGUI();

        BibTeX::FileImporter *importer = new BibTeX::FileImporterBibTeX( false );
        BibTeX::File *file = importer->load( exampleBibTeXEntry );
        m_example = new BibTeX::Entry( dynamic_cast<BibTeX::Entry*>( *( file->begin() ) ) );
        delete file;
        delete importer;
    }

    SettingsIdSuggestions::~SettingsIdSuggestions()
    {
        delete m_example;
        delete m_validator;
    }

    void SettingsIdSuggestions::applyData()
    {
        Settings * settings = Settings::self();

        settings->idSuggestions_formatStrList.clear();
        settings->idSuggestions_default = -1;
        settings->idSuggestions_forceDefault = m_checkBoxForceDefault->isChecked();
        int i = 0;
        for ( TQListViewItemIterator it( m_listIdSuggestions ); it.current(); ++i, ++it )
        {
            IdSuggestionsListViewItem *item = dynamic_cast<IdSuggestionsListViewItem*>( *it );
            settings->idSuggestions_formatStrList.append( item->originalText() );
            if ( item == m_defaultSuggestionItem )
                settings->idSuggestions_default = i;
        }
        settings->idSuggestions_smallWords.clear();
        for ( TQListViewItemIterator it( m_listSmallWords ); it.current(); ++it )
            settings->idSuggestions_smallWords << it.current()->text( 0 );
    }

    void SettingsIdSuggestions::readData()
    {
        Settings * settings = Settings::self();

        IdSuggestionsListViewItem *prev = NULL;
        m_listIdSuggestions->clear();
        m_defaultSuggestionItem = NULL;
        m_checkBoxForceDefault->setChecked( settings->idSuggestions_forceDefault );
        m_checkBoxForceDefault->setEnabled( settings->idSuggestions_default >= 0 );
        int i = 0;
        for ( TQStringList::ConstIterator it = settings->idSuggestions_formatStrList.begin(); it != settings->idSuggestions_formatStrList.end(); ++i, ++it )
        {
            prev = new IdSuggestionsListViewItem( m_listIdSuggestions, prev, *it, m_example );
            prev->setPixmap( 0, SmallIcon( "filter" ) );
            if ( i == settings->idSuggestions_default )
                m_defaultSuggestionItem = prev;
        }

        if ( m_defaultSuggestionItem != NULL )
            m_defaultSuggestionItem->setPixmap( 0, SmallIcon( "favorites" ) );

        m_listSmallWords->clear();
        for ( TQStringList::ConstIterator it = settings->idSuggestions_smallWords.begin(); it != settings->idSuggestions_smallWords.end(); ++it )
            new TDEListViewItem( m_listSmallWords, *it );
        m_lineEditSmallWords->setText( "" );
        slotListSmallWordsChanged();

        updateGUI();
    }

    void SettingsIdSuggestions::slotConfigChanged()
    {
        emit configChanged();
    }

    void SettingsIdSuggestions::slotNewIdSuggestion()
    {
        IdSuggestionsListViewItem * item = new IdSuggestionsListViewItem( m_listIdSuggestions, "a|Y|T", m_example );
        item->setPixmap( 0, SmallIcon( "filter" ) );
        m_listIdSuggestions->setSelected( item, true );
        TQTimer::singleShot( 100, this, TQ_SLOT( slotEditIdSuggestion() ) );
    }

    void SettingsIdSuggestions::slotEditIdSuggestion()
    {
        IdSuggestionsListViewItem * item = static_cast<IdSuggestionsListViewItem*>( m_listIdSuggestions->selectedItem() );
        if ( item != NULL )
        {
            TQString formatStr = item->originalText();
            if ( IdSuggestionsWidget::execute( formatStr, this ) == TQDialog::Accepted )
            {
                item->setText( 0, formatStr );
                emit configChanged();
            }
        }
        updateGUI();
    }

    void SettingsIdSuggestions::slotDeleteIdSuggestion()
    {
        IdSuggestionsListViewItem * item = static_cast<IdSuggestionsListViewItem*>( m_listIdSuggestions->selectedItem() );
        if ( item != NULL )
        {
            if ( m_defaultSuggestionItem == item )
                m_defaultSuggestionItem = NULL;
            m_checkBoxForceDefault->setEnabled( m_defaultSuggestionItem != NULL );

            delete item;
            emit configChanged();
        }
        updateGUI();
    }

    void SettingsIdSuggestions::slotMoveUpIdSuggestion()
    {
        IdSuggestionsListViewItem * item = dynamic_cast<IdSuggestionsListViewItem*>( m_listIdSuggestions->selectedItem() );
        if ( item != NULL && item -> itemAbove() != NULL )
        {
            IdSuggestionsListViewItem *itemAbove = dynamic_cast<IdSuggestionsListViewItem*>( item->itemAbove() );
            TQString text = item->originalText();
            item->setText( 0, itemAbove->originalText( ) );
            itemAbove->setText( 0, text );
            m_listIdSuggestions->setCurrentItem( itemAbove );
            m_listIdSuggestions->ensureItemVisible( itemAbove );

            if ( m_defaultSuggestionItem == itemAbove )
            {
                itemAbove->setPixmap( 0, SmallIcon( "filter" ) );
                m_defaultSuggestionItem = item;
            }
            else if ( m_defaultSuggestionItem == item )
            {
                item->setPixmap( 0, SmallIcon( "filter" ) );
                m_defaultSuggestionItem = itemAbove;
            }
            if ( m_defaultSuggestionItem != NULL )
                m_defaultSuggestionItem->setPixmap( 0, SmallIcon( "favorites" ) );
        }
    }

    void SettingsIdSuggestions::slotMoveDownIdSuggestion()
    {
        IdSuggestionsListViewItem * item = dynamic_cast<IdSuggestionsListViewItem*>( m_listIdSuggestions->selectedItem() );
        if ( item != NULL && item -> itemBelow() != NULL )
        {
            IdSuggestionsListViewItem *itemBelow = dynamic_cast<IdSuggestionsListViewItem*>( item->itemBelow() );
            TQString text = item->originalText();
            item->setText( 0, itemBelow->originalText( ) );
            itemBelow->setText( 0, text );
            m_listIdSuggestions->setCurrentItem( itemBelow );
            m_listIdSuggestions->ensureItemVisible( itemBelow );

            if ( m_defaultSuggestionItem == itemBelow )
            {
                itemBelow->setPixmap( 0, SmallIcon( "filter" ) );
                m_defaultSuggestionItem = item;
            }
            else if ( m_defaultSuggestionItem == item )
            {
                item->setPixmap( 0, SmallIcon( "filter" ) );
                m_defaultSuggestionItem = itemBelow;
            }
            if ( m_defaultSuggestionItem != NULL )
                m_defaultSuggestionItem->setPixmap( 0, SmallIcon( "favorites" ) );
        }
    }

    void SettingsIdSuggestions::slotToggleDefault()
    {
        if ( m_defaultSuggestionItem != NULL )
            m_defaultSuggestionItem->setPixmap( 0, SmallIcon( "filter" ) );

        TQListViewItem *item = m_listIdSuggestions->selectedItem();
        if ( item == m_defaultSuggestionItem )
            m_defaultSuggestionItem = NULL;
        else
        {
            m_defaultSuggestionItem = item;
            m_defaultSuggestionItem->setPixmap( 0, SmallIcon( "favorites" ) );
        }
        m_checkBoxForceDefault->setEnabled( m_defaultSuggestionItem != NULL );
    }

    void SettingsIdSuggestions::slotEditSmallWordsChanged( const TQString &newWord )
    {
        bool result = !newWord.isEmpty();

        for ( TQListViewItemIterator it( m_listSmallWords ); result && it.current(); ++it )
            result &= it.current()->text( 0 ) != newWord;

        m_buttonAddSmallWord->setEnabled( result );
    }

    void SettingsIdSuggestions::slotAddSmallWord()
    {
        new TDEListViewItem( m_listSmallWords, m_lineEditSmallWords->text() );
        m_lineEditSmallWords->setText( "" );
        emit configChanged();
    }

    void SettingsIdSuggestions::slotDeleteSmallWord()
    {
        bool changed = false;
        TQListViewItemIterator it( m_listSmallWords, TQListViewItemIterator::Selected );
        while ( it.current() )
        {
            TQListViewItem *cur = it.current();
            ++it;
            delete cur;
            changed = true;
        }

        if ( changed )
            emit configChanged();
    }

    void SettingsIdSuggestions::slotListSmallWordsChanged()
    {
        TQListViewItemIterator it( m_listSmallWords, TQListViewItemIterator::Selected );
        m_buttonDeleteSmallWord->setEnabled( it.current() != NULL );
    }

    void SettingsIdSuggestions::updateGUI()
    {
        TQListViewItem *item = m_listIdSuggestions->selectedItem();
        bool selected = item != NULL;
        m_buttonEditIdSuggestion->setEnabled( selected );
        m_buttonDeleteIdSuggestion->setEnabled( selected );
        m_buttonMoveDownIdSuggestion->setEnabled( selected && item->itemBelow() != NULL );
        m_buttonMoveUpIdSuggestion->setEnabled( selected && item->itemAbove() != NULL );
        m_buttonToggleDefault->setEnabled( selected );
    }

    void SettingsIdSuggestions::setupGUI()
    {
        TQGridLayout * gridLayout = new TQGridLayout( this, 13, 2, 0, KDialog::spacingHint(), "gridLayout" );
        gridLayout->setRowStretch( 6, 2 );
        gridLayout->setRowSpacing( 8, KDialog::spacingHint() );
        gridLayout->setRowStretch( 12, 1 );
        gridLayout->setColStretch( 0, 1 );

        m_listIdSuggestions = new TDEListView( this );
        m_listIdSuggestions->setSorting( -1, false );
        m_listIdSuggestions->addColumn( i18n( "Id Suggestions" ) );
        m_listIdSuggestions->header()->setClickEnabled( false );
        m_listIdSuggestions->setFullWidth( true );
        gridLayout->addMultiCellWidget( m_listIdSuggestions, 0, 6, 0, 0 );
        connect( m_listIdSuggestions, TQ_SIGNAL( selectionChanged() ), this, TQ_SLOT( updateGUI() ) );
        connect( m_listIdSuggestions, TQ_SIGNAL( currentChanged( TQListViewItem * ) ), this, TQ_SLOT( updateGUI() ) );
        connect( m_listIdSuggestions, TQ_SIGNAL( doubleClicked( TQListViewItem*, const TQPoint &, int ) ), this, TQ_SLOT( slotEditIdSuggestion() ) );

        m_buttonNewIdSuggestion = new KPushButton( i18n( "id suggestion",  "New" ), this );
        m_buttonNewIdSuggestion->setIconSet( TQIconSet( SmallIcon( "add" ) ) );
        gridLayout->addWidget( m_buttonNewIdSuggestion, 0, 1 );
        connect( m_buttonNewIdSuggestion, TQ_SIGNAL( clicked() ), this, TQ_SLOT( slotNewIdSuggestion() ) );
        m_buttonEditIdSuggestion = new KPushButton( i18n( "id suggestion", "Edit" ), this );
        m_buttonEditIdSuggestion->setIconSet( TQIconSet( SmallIcon( "edit" ) ) );
        gridLayout->addWidget( m_buttonEditIdSuggestion, 1, 1 );
        connect( m_buttonEditIdSuggestion, TQ_SIGNAL( clicked() ), this, TQ_SLOT( slotEditIdSuggestion() ) );
        m_buttonDeleteIdSuggestion = new KPushButton( i18n( "id suggestion", "Delete" ), this );
        m_buttonDeleteIdSuggestion->setIconSet( TQIconSet( SmallIcon( "edit-delete" ) ) );
        gridLayout->addWidget( m_buttonDeleteIdSuggestion, 2, 1 );
        connect( m_buttonDeleteIdSuggestion, TQ_SIGNAL( clicked() ), this, TQ_SLOT( slotDeleteIdSuggestion() ) );
        m_buttonMoveUpIdSuggestion = new KPushButton( i18n( "id suggestion", "Up" ), this );
        m_buttonMoveUpIdSuggestion->setIconSet( TQIconSet( SmallIcon( "go-up" ) ) );
        gridLayout->addWidget( m_buttonMoveUpIdSuggestion, 3, 1 );
        connect( m_buttonMoveUpIdSuggestion, TQ_SIGNAL( clicked() ), this, TQ_SLOT( slotMoveUpIdSuggestion() ) );
        m_buttonMoveDownIdSuggestion = new KPushButton( i18n( "id suggestion", "Down" ), this );
        m_buttonMoveDownIdSuggestion->setIconSet( TQIconSet( SmallIcon( "go-down" ) ) );
        gridLayout->addWidget( m_buttonMoveDownIdSuggestion, 4, 1 );
        connect( m_buttonMoveDownIdSuggestion, TQ_SIGNAL( clicked() ), this, TQ_SLOT( slotMoveDownIdSuggestion() ) );
        m_buttonToggleDefault = new KPushButton( i18n( "Toogle default" ), this );
        m_buttonToggleDefault->setIconSet( TQIconSet( SmallIcon( "favorites" ) ) );
        gridLayout->addWidget( m_buttonToggleDefault, 5, 1 );
        connect( m_buttonToggleDefault, TQ_SIGNAL( clicked() ), this, TQ_SLOT( slotToggleDefault() ) );

        m_checkBoxForceDefault = new TQCheckBox( i18n( "Use default id suggestion when editing new entries" ), this );
        gridLayout->addMultiCellWidget( m_checkBoxForceDefault, 7, 7, 0, 1 );

        TQLabel *label = new TQLabel( i18n( "Small Words:" ), this );
        gridLayout->addMultiCellWidget( label, 9, 9, 0, 1 );
        m_lineEditSmallWords = new KLineEdit( this );
        label->setBuddy( m_lineEditSmallWords );
        gridLayout->addWidget( m_lineEditSmallWords, 10, 0 );
        connect( m_lineEditSmallWords, TQ_SIGNAL( textChanged( const TQString & ) ), this, TQ_SLOT( slotEditSmallWordsChanged( const TQString & ) ) );
        m_buttonAddSmallWord = new KPushButton( i18n( "Add" ), this );
        m_buttonAddSmallWord->setEnabled( false );
        m_buttonAddSmallWord->setIconSet( TQIconSet( SmallIcon( "add" ) ) );
        gridLayout->addWidget( m_buttonAddSmallWord, 10, 1 );
        connect( m_buttonAddSmallWord, TQ_SIGNAL( clicked() ), this, TQ_SLOT( slotAddSmallWord() ) );

        m_listSmallWords = new TDEListView( this );
        m_listSmallWords->addColumn( i18n( "Small Words" ) );
        m_listSmallWords->header()->setClickEnabled( true );
        m_listSmallWords->setFullWidth( true );
        connect( m_listSmallWords, TQ_SIGNAL( selectionChanged() ), this, TQ_SLOT( slotListSmallWordsChanged() ) );
        connect( m_listSmallWords, TQ_SIGNAL( currentChanged( TQListViewItem * ) ), this, TQ_SLOT( slotListSmallWordsChanged() ) );
        gridLayout->addMultiCellWidget( m_listSmallWords, 11, 12, 0, 0 );
        m_buttonDeleteSmallWord = new KPushButton( i18n( "Delete" ), this );
        m_buttonDeleteSmallWord->setEnabled( false );
        m_buttonDeleteSmallWord->setIconSet( TQIconSet( SmallIcon( "edit-delete" ) ) );
        gridLayout->addWidget( m_buttonDeleteSmallWord, 11, 1 );
        connect( m_buttonDeleteSmallWord, TQ_SIGNAL( clicked() ), this, TQ_SLOT( slotDeleteSmallWord() ) );
    }
}

#include "settingsidsuggestions.moc"
