/*
    kwidgetlister.cpp

    This file is part of libtdenetwork.
    Copyright (c) 2001 Marc Mutz <mutz@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License,
    version 2, as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifh Floor, Boston, MA  02110-1301  USA

    In addition, as a special exception, the copyright holders give
    permission to link the code of this library with any edition of
    the TQt library by Trolltech AS, Norway (or with modified versions
    of TQt that use the same license as TQt), and distribute linked
    combinations including the two.  You must obey the GNU General
    Public License in all respects for all of the code used other than
    TQt.  If you modify this file, you may extend this exception to
    your version of the file, but you are not obligated to do so.  If
    you do not wish to do so, delete this exception statement from
    your version.
*/

#include "kwidgetlister.h"

#include <tdelocale.h>
#include <kdebug.h>
#include <kpushbutton.h>
#include <kiconloader.h>

#include <tqlayout.h>
#include <tqhbox.h>

#include <assert.h>

KWidgetLister::KWidgetLister( int minWidgets, int maxWidgets, TQWidget *parent, const char* name )
  : TQWidget( parent, name )
{
  mWidgetList.setAutoDelete(true);

  mMinWidgets = TQMAX( minWidgets, 1 );
  mMaxWidgets = TQMAX( maxWidgets, mMinWidgets + 1 );

  //--------- the button box
  mLayout = new TQVBoxLayout(this, 0, 4);
  mButtonBox = new TQHBox(this);
  mButtonBox->setSpacing(4);
  mLayout->addWidget( mButtonBox );

  mBtnMore = new KPushButton( i18n("more widgets","More"), mButtonBox );
  mBtnMore->setIconSet(SmallIconSet(TQString::fromLatin1("go-down")));
  mButtonBox->setStretchFactor( mBtnMore, 0 );

  mBtnFewer = new KPushButton( i18n("fewer widgets","Fewer"), mButtonBox );
  mBtnFewer->setIconSet(SmallIconSet(TQString::fromLatin1("go-up")));
  mButtonBox->setStretchFactor( mBtnFewer, 0 );

  TQWidget *spacer = new TQWidget( mButtonBox );
  mButtonBox->setStretchFactor( spacer, 1 );

  mBtnClear = new KPushButton( i18n("clear widgets","Clear"), mButtonBox );
  mBtnClear->setIconSet(SmallIconSet(TQString::fromLatin1("locationbar_erase")));
  mButtonBox->setStretchFactor( mBtnClear, 0 );

  //---------- connect everything
  connect( mBtnMore, TQ_SIGNAL(clicked()),
     this, TQ_SLOT(slotMore()) );
  connect( mBtnFewer, TQ_SIGNAL(clicked()),
     this, TQ_SLOT(slotFewer()) );
  connect( mBtnClear, TQ_SIGNAL(clicked()),
     this, TQ_SLOT(slotClear()) );

  enableControls();
}

KWidgetLister::~KWidgetLister()
{
}

void KWidgetLister::slotMore()
{
  // the class should make certain that slotMore can't
  // be called when mMaxWidgets are on screen.
  assert( (int)mWidgetList.count() < mMaxWidgets );

  addWidgetAtEnd();
  //  adjustSize();
  enableControls();
}

void KWidgetLister::slotFewer()
{
  // the class should make certain that slotFewer can't
  // be called when mMinWidgets are on screen.
  assert( (int)mWidgetList.count() > mMinWidgets );

  removeLastWidget();
  //  adjustSize();
  enableControls();
}

void KWidgetLister::slotClear()
{
  setNumberOfShownWidgetsTo( mMinWidgets );

  // clear remaining widgets
  TQPtrListIterator<TQWidget> it( mWidgetList );
  for ( it.toFirst() ; it.current() ; ++it )
    clearWidget( (*it) );

  //  adjustSize();
  enableControls();
  emit clearWidgets();
}

void KWidgetLister::addWidgetAtEnd(TQWidget *w)
{
  if (!w) w = this->createWidget(this);

  mLayout->insertWidget( mLayout->findWidget( mButtonBox ), w );
  mWidgetList.append( w );
  w->show();
  enableControls();
  emit widgetAdded();
  emit widgetAdded(w);
}

void KWidgetLister::removeLastWidget()
{
  // The layout will take care that the
  // widget is removed from screen, too.
  mWidgetList.removeLast();
  enableControls();
  emit widgetRemoved();
}

void KWidgetLister::clearWidget( TQWidget* /*aWidget*/ )
{
}

TQWidget* KWidgetLister::createWidget( TQWidget* parent )
{
  return new TQWidget( parent );
}

void KWidgetLister::setNumberOfShownWidgetsTo( int aNum )
{
  int superfluousWidgets = TQMAX( (int)mWidgetList.count() - aNum, 0 );
  int missingWidgets     = TQMAX( aNum - (int)mWidgetList.count(), 0 );

  // remove superfluous widgets
  for ( ; superfluousWidgets ; superfluousWidgets-- )
    removeLastWidget();

  // add missing widgets
  for ( ; missingWidgets ; missingWidgets-- )
    addWidgetAtEnd();
}

void KWidgetLister::enableControls()
{
  int count = mWidgetList.count();
  bool isMaxWidgets = ( count >= mMaxWidgets );
  bool isMinWidgets = ( count <= mMinWidgets );

  mBtnMore->setEnabled( !isMaxWidgets );
  mBtnFewer->setEnabled( !isMinWidgets );
}

#include "kwidgetlister.moc"
