/* This file is part of KCachegrind.
   Copyright (C) 2003 Josef Weidendorfer <Josef.Weidendorfer@gmx.de>

   KCachegrind is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation, version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

/*
 * Tab View, enclosing detailed views for one trace item in
 * two tab widgets, separated by a splitter
 */

#ifndef TABVIEW_H
#define TABVIEW_H

#include <tqptrlist.h>
#include <tqwidget.h>
#include <tqtabwidget.h>
#include <tqtabbar.h>
#include <ksqueezedtextlabel.h>
#include "traceitemview.h"

class TQSplitter;
class TabView;

/**
 * Subclass of TQTabBar to enable context menu on tabs
 */
class TabBar : public TQTabBar
{
  TQ_OBJECT
  

 public:
  TabBar(TabView*, TQTabWidget* parent, const char *name = 0);
 protected:
  void mousePressEvent(TQMouseEvent *e);

 private:
  TQTabWidget* _tabWidget;
  TabView* _tabView;
};


/**
 * Own Splitter:
 * Call checkVisiblity for all TabWidget children of the splitter
 * on a MoveEvent. This typically is produced when collapsing the widget
 * inside of another splitter.
 */
class Splitter: public TQSplitter
{
  TQ_OBJECT
  

public:
  Splitter(Qt::Orientation o, TQWidget* parent = 0, const char* name = 0);
  void checkVisiblity();

protected:
  void moveEvent(TQMoveEvent *);
};


/**
 * Own TabView:
 * - A TQTabWidget able to track its visible rect via resizeEvents.
 *   This is needed to track if this widget is collapsed in a TQSplitter.
 * - Use own TabBar for context menu
 */
class TabWidget: public TQTabWidget
{
  TQ_OBJECT
  

public:

  TabWidget(TabView*, TQWidget* parent = 0,
            const char* name = 0, WFlags f = 0);

  bool hasVisibleRect() { return _hasVisibleRect; }
  void checkVisibility();

signals:
  void visibleRectChanged(TabWidget*);

protected:
  void resizeEvent(TQResizeEvent *);
  void showEvent(TQShowEvent *);
  void hideEvent(TQHideEvent *);
  void moveEvent(TQMoveEvent *);

private:
  bool _hasVisibleRect;
};



class TabView : public TQWidget, public TraceItemView
{
  TQ_OBJECT
  

public:

  TabView(TraceItemView* parentView,
          TQWidget* parent = 0, const char* name = 0);

  virtual TQWidget* widget() { return this; }
  TQString whatsThis() const ;
  void setData(TraceData*);
  bool isViewVisible() const { return !_isCollapsed; }
  void selected(TraceItemView*, TraceItem*);
  bool active() const { return _active; }
  void setActive(bool);

  /**
   * Rearrange tabs
   * if <w> == 0, move hidden tabs
   */
  void moveTab(TQWidget* w, Position, bool wholeArea = false);

  Position tabPosition(TQWidget*);
  int visibleTabs();
  int visibleAreas();

  void readViewConfig(TDEConfig*, TQString prefix, TQString postfix, bool);
  void saveViewConfig(TDEConfig*, TQString prefix, TQString postfix, bool);

public slots:
  void tabChanged(TQWidget*);
  void visibleRectChangedSlot(TabWidget*);

signals:
  void activated(TabView*);

protected:
  void resizeEvent(TQResizeEvent *);
  bool eventFilter(TQObject*, TQEvent*);
  void mousePressEvent(TQMouseEvent*);

private:
  TraceItemView* addTab(TQString, TraceItemView*);
  void addTop(TraceItemView*);
  void addBottom(TraceItemView*);
  TabWidget* tabWidget(Position);
  void updateVisibility();
  void doUpdate(int);
  void installFocusFilters();

  // this is true if width or height <= 1, and no child updates are done
  bool _isCollapsed;

  KSqueezedTextLabel* _nameLabel;
  TQSplitter *_mainSplitter, *_leftSplitter, *_bottomSplitter;
  TabWidget *_topTW, *_leftTW, *_bottomTW, *_rightTW;
  TQPtrList<TraceItemView> _tabs;

  TQWidget* _lastFocus;
  bool _active;
};

#endif
