/*
 *
 *  Authorization Agent implementation of bluez5
 *
 *  Copyright (C) 2018  Emanoil Kotsev <deloptes@gmail.com>
 *
 *
 *  This file is part of tdebluezauth.
 *
 *  tdebluezauth is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  tdebluezauth is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with kbluetooth; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
#ifndef AGENTIMPL_H
#define AGENTIMPL_H

#include <tqdbusconnection.h>

#include <interfaces/agent1Interface.h>
#include <interfaces/tdebluezNode.h>
#include <interfaces/dbusbaseNode.h>

class Agent1InterfaceImpl : public org::bluez::Agent1Interface
{

public:
    Agent1InterfaceImpl(TQT_DBusConnection&);
    virtual ~Agent1InterfaceImpl();
protected:
    /**
     * void Release()
     *       This method gets called when the service daemon
     *       unregisters the agent. An agent can use it to do
     *       cleanup tasks. There is no need to unregister the
     *       agent, because when this method gets called it has
     *       already been unregistered.
     */
    virtual bool Release(TQT_DBusError& error);
    /**
     * string RequestPinCode(object device)
     *
     *       This method gets called when the service daemon
     *       needs to get the passkey for an authentication.
     *
     *       The return value should be a string of 1-16 characters
     *       length. The string can be alphanumeric.
     *
     *       Possible errors: org.bluez.Error.Rejected
     *                        org.bluez.Error.Canceled
     */
    virtual void RequestPinCodeAsync(int asyncCallId, const TQT_DBusObjectPath& device);
    /**
     *      void DisplayPinCode(object device, string pincode)

            This method gets called when the service daemon
            needs to display a pincode for an authentication.

            An empty reply should be returned. When the pincode
            needs no longer to be displayed, the Cancel method
            of the agent will be called.

            This is used during the pairing process of keyboards
            that don't support Bluetooth 2.1 Secure Simple Pairing,
            in contrast to DisplayPasskey which is used for those
            that do.

            This method will only ever be called once since
            older keyboards do not support typing notification.

            Note that the PIN will always be a 6-digit number,
            zero-padded to 6 digits. This is for harmony with
            the later specification.

            Possible errors: org.bluez.Error.Rejected
                             org.bluez.Error.Canceled
     */
    virtual void DisplayPinCodeAsync(int asyncCallId, const TQT_DBusObjectPath& device, const TQString& pincode);
    /**
     *      uint32 RequestPasskey(object device)

            This method gets called when the service daemon
            needs to get the passkey for an authentication.

            The return value should be a numeric value
            between 0-999999.

            Possible errors: org.bluez.Error.Rejected
                             org.bluez.Error.Canceled
     */
    virtual void RequestPasskeyAsync(int asyncCallId, const TQT_DBusObjectPath& device);
    /**
     *      void DisplayPasskey(object device, uint32 passkey,
                                uint16 entered)

            This method gets called when the service daemon
            needs to display a passkey for an authentication.

            The entered parameter indicates the number of already
            typed keys on the remote side.

            An empty reply should be returned. When the passkey
            needs no longer to be displayed, the Cancel method
            of the agent will be called.

            During the pairing process this method might be
            called multiple times to update the entered value.

            Note that the passkey will always be a 6-digit number,
            so the display should be zero-padded at the start if
            the value contains less than 6 digits.
     */
    virtual void DisplayPasskeyAsync(int asyncCallId, const TQT_DBusObjectPath& device, TQ_UINT32 passkey, TQ_UINT16 entered);
    /**
     *      void RequestConfirmation(object device, uint32 passkey)

            This method gets called when the service daemon
            needs to confirm a passkey for an authentication.

            To confirm the value it should return an empty reply
            or an error in case the passkey is invalid.

            Note that the passkey will always be a 6-digit number,
            so the display should be zero-padded at the start if
            the value contains less than 6 digits.

            Possible errors: org.bluez.Error.Rejected
                             org.bluez.Error.Canceled
     */
    virtual void RequestConfirmationAsync(int asyncCallId, const TQT_DBusObjectPath& device, TQ_UINT32 passkey);
    /**
     *      void RequestAuthorization(object device)

            This method gets called to request the user to
            authorize an incoming pairing attempt which
            would in other circumstances trigger the just-works
            model, or when the user plugged in a device that
            implements cable pairing. In the latter case, the
            device would not be connected to the adapter via
            Bluetooth yet.

            Possible errors: org.bluez.Error.Rejected
                             org.bluez.Error.Canceled
     */
    virtual void RequestAuthorizationAsync(int asyncCallId, const TQT_DBusObjectPath& device);
    /**
     *      void AuthorizeService(object device, string uuid)

            This method gets called when the service daemon
            needs to authorize a connection/service request.

            Possible errors: org.bluez.Error.Rejected
                             org.bluez.Error.Canceled
     */
    virtual void AuthorizeServiceAsync(int asyncCallId, const TQT_DBusObjectPath& device, const TQString& uuid);
    /**
     *      void Cancel()

            This method gets called to indicate that the agent
            request failed before a reply was returned.
     */
    virtual bool Cancel(TQT_DBusError& error);

    virtual void handleMethodReply(const TQT_DBusMessage& reply);

private:
    TQT_DBusConnection *m_connection;
};

class AuthService: public org::trinitydesktop::tdebluezNode
{
public:
    AuthService(TQT_DBusConnection&);
    ~AuthService();

protected:
    virtual TQT_DBusObjectBase* createInterface(const TQString&);

private:
    TQMap<TQString, TQT_DBusObjectBase*> m_interfaces;
    TQT_DBusConnection m_connection;
};

class RootNodeService: public DBusBaseNode
{
public:
    RootNodeService(TQT_DBusConnection&);
    ~RootNodeService();
protected:
    virtual TQT_DBusObjectBase* createInterface(const TQString&);
private:
    TQMap<TQString, TQT_DBusObjectBase*> m_interfaces;
    TQT_DBusConnection m_connection;
};

class OrgNodeService: public DBusBaseNode
{
public:
    OrgNodeService(TQT_DBusConnection&);
    ~OrgNodeService();
protected:
    virtual TQT_DBusObjectBase* createInterface(const TQString&);
private:
    TQMap<TQString, TQT_DBusObjectBase*> m_interfaces;
    TQT_DBusConnection m_connection;
};

class TrinityDekstopNodeService: public DBusBaseNode
{
public:
    TrinityDekstopNodeService(TQT_DBusConnection&);
    ~TrinityDekstopNodeService();
protected:
    virtual TQT_DBusObjectBase* createInterface(const TQString&);
private:
    TQMap<TQString, TQT_DBusObjectBase*> m_interfaces;
    TQT_DBusConnection m_connection;
};

#endif // AGENTIMPL_H
