# This script generates the PyTQt configuration and generates the Makefiles.
#
# Copyright (c) 2007
# 	Riverbank Computing Limited <info@riverbankcomputing.co.uk>
# 
# This file is part of PyTQt.
# 
# This copy of PyTQt is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2, or (at your option) any later
# version.
# 
# PyTQt is supplied in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
# 
# You should have received a copy of the GNU General Public License along with
# PyTQt; see the file LICENSE.  If not, write to the Free Software Foundation,
# Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.


import sys
import os
import string
import glob
import getopt
import shutil
import py_compile

import sip_tqt_config

src_dir = os.path.dirname(os.path.abspath(__file__))

# Initialise the globals.
pytqt_version = 0x031201
pytqt_version_str = "3.18.1"

sip_min_version = 0x040800


# Try and find a TQt installation to use as the default.
try:
    tqt_dir = os.environ["TQTDIR"]
except KeyError:
    tqt_dir = ""


tqt_version = 0
tqt_edition = ""
tqt_incdir = None
tqt_libdir = None
tqt_threaded = 0
tqt_winconfig = ""

pytqt = None
pytqt_modules = []
tqt_sip_flags = []
tqtext_sip_flags = []
tqtpe_sip_flags = []
tqsci_version = 0
disabled_classes = []

if sys.platform == "win32":
    tqsci_define = "TQEXTSCINTILLA_DLL"
else:
    tqsci_define = ""

# Get the SIP-TQt configuration.
sipcfg = sip_tqt_config.Configuration()

# Command line options.
opt_tqtlib = None
opt_tqconfigdir = None
opt_pytqtbindir = sipcfg.default_bin_dir
opt_pytqtmoddir = os.path.join(sipcfg.default_mod_dir, "PyTQt")
opt_pytqtsipdir = sipcfg.default_sip_dir
opt_tqtpetag = None
opt_tqsciincdir = None
opt_tqscilibdir = None
opt_static = 0
opt_debug = 0
opt_concat = 0
opt_split = 1
opt_tracing = 0
opt_verbose = 0
opt_keepfeatures = 0
opt_accept_license = 0

opt_vendorcheck = 0
opt_vendincdir = sipcfg.py_inc_dir
opt_vendlibdir = sipcfg.py_lib_dir


def usage(rcode = 2):
    """Display a usage message and exit.

    rcode is the return code passed back to the calling process.
    """
    if tqt_dir:
        def_tqt_dir = tqt_dir
    else:
        def_tqt_dir = "none"

    sys.stdout.write("Usage:\n")
    sys.stdout.write("    python configure.py [-h] [-a tag] [-b dir] [-c] [-d dir] [-e lib] [-f] [-g dir] [-i] [-j #] [-k] [-l dir] [-m dir] [-n dir] [-o dir] [-q dir] [-r] [-s] [-u] [-v dir] [-w] [-y lib] option=value option+=value ...\n")
    sys.stdout.write("where:\n")
    sys.stdout.write("  -h      display this help message\n")
    sys.stdout.write("  -a tag  explicitly enable the tqtpe module\n")
    sys.stdout.write("  -b dir  where pytquic and pytqlupdate will be installed [default %s]\n" % opt_pytqtbindir)
    sys.stdout.write("  -c      concatenate each module's C/C++ source files\n")
    sys.stdout.write("  -d dir  where the PyTQt modules will be installed [default %s]\n" % opt_pytqtmoddir)
    sys.stdout.write("  -e lib  explicitly specify the python library\n")
    sys.stdout.write("  -f      keep any existing features file (when cross-compiling) [default remove]\n")
    sys.stdout.write("  -g dir  where the TQt tqconfig.h file can be found [default TQt include directory]\n")
    sys.stdout.write("  -i      enable checking of signed interpreters using the VendorID package [default disabled]\n")
    sys.stdout.write("  -j #    split the concatenated C++ source files into # pieces [default 1]\n")
    sys.stdout.write("  -k      build the PyTQt modules as static libraries\n")
    sys.stdout.write("  -l dir  the directory containing the VendorID header file [default %s]\n" % opt_vendincdir)
    sys.stdout.write("  -m dir  the directory containing the VendorID library [default %s]\n" % opt_vendlibdir)
    sys.stdout.write("  -n dir  the directory containing the TQScintilla header files [default TQt include directory]\n")
    sys.stdout.write("  -o dir  the directory containing the TQScintilla library [default TQt lib directory]\n")
    sys.stdout.write("  -q dir  the root directory of the TQt installation [default %s]\n" % def_tqt_dir)
    sys.stdout.write("  -r      generate code with tracing enabled [default disabled]\n")
    sys.stdout.write("  -s      TQScintilla is a static library and not a DLL (Windows only)\n")
    sys.stdout.write("  -u      build with debugging symbols (requires a debug build of Python on Windows\n")
    sys.stdout.write("  -v dir  where the PyTQt .sip files will be installed [default %s]\n" % opt_pytqtsipdir)
    sys.stdout.write("  -w      don't suppress compiler output during configuration\n")
    sys.stdout.write("  -y lib  explicitly specify the type of TQt library, either tqt, tqt-mt, tqte, tqte-mt or tqtmt\n")
    sys.stdout.write("  -z      accept the license terms without prompting\n")

    sys.exit(rcode)


class ConfigureBase:
    """This is the base class for all PyTQt version specific configurer classes.
    Anything here is common to all configurers.
    """
    def check_modules(self):
        """Check which modules should be built and add them to the global list.

        Returns the name of any additional library that needs to be linked with
        the main module.
        """
        return None

    def sip_flags(self):
        """Get the configuration specific SIP-TQt flags.

        Returns a list of flags.
        """
        return []

    def tqt_version_tags(self):
        """Get the versions tags for the configuration.

        Returns a dictionary of versions and corresponding tags.
        """
        return {}

    def code(self, extra_include_dirs, extra_lib_dir, extra_libs):
        """Generate the code for a configuration.

        extra_include_dirs is a list of directories to add to those supplied by
        the SIP-TQt configuration.
        extra_lib_dir is an optional directory to add to those supplied by the
        SIP-TQt configuration.
        extra_lib_dirs is an optional list of directories to add to those
        supplied by the SIP-TQt configuration.
        """
        pass

    def tools(self):
        """Create the Makefiles for any other sub-directories and return a list
        of those directories.

        Returns a list of sub-directories with Makefile.
        """
        return []

    def module_dir(self):
        """Return the configuration's module directory.
        """
        return opt_pytqtmoddir

    def module_installs(self):
        """Return a list of files to install in the module directory other than
        the modules themselves.
        """
        return ["__init__.py", "pytqtconfig.py"]

    def sip_dir(self):
        """Return the configuration's .sip files directory.
        """
        return opt_pytqtsipdir


class ConfigurePyTQt3(ConfigureBase):
    """This class defines the methods to configure PyTQt v3.
    """
    def check_modules(self):
        pytqt_modules.append("tqt")

        check_module("tqtcanvas", "tqcanvas.h", "TQCanvas()")
        check_module("tqtnetwork", "tqsocket.h", "TQSocket()")
        check_module("tqttable", "tqtable.h", "TQTable()")
        check_module("tqtxml", "tqdom.h", "TQDomImplementation()")
        check_module("tqtgl", "tqgl.h", "TQGLWidget()", opengl=1)

        check_module("tqtui", "tqwidgetfactory.h", "TQWidgetFactory()", lib="tqui")

        if tqt_edition in ("enterprise", "free"):
            check_module("tqtsql", "tqsql.h", "TQSql()")

            if sys.platform == "win32" and sipcfg.sip_version >= 0x040200:
                check_module("tqtaxcontainer", "tqaxobject.h", "TQAxObject()", lib="tqaxcontainer")

        if tqsci_version:
            check_module("tqtext", "tqextscintillabase.h", "TQextScintillaBase()", define=tqsci_define, include_dir=opt_tqsciincdir, lib_dir=opt_tqscilibdir, lib="tqscintilla")

        if opt_tqtpetag:
            pytqt_modules.append("tqtpe")

        tqtmod_lib = None

        sip_tqt_config.inform("Checking to see if the TQAssistantClient class is available...")

        if check_class("tqassistantclient.h", "TQAssistantClient(\"foo\")", lib="tqassistantclient"):
            tqtmod_lib = "tqassistantclient"
        else:
            if check_class("ntqassistantclient.h", "TQAssistantClient(\"foo\")", lib="tqassistantclient"):
                tqtmod_lib = "tqassistantclient"
            else:
                disabled_classes.append("TQAssistantClient")

        return tqtmod_lib

    def sip_flags(self):
        return get_feature_flags()

    def tqt_version_tags(self):
        return {
            0x010403: None,
            0x020000: "TQt_1_43",
            0x020100: "TQt_2_00",
            0x020200: "TQt_2_1_0",
            0x020300: "TQt_2_2_0",
            0x020301: "TQt_2_3_0",
            0x030000: "TQt_2_3_1",
            0x030001: "TQt_3_0_0",
            0x030002: "TQt_3_0_1",
            0x030004: "TQt_3_0_2",
            0x030005: "TQt_3_0_4",
            0x030006: "TQt_3_0_5",
            0x030100: "TQt_3_0_6",
            0x030101: "TQt_3_1_0",
            0x030102: "TQt_3_1_1",
            0x030200: "TQt_3_1_2",
            0x030300: "TQt_3_2_0",
            0x030305: "TQt_3_3_0",
            0x030306: "TQt_3_3_5",
            0x040000: "TQt_3_3_6"
        }

    def code(self, extra_include_dirs, extra_lib_dir, extra_libs):
        generate_code("tqt", extra_include_dirs=extra_include_dirs, extra_lib_dir=extra_lib_dir, extra_libs=extra_libs)

        if "tqtext" in pytqt_modules:
            generate_code("tqtext", extra_define=tqsci_define, extra_include_dirs=[opt_tqsciincdir], extra_lib_dir=opt_tqscilibdir, extra_libs=["tqscintilla"]+extra_libs, sip_flags=tqtext_sip_flags)

        if "tqtgl" in pytqt_modules:
            generate_code("tqtgl", opengl=1, extra_include_dirs=extra_include_dirs, extra_lib_dir=extra_lib_dir, extra_libs=extra_libs)

        if "tqtpe" in pytqt_modules:
            generate_code("tqtpe", extra_include_dirs=extra_include_dirs, extra_lib_dir=extra_lib_dir, extra_libs=["tqpe"]+extra_libs, sip_flags=tqtpe_sip_flags)

        if "tqtui" in pytqt_modules:
            generate_code("tqtui", extra_include_dirs=extra_include_dirs, extra_lib_dir=extra_lib_dir, extra_libs=["tqui"]+extra_libs)

        if "tqtaxcontainer" in pytqt_modules:
            generate_code("tqtaxcontainer", extra_include_dirs=extra_include_dirs, extra_lib_dir=extra_lib_dir, extra_libs=["tqaxcontainer"]+extra_libs)

        # The rest don't need special handling.
        for m in ("tqtcanvas", "tqtnetwork", "tqtsql", "tqttable", "tqtxml"):
            if m in pytqt_modules:
                generate_code(m, extra_include_dirs=extra_include_dirs, extra_lib_dir=extra_lib_dir, extra_libs=extra_libs)

    def tools(self):
        tool_dirs = []

        # The Professional Edition needs special handling.
        prof = (tqt_edition == "professional")

        sip_tqt_config.inform("Creating pytquic Makefile...")

        if prof or "tqtxml" not in pytqt_modules:
            buildfile= "pytquic-prof.sbf"

            for xml in ("tqdom.cpp", "tqxml.cpp"):
                shutil.copyfile(tqt_dir + "/src/xml/" + xml, "pytquic3/" + xml)
        else:
            buildfile= "pytquic.sbf"

        makefile = sip_tqt_config.ProgramMakefile(
            configuration=sipcfg,
            build_file=os.path.join(src_dir, "pytquic3", buildfile),
            dir="pytquic3",
            install_dir=opt_pytqtbindir,
            console=1,
            tqt=1,
            warnings=1
        )

        makefile.extra_defines.append("UIC")
        makefile.extra_defines.append("TQT_INTERNAL_XML")

        if prof or "tqtxml" not in pytqt_modules:
            makefile.extra_defines.append("TQT_MODULE_XML")

        makefile.extra_include_dirs.append(os.path.join(src_dir, "pytquic3"))
        
        if not os.access("pytquic3", os.F_OK):
            os.mkdir("pytquic3")

        makefile.generate()
        tool_dirs.append("pytquic3")

        sip_tqt_config.inform("Creating pytqlupdate Makefile...")

        if prof or "tqtxml" not in pytqt_modules:
            buildfile= "pytqlupdate-prof.sbf"

            shutil.copyfile(tqt_dir + "/src/xml/tqxml.cpp", "pytqlupdate3/tqxml.cpp")
        else:
            buildfile= "pytqlupdate.sbf"

        makefile = sip_tqt_config.ProgramMakefile(
            configuration=sipcfg,
            build_file=os.path.join(src_dir, "pytqlupdate3", buildfile),
            dir="pytqlupdate3",
            install_dir=opt_pytqtbindir,
            console=1,
            tqt=1,
            warnings=1
        )

        makefile.extra_defines.append("TQT_INTERNAL_XML")

        if prof or "tqtxml" not in pytqt_modules:
            makefile.extra_defines.append("TQT_MODULE_XML")

        makefile.extra_include_dirs.append(os.path.join(src_dir, "pytqlupdate3"))

        if not os.access("pytqlupdate3", os.F_OK):
            os.mkdir("pytqlupdate3")

        makefile.generate()
        tool_dirs.append("pytqlupdate3")

        return tool_dirs


def inform_user():
    """Tell the user the option values that are going to be used.
    """
    if tqt_edition:
        edstr = tqt_edition + " edition "
    else:
        edstr = ""

    sip_tqt_config.inform("TQt v%s %sis being used." % (sip_tqt_config.version_to_string(tqt_version), edstr))
    sip_tqt_config.inform("SIP-TQt %s is being used." % sipcfg.sip_version_str)
    sip_tqt_config.inform("These PyTQt modules will be built: %s." % ' '.join(pytqt_modules))

    if disabled_classes:
        sip_tqt_config.inform("Support for these TQt classes has been disabled: %s." % ' '.join(disabled_classes))

    sip_tqt_config.inform("The PyTQt modules will be installed in %s." % opt_pytqtmoddir)
    sip_tqt_config.inform("The PyTQt .sip files will be installed in %s." % opt_pytqtsipdir)

    sip_tqt_config.inform("The TQt header files are in %s." % tqt_incdir)
    sip_tqt_config.inform("The %s TQt library is in %s." % (opt_tqtlib, tqt_libdir))

    sip_tqt_config.inform("pyuic will be installed in %s." % opt_pytqtbindir)
    sip_tqt_config.inform("pylupdate will be installed in %s." % opt_pytqtbindir)

    if opt_vendorcheck:
        sip_tqt_config.inform("PyTQt will only be usable with signed interpreters.")


def create_config(module, template, macros):
    """Create the PyTQt configuration module so that it can be imported by build
    scripts.

    module is the module file name.
    template is the template file name.
    macros is the dictionary of platform specific build macros.
    """
    sip_tqt_config.inform("Creating %s..." % module)

    content = {
        "pytqt_config_args":   sys.argv[1:],
        "pytqt_version":       pytqt_version,
        "pytqt_version_str":   pytqt_version_str,
        "pytqt_bin_dir":       opt_pytqtbindir,
        "pytqt_mod_dir":       opt_pytqtmoddir,
        "pytqt_sip_dir":       opt_pytqtsipdir,
        "pytqt_modules":       pytqt_modules,
        "pytqt_tqt_sip_flags": tqt_sip_flags,
        "tqt_version":         tqt_version,
        "tqt_edition":         tqt_edition,
        "tqt_winconfig":       tqt_winconfig,
        "tqt_framework":       0,
        "tqt_threaded":        tqt_threaded,
        "tqt_dir":             tqt_dir,
        "tqt_inc_dir":         tqt_incdir,
        "tqt_lib":             opt_tqtlib,
        "tqt_lib_dir":         tqt_libdir
    }

    if "tqtaxcontainer" in pytqt_modules:
        content["pytqt_tqtaxcontainer_sip_flags"] = tqt_sip_flags

    if "tqtcanvas" in pytqt_modules:
        content["pytqt_tqtcanvas_sip_flags"] = tqt_sip_flags

    if "tqtext" in pytqt_modules:
        content["pytqt_tqtext_sip_flags"] = tqtext_sip_flags

        # These are internal.
        content["_pytqt_tqscintilla_defines"] = tqsci_define
        content["_pytqt_tqscintilla_inc_dir"] = opt_tqsciincdir
        content["_pytqt_tqscintilla_lib_dir"] = opt_tqscilibdir

    if "tqtgl" in pytqt_modules:
        content["pytqt_tqtgl_sip_flags"] = tqt_sip_flags

    if "tqtnetwork" in pytqt_modules:
        content["pytqt_tqtnetwork_sip_flags"] = tqt_sip_flags

    if "tqtpe" in pytqt_modules:
        content["pytqt_tqtpe_sip_flags"] = tqtpe_sip_flags

    if "tqtstql" in pytqt_modules:
        content["pytqt_tqtsql_sip_flags"] = tqt_sip_flags

    if "tqttable" in pytqt_modules:
        content["pytqt_tqttable_sip_flags"] = tqt_sip_flags

    if "tqtui" in pytqt_modules:
        content["pytqt_tqtui_sip_flags"] = tqt_sip_flags

    if "tqtxml" in pytqt_modules:
        content["pytqt_tqtxml_sip_flags"] = tqt_sip_flags

    sip_tqt_config.create_config_module(module, template, content, macros)


def compile_tqt_program(name, define=None, include_dir=None, lib_dir=None, lib=None, opengl=0, python=0, debug=0):
    """Compile a simple TQt application.

    name is the name of the single source file.
    define is a name to add to the list of preprocessor defines.
    include_dir is the name of a directory to add to the list of include
    directories.
    lib_dir is the name of a directory to add to the list of library
    directories.
    lib is the name of a library to add to the list of libraries.
    opengl is set if the application uses OpenGL.
    python is set if the application #includes Python.h.
    debug is set if this is a debug build.

    Returns the name of the executable suitable for running or None if it
    wasn't created.
    """
    makefile = sip_tqt_config.ProgramMakefile(sipcfg, console=1, tqt=1, warnings=0, opengl=opengl, python=python, debug=debug)

    if define:
        makefile.extra_defines.append(define)

    if include_dir:
        makefile.extra_include_dirs.append(include_dir)

    if lib_dir:
        makefile.extra_lib_dirs.append(lib_dir)

    if lib:
        makefile.extra_libs.append(lib)

    exe, build = makefile.build_command(name)

    # Make sure the executable file doesn't exist.
    try:
        os.remove(exe)
    except OSError:
        pass

    if not opt_verbose:
        try:
            import subprocess

            p = subprocess.Popen(build, shell=True, stdin=subprocess.PIPE,
                    stdout=subprocess.PIPE, stderr=subprocess.STDOUT)

            fout = p.stdout
        except ImportError:
            _, fout = os.popen4(build)

        # Read stdout and stderr until there is no more output.
        lout = fout.readline()
        while lout:
            lout = fout.readline()

        fout.close()

        try:
            os.wait()
        except:
            pass
    else:
        os.system(build)

    if not os.access(exe, os.X_OK):
        return None

    if sys.platform != "win32":
        exe = "./" + exe

    return exe


def check_tqscintilla():
    """See if TQScintilla can be found and what its version is.
    """
    # Set the defaults if they haven't been explicitly specified.
    global opt_tqsciincdir, opt_tqscilibdir

    if opt_tqsciincdir is None:
        opt_tqsciincdir = tqt_incdir

    if opt_tqscilibdir is None:
        opt_tqscilibdir = tqt_libdir

    # Find the TQScintilla header files.
    sciglobal = os.path.join(opt_tqsciincdir, "tqextscintillaglobal.h")

    if os.access(sciglobal, os.F_OK):
        # Get the TQScintilla version number.
        global tqsci_version

        tqsci_version, sciversstr = sip_tqt_config.read_version(sciglobal, "TQScintilla", "TQSCINTILLA_VERSION", "TQSCINTILLA_VERSION_STR")

        if glob.glob(os.path.join(opt_tqscilibdir, "*tqscintilla*")):
            sip_tqt_config.inform("TQScintilla %s is being used." % sciversstr)

            # If we find a snapshot then set a negative version number as a
            # special case.
            if sciversstr.find("snapshot") >= 0:
                tqsci_version = -1
        else:
            sip_tqt_config.inform("The TQScintilla library could not be found in %s and so the tqtext module will not be built. If TQScintilla is installed then use the -o argument to explicitly specify the correct directory." % opt_tqscilibdir)
    else:
        sip_tqt_config.inform("tqextscintillaglobal.h could not be found in %s and so the tqtext module will not be built. If TQScintilla is installed then use the -n argument to explicitly specify the correct directory." % opt_tqsciincdir)


def check_vendorid():
    """See if the VendorID library and include file can be found.
    """
    global opt_vendorcheck

    if opt_vendorcheck:
        if os.access(os.path.join(opt_vendincdir, "vendorid.h"), os.F_OK):
            if glob.glob(os.path.join(opt_vendlibdir, "*vendorid*")):
                sip_tqt_config.inform("The VendorID package was found.")
            else:
                opt_vendorcheck = 0
                sip_tqt_config.inform("The VendorID library could not be found in %s and so signed interpreter checking will be disabled. If the VendorID library is installed then use the -m argument to explicitly specify the correct directory." % opt_vendlibdir)
        else:
            opt_vendorcheck = 0
            sip_tqt_config.inform("vendorid.h could not be found in %s and so signed interpreter checking will be disabled. If the VendorID package is installed then use the -l argument to explicitly specify the correct directory." % opt_vendincdir)


def check_module(mname, incfile, ctor, define=None, include_dir=None, lib_dir=None, lib=None, opengl=0):
    """See if a module can be built and, if so, add it to the global list of
    modules.

    mname is the name of the module.
    incfile is the name of the include file needed for the test.
    ctor is the C++ constructor of the class being used for the test.
    define is a name to add to the list of preprocessor defines.
    include_dir is the name of a directory to add to the list of include
    directories.
    lib_dir is the name of a directory to add to the list of library
    directories.
    lib is the name of a library to add to the list of libraries.
    opengl is set if the application uses OpenGL.
    """
    # Check the module's main .sip file exists.
    if os.access(os.path.join(src_dir, "sip", mname, mname + "mod.sip"), os.F_OK):
        sip_tqt_config.inform("Checking to see if the %s module should be built..." % mname)

        if check_class(incfile, ctor, define, include_dir, lib_dir, lib, opengl):
            pytqt_modules.append(mname)
        else:
            if check_class("nt" + incfile, ctor, define, include_dir, lib_dir, lib, opengl):
                pytqt_modules.append(mname)


def check_class(incfile, ctor, define=None, include_dir=None, lib_dir=None, lib=None, opengl=0):
    """Return non-zero if a class is available.

    incfile is the name of the include file needed for the test.
    ctor is the C++ constructor of the class.
    define is a name to add to the list of preprocessor defines.
    include_dir is the name of a directory to add to the list of include
    directories.
    lib_dir is the name of a directory to add to the list of library
    directories.
    lib is the name of a library to add to the list of libraries.
    opengl is set if the application uses OpenGL.
    """
    cfgtest = "cfgtest.cpp"

    f = open(cfgtest, "w")

    f.write("""#include <%s>

int main(int argc, char **argv)
{
    new %s;
}
""" % (incfile, ctor))

    f.close()

    return compile_tqt_program(cfgtest, define, include_dir, lib_dir, lib, opengl)


def check_plugin(cname, incfile):
    """Return non-zero if a class that might be a plugin is in the main TQt
    library.

    cname is the name of the class.
    incfile is the name of the include file needed for the test.
    """
    sip_tqt_config.inform("Checking to see if the %s class is built in..." % cname)

    return check_class(incfile, cname + "()")


def create_features_file(name):
    """Create the features file.

    name is the name of the features file in the current directory.
    """
    # The features that a given TQt configuration may or may not support.  Note
    # that STYLE_WINDOWSXP and ASSISTANTCLIENT require special handling.
    flist = ["ACTION", "CLIPBOARD", "CODECS", "COLORDIALOG", "DATASTREAM",
             "DIAL", "DNS", "DOM", "DRAGANDDROP", "ICONVIEW", "IMAGE_TEXT",
             "INPUTDIALOG", "FILEDIALOG", "FONTDATABASE", "FONTDIALOG",
             "MESSAGEBOX", "MIMECLIPBOARD",
             "NETWORKPROTOCOL", "NETWORKPROTOCOL_FTP", "NETWORKPROTOCOL_HTTP",
             "PICTURE", "PRINTDIALOG", "PRINTER", "PROGRESSDIALOG",
             "PROPERTIES",
             "SEMIMODAL", "SIZEGRIP", "SOUND", "SPLITTER", "STYLE_CDE",
             "STYLE_INTERLACE", "STYLE_MOTIF", "STYLE_MOTIFPLUS",
             "STYLE_PLATINUM", "STYLE_SGI", "STYLE_WINDOWS",
             "TABDIALOG", "TABLE", "TABLEVIEW", "TRANSFORMATIONS",
             "TRANSLATION", "WIZARD", "WORKSPACE"]

    # Generate the program which will generate the features file.
    f = open("mkfeatures.cpp", "w")

    f.write(
"""#include <Python.h>
#include <stdio.h>
#include <tqglobal.h>
#include <tqapplication.h>

int main(int argc,char **argv)
{
    FILE *fp;
    TQApplication app(argc,argv,0);

    if ((fp = fopen("%s","w")) == NULL)
    {
        printf("Unable to create '%s'\\n");
        return 1;
    }

#if !defined(WITH_THREAD) || !defined(TQT_THREAD_SUPPORT)
    fprintf(fp,"-x TQt_THREAD_SUPPORT\\n");
#endif

#if !defined(TQ_WS_WIN) || defined(TQT_NO_STYLE_WINDOWSXP)
    fprintf(fp,"-x TQt_STYLE_WINDOWSXP\\n");
#endif

#if defined(Q_OS_WIN64)
    fprintf(fp,"-x TQt_TQ_LONG_IS_long\\n");
#endif
""" % (name, name))

    for feat in flist:
        f.write(
"""
#if defined(TQT_NO_%s)
    fprintf(fp,"-x TQt_%s\\n");
#endif
""" % (feat, feat))

    # Disable TQAssistantClient for the Professional Edition.
    if "TQAssistantClient" in disabled_classes:
        f.write(
"""
    fprintf(fp,"-x TQt_ASSISTANTCLIENT\\n");
""")

    f.write(
"""
    fclose(fp);

    return 0;
}
""")

    f.close()

    # Build the program.
    exe = compile_tqt_program("mkfeatures.cpp", include_dir=sipcfg.py_inc_dir, python=1)
    if not exe:
        sip_tqt_config.error("Unable to build mkfeatures utility.")

    os.system(exe)

    # Check the file was created.
    if not os.access(name, os.F_OK):
        sip_tqt_config.error("There was an error creating the features file.")

    # Check what features have been implemented as plugins and disable them.
    plugins = [("STYLE_CDE",       "tqcdestyle.h",       "TQCDEStyle"),
               ("STYLE_INTERLACE", "tqinterlacestyle.h", "TQInterlaceStyle"),
               ("STYLE_MOTIF",     "tqmotifstyle.h",     "TQMotifStyle"),
               ("STYLE_MOTIFPLUS", "tqmotifplusstyle.h", "TQMotifPlusStyle"),
               ("STYLE_PLATINUM",  "tqplatinumstyle.h",  "TQPlatinumStyle"),
               ("STYLE_SGI",       "tqsgistyle.h",       "TQSGIStyle"),
               ("STYLE_WINDOWSXP", "tqwindowsxpstyle.h", "TQWindowsXPStyle"),
               ("STYLE_WINDOWS",   "tqwindowsstyle.h",   "TQWindowsStyle")]

    f = open(name, "a")

    for (feat, incfile, cname) in plugins:
        if not check_plugin(cname, incfile):
            if not check_plugin(cname, "nt" + incfile):
                f.write("-x TQt_%s\n" % feat)
                disabled_classes.append(cname)

    f.close()


def get_feature_flags():
    """Return the list of SIP-TQt flags that exclude unsupported TQt features.
    """
    featfile = "features"

    # Create the features file if it doesn't exist and we are not keeping it.
    if opt_keepfeatures and os.access(featfile,os.F_OK):
        sip_tqt_config.inform("Using existing features file.")
    else:
        sip_tqt_config.inform("Creating features file...")
        create_features_file(featfile)

    # Parse the features file.
    ff = open(featfile, "r")

    flags = []

    line = ff.readline()
    while line:
        flags.extend(line.split())
        line = ff.readline()

    if sipcfg.sip_version >= 0x040702:
        flags.extend(['-x', 'TQt_SIP_PRE_4_7_2'])

    return flags


def set_sip_flags():
    """Set the SIP-TQt platform, version and feature flags.
    """
    tqt_sip_flags.extend(pytqt.sip_flags())

    # If we don't check for signed interpreters, we exclude the 'VendorID'
    # feature
    if not opt_vendorcheck:
        tqt_sip_flags.append("-x")
        tqt_sip_flags.append("VendorID")

    # Handle the platform tag.
    if opt_tqtpetag:
        plattag = "WS_QWS"
    elif sys.platform == "win32":
        plattag = "WS_WIN"
    elif sys.platform == "darwin":
        if "__DARWIN_X11__" in sipcfg.build_macros()["DEFINES"]:
            plattag = "WS_X11"
        else:
            plattag = "WS_MACX"
    else:
        plattag = "WS_X11"

    tqt_sip_flags.append("-t")
    tqt_sip_flags.append(plattag)

    # Handle the TQt version tag.
    verstag = sip_tqt_config.version_to_sip_tag(tqt_version, pytqt.tqt_version_tags(), "TQt")

    if verstag:
        tqt_sip_flags.append("-t")
        tqt_sip_flags.append(verstag)

    # The flags so far are common.
    for f in tqt_sip_flags:
        tqtext_sip_flags.append(f)
        tqtpe_sip_flags.append(f)

    # Handle the TQScintilla version tag.
    if tqsci_version:
        tqscitags = {
            0x010100: None,
            0x010200: "TQScintilla_1_1",
            0x010300: "TQScintilla_1_2",
            0x010400: "TQScintilla_1_3",
            0x010500: "TQScintilla_1_4",
            0x010600: "TQScintilla_1_5",
            0x010700: "TQScintilla_1_6",
            0x020000: "TQScintilla_1_7"
        }

        verstag = sip_tqt_config.version_to_sip_tag(tqsci_version, tqscitags, "TQScintilla")

        if verstag:
            tqtext_sip_flags.append("-t")
            tqtext_sip_flags.append(verstag)

    # Handle the TQtopia tag.
    if opt_tqtpetag:
        tqtpe_sip_flags.append("-t")
        tqtpe_sip_flags.append(opt_tqtpetag)


def generate_code(mname, extra_cflags=None, extra_cxxflags=None, extra_define=None, extra_include_dirs=None, extra_lflags=None, extra_lib_dir=None, extra_libs=None, opengl=0, sip_flags=None):
    """Generate the code for a module.

    mname is the name of the module.
    extra_cflags is a string containing additional C compiler flags.
    extra_cxxflags is a string containing additional C++ compiler flags.
    extra_define is a name to add to the list of preprocessor defines.
    extra_include_dirs is a list of directories to add to the list of include
    directories.
    extra_lflags is a string containing additional linker flags.
    extra_lib_dir is the name of a directory to add to the list of library
    directories.
    extra_libs is a list of the names of extra libraries to add to the list of
    libraries.
    opengl is set if the module needs OpenGL support.
    sip_flags is the list of sip-tqt flags to use instead of the defaults.
    """
    sip_tqt_config.inform("Generating the C++ source for the %s module..." % mname)

    try:
        shutil.rmtree(mname)
    except:
        pass

    try:
        os.mkdir(mname)
    except:
        sip_tqt_config.error("Unable to create the %s directory." % mname)

    # Build the SIP-TQt command line.
    argv = ['"' + sipcfg.sip_bin + '"']

    if sip_flags is None:
        sip_flags = tqt_sip_flags

    argv.extend(sip_flags)

    if opt_concat:
        argv.append("-j")
        argv.append(str(opt_split))

    if opt_tracing:
        argv.append("-r")

    argv.append("-c")
    argv.append(mname)

    buildfile = os.path.join(mname, mname + ".sbf")
    argv.append("-b")
    argv.append(buildfile)

    argv.append("-I")
    argv.append(os.path.join(src_dir, "sip"))

    # SIP-TQt assumes POSIX style path separators.
    argv.append('/'.join([src_dir, "sip", mname, mname + "mod.sip"]))

    os.system(' '.join(argv))

    # Check the result.
    if not os.access(buildfile, os.F_OK):
        sip_tqt_config.error("Unable to create the C++ code.")

    # Generate the Makefile.
    sip_tqt_config.inform("Creating the Makefile for the %s module..." % mname)

    installs = []
    sipfiles = []

    for s in glob.glob("sip/" + mname + "/*.sip"):
        sipfiles.append(os.path.join(src_dir, "sip", mname, os.path.basename(s)))

    installs.append([sipfiles, os.path.join(pytqt.sip_dir(), mname)])

    makefile = sip_tqt_config.SIPModuleMakefile(
        configuration=sipcfg,
        build_file=mname + ".sbf",
        dir=mname,
        install_dir=pytqt.module_dir(),
        installs=installs,
        tqt=1,
        opengl=opengl,
        warnings=1,
        static=opt_static,
        debug=opt_debug
    )

    if extra_cflags:
        makefile.extra_cflags.append(extra_cflags)

    if extra_cxxflags:
        makefile.extra_cxxflags.append(extra_cxxflags)

    if extra_define:
        makefile.extra_defines.append(extra_define)

    if extra_include_dirs:
        makefile.extra_include_dirs.extend(extra_include_dirs)

    if extra_lflags:
        makefile.extra_lflags.append(extra_lflags)

    if extra_lib_dir:
        makefile.extra_lib_dirs.append(extra_lib_dir)

    if extra_libs:
        makefile.extra_libs.extend(extra_libs)

    makefile.generate()


def check_license():
    """Handle the validation of the PyTQt license.
    """
    try:
        import license
        ltype = license.LicenseType
        lname = license.LicenseName

        try:
            lfile = license.LicenseFile
        except AttributeError:
            lfile = None
    except ImportError:
        ltype = None

    if ltype is None:
        ltype = "GPL"
        lname = "GNU General Public License"
        lfile = None

    sip_tqt_config.inform("This is the %s version of PyTQt %s (licensed under the %s) for Python %s on %s." % (ltype, pytqt_version_str, lname, sys.version[0].split(), sys.platform))

    # Common checks.
    if ltype == "GPL" and sys.platform == "win32":
        error("You cannot use the GPL version of PyTQt under Windows.")

    try:
        tqted = tqt_edition
    except AttributeError:
        tqted = None

    if tqted and ltype != "internal":
        if (tqted == "free" and ltype != "GPL") or (tqted != "free" and ltype == "GPL"):
            sip_tqt_config.error("This version of PyTQt and the %s edition of TQt have incompatible licenses." % tqted)

    # Confirm the license.
    sys.stdout.write("""
Type 'L' to view the license.
Type 'yes' to accept the terms of the license.
Type 'no' to decline the terms of the license.

""")

    while 1:
        sys.stdout.write("Do you accept the terms of the license? ")
        sys.stdout.flush()

        try:
            resp = sys.stdin.readline()
        except KeyboardInterrupt:
            raise SystemExit
        except:
            resp = ""

        resp = resp.strip().lower()

        if resp == "yes":
            break

        if resp == "no":
            sys.exit(0)

        if resp == "l":
            os.system("more LICENSE")

    # If there should be a license file then check it is where it should be.
    if lfile:
        if os.access(os.path.join("sip", lfile), os.F_OK):
            sip_tqt_config.inform("Found the license file %s." % lfile)
        else:
            sip_tqt_config.error("Please copy the license file %s to the sip directory." % lfile)


def get_build_macros(overrides):
    """Return the dictionary of platform specific build macros from the TQt
    installation.  Return None if any of the overrides was invalid.

    overrides is a list of macros overrides from the user.
    """
    # Get the name of the tqmake configuration file to take the macros from.
    if "QMAKESPEC" in list(os.environ.keys()):
        fname = os.path.join(tqt_dir, "mkspecs", os.environ["QMAKESPEC"], "qmake.conf")
    else:
        fname = os.path.join(tqt_dir, "mkspecs", "default", "qmake.conf")

        if not os.access(fname, os.F_OK):
            sip_tqt_config.error("Unable to find the default configuration file %s. You can use the QMAKESPEC environment variable to specify the correct platform instead of \"default\"." % fname)

    # Add the TQt specific macros to the default.
    names = list(sipcfg.build_macros().keys())
    names.append("INCDIR_TQT")
    names.append("LIBDIR_TQT")
    names.append("MOC")

    # Make sure $TQTDIR reflects any directory passed on the command line.
    os.environ["TQTDIR"] = tqt_dir

    properties = {
        "TQT_INSTALL_BINS":      os.path.join(tqt_dir, "bin"),
        "TQT_INSTALL_HEADERS":   os.path.join(tqt_dir, "include"),
        "TQT_INSTALL_LIBS":      os.path.join(tqt_dir, "lib")
    }

    return sip_tqt_config.parse_build_macros(fname, names, overrides, properties)


def check_tqt_installation(macros):
    """Check the TQt installation and get the version number and edition.

    macros is the dictionary of build macros.
    """
    # Get the Makefile generator.
    generator = macros["MAKEFILE_GENERATOR"]

    # Set the TQt include and lib directories.
    global tqt_incdir, tqt_libdir

    tqt_incdir = macros["INCDIR_TQT"]

    if not tqt_incdir:
        tqt_incdir = os.path.join(tqt_dir, "include")
        macros["INCDIR_TQT"] = tqt_incdir

    tqt_libdir = macros["LIBDIR_TQT"]

    if not tqt_libdir:
        tqt_libdir = os.path.join(tqt_dir, "lib")
        macros["LIBDIR_TQT"] = tqt_libdir

    # Check the TQt header files have been installed.
    tqglobal = os.path.join(tqt_incdir, "tqglobal.h")

    if not os.access(tqglobal, os.F_OK):
        tqglobal = os.path.join(tqt_incdir, "ntqglobal.h")

        if not os.access(tqglobal, os.F_OK):
            sip_tqt_config.error("tqglobal.h or ntqglobal.h could not be found in %s." % tqt_incdir)

    # Get the TQt version number.
    global tqt_version

    tqt_version, ignore = sip_tqt_config.read_version(tqglobal, "TQt", "TQT_VERSION")

    # Try and work out which edition it is.
    global tqt_edition

    if opt_tqconfigdir:
        tqconfigdir = opt_tqconfigdir
    else:
        tqconfigdir = tqt_incdir

    tqconfig = os.path.join(tqconfigdir, "tqconfig.h")

    if not os.access(tqconfig,os.F_OK):
        tqconfig = os.path.join(tqconfigdir, "ntqconfig.h")

        if not os.access(tqconfig,os.F_OK):
            sip_tqt_config.error("tqconfig.h or ntqconfig.h could not be found in %s." % tqconfigdir)

    f = open(tqconfig)
    l = f.readline()

    while l:
        wl = l.split()
        if len(wl) == 3 and wl[0] == "#define" and wl[1] == "QT_PRODUCT_LICENSE":
                tqt_edition = wl[2][4:-1]
                break

        l = f.readline()

    f.close()

    if not tqt_edition:
        sip_tqt_config.error("The TQt edition could not be determined by parsing %s." % tqconfig)

    if sys.platform == "win32":
        # Work out how TQt was built on Windows.

        global tqt_winconfig

        try:
            f = open(os.path.join(tqt_dir, ".tqtwinconfig"), "r")
        except IOError:
            f = None

        if f:
            cfg = f.readline()
            f.close()

            val = cfg.find("=")

            if val >= 0:
                tqt_winconfig = string.strip(cfg[val + 1:])
        else:
            # Assume it was built as a DLL.
            tqt_winconfig = "shared"

    # Determine the TQt library to link against and if it has thread support.
    global tqt_threaded

    resolve_tqt3_library(generator)

    if opt_tqtlib in ("tqt-mt", "tqt-mtedu", "tqt-mteval", "tqte-mt", "tqtmt", "tqtmtedu", "tqtmteval", "tqt-mt", "tqt-mtedu", "tqt-mteval", "tqte-mt", "tqtmt", "tqtmtedu", "tqtmteval"):
        tqt_threaded = 1

    global pytqt

    pytqt = ConfigurePyTQt3()

    # We haven't yet factored out sip_tqt_config's knowledge of how to build TQt
    # binaries and it is expecting to find these in the configuration when it
    # generates the Makefiles.
    sipcfg.tqt_version = tqt_version
    sipcfg.tqt_edition = tqt_edition
    sipcfg.tqt_winconfig = tqt_winconfig
    sipcfg.tqt_framework = 0
    sipcfg.tqt_threaded = tqt_threaded
    sipcfg.tqt_dir = tqt_dir
    sipcfg.tqt_lib = opt_tqtlib
    sipcfg.tqt_lib_dir = tqt_libdir


def resolve_tqt3_library(generator):
    """See which version of the TQt v3 library can be found.  (We can't trust
    the configuration files.)

    generator is the name of the Makefile generator.
    """
    global opt_tqtlib

    if opt_tqtlib:
        if not is_tqt_library(generator, opt_tqtlib):
            sip_tqt_config.error("The %s TQt library could not be found in %s." % (opt_tqtlib, tqt_libdir))
    else:
        stlib = is_tqt_library(generator, "tqt")
        mtlib = is_tqt_library(generator, "tqt-mt")
        edlib = is_tqt_library(generator, "tqt-mtedu")
        evlib = is_tqt_library(generator, "tqt-mteval")
        emlib = is_tqt_library(generator, "tqte")
        etlib = is_tqt_library(generator, "tqte-mt")

        # Borland likes to be a little different.
        bmtlib = is_tqt_library(generator, "tqtmt")
        bedlib = is_tqt_library(generator, "tqtmtedu")
        bevlib = is_tqt_library(generator, "tqtmteval")

        names = []

        if stlib:
            opt_tqtlib = "tqt"
            names.append(opt_tqtlib)

        if mtlib:
            opt_tqtlib = "tqt-mt"
            names.append(opt_tqtlib)

        if edlib:
            opt_tqtlib = "tqt-mtedu"
            names.append(opt_tqtlib)

        if evlib:
            opt_tqtlib = "tqt-mteval"
            names.append(opt_tqtlib)

        if emlib:
            opt_tqtlib = "tqte"
            names.append(opt_tqtlib)

        if etlib:
            opt_tqtlib = "tqte-mt"
            names.append(opt_tqtlib)

        if bmtlib:
            opt_tqtlib = "tqtmt"
            names.append(opt_tqtlib)

        if bedlib:
            opt_tqtlib = "tqtmtedu"
            names.append(opt_tqtlib)

        if bevlib:
            opt_tqtlib = "tqtmteval"
            names.append(opt_tqtlib)

        if not names:
            sip_tqt_config.error("No TQt libraries could be found in %s." % tqt_libdir)

        if len(names) > 1:
            sip_tqt_config.error("These TQt libraries were found: %s. Use the -y argument to explicitly specify which you want to use." % ' '.join(names))


def is_tqt_library(generator, lib):
    """See if a particular TQt library is installed.

    generator is the name of the Makefile generator.
    lib is the name of the library.
    """
    if generator in ("MSVC", "MSVC.NET", "BMAKE"):
        lpatts = [lib + "[0-9]*.lib", lib + ".lib"]
    else:
        lpatts = ["lib" + lib + ".*"]

    for lpatt in lpatts:
        lmatch = glob.glob(os.path.join(tqt_libdir, lpatt))

        if lmatch:
            return lmatch

    return []


def main(argv):
    """Create the configuration module module.

    argv is the list of command line arguments.
    """
    # Check SIP-TQt is new enough.
    if sipcfg.sip_version_str[:8] != "snapshot":
        if sipcfg.sip_version < sip_min_version:
            sip_tqt_config.error("This version of PyTQt requires SIP-TQt v%s or later" % sip_tqt_config.version_to_string(sip_min_version))

    # Parse the command line.
    try:
        optlist, args = getopt.getopt(argv[1:], "ha:b:cd:e:fg:ij:kl:m:n:o:q:rsuv:wy:z")
    except getopt.GetoptError:
        usage()

    global tqt_dir, opt_tqtlib, opt_tqconfigdir
    global opt_pytqtbindir, opt_pytqtmoddir, opt_pytqtsipdir
    global opt_tqtpetag, opt_static, opt_debug, opt_concat
    global opt_split, opt_tracing, opt_verbose, opt_keepfeatures
    global opt_tqsciincdir, opt_tqscilibdir, tqsci_define
    global opt_vendorcheck, opt_vendincdir, opt_vendlibdir
    global opt_libpython
    global opt_accept_license

    opt_libpython = None

    for opt, arg in optlist:
        if opt == "-h":
            usage(0)
        elif opt == "-a":
            opt_tqtpetag = arg
        elif opt == "-b":
            opt_pytqtbindir = os.path.abspath(arg)
        elif opt == "-c":
            opt_concat = 1
        elif opt == "-d":
            opt_pytqtmoddir = os.path.abspath(arg)
        elif opt == "-e":
            opt_libpython = arg
        elif opt == "-f":
            opt_keepfeatures = 1
        elif opt == "-g":
            opt_tqconfigdir = os.path.abspath(arg)
        elif opt == "-i":
            opt_vendorcheck = 1
        elif opt == "-j":
            try:
                opt_split = int(arg)
            except:
                usage()
        elif opt == "-k":
            opt_static = 1
        elif opt == "-l":
            opt_vendincdir = arg
        elif opt == "-m":
            opt_vendlibdir = arg
        elif opt == "-n":
            opt_tqsciincdir = arg
        elif opt == "-o":
            opt_tqscilibdir = arg
        elif opt == "-q":
            tqt_dir = os.path.abspath(arg)
        elif opt == "-r":
            opt_tracing = 1
        elif opt == "-s":
            tqsci_define = ""
        elif opt == "-u":
            opt_debug = 1
        elif opt == "-v":
            opt_pytqtsipdir = os.path.abspath(arg)
        elif opt == "-w":
            opt_verbose = 1
        elif opt == "-y":
            if arg in ("tqt", "tqt-mt", "tqt-mtedu", "tqt-mteval", "tqte", "tqte-mt", "tqtmt", "tqtmtedu", "tqt", "tqt-mt", "tqt-mtedu", "tqt-mteval", "tqte", "tqte-mt", "tqtmt", "tqtmtedu"):
                opt_tqtlib = arg
            else:
                usage()
        elif opt == "-z":
            opt_accept_license = 1

    # Check that we know the name of the TQt root directory.
    if not tqt_dir:
        sip_tqt_config.error("A TQt installation could not be found. Use use the -q argument or the TQTDIR environment variable to explicitly specify the correct directory.")

    # When building static libraries, signed interpreter checking makes no
    # sense.
    if opt_vendorcheck and opt_static:
        sip_tqt_config.error("Using the VendorID package when building static libraries makes no sense.")

    # Replace the existing build macros with the ones from the TQt installation.
    macros = get_build_macros(args)

    if macros is None:
        usage()

    sipcfg.set_build_macros(macros)

    # Check TQt is what we need.
    check_tqt_installation(macros)

    # Check the licenses are compatible.
    if opt_accept_license == 1:
        print("License accepted by command line option.")
    else:
        check_license()

    # Check for TQScintilla.
    check_tqscintilla()

    # Check which modules to build.
    tqtmod_lib = pytqt.check_modules()

    # Check for the VendorID package.
    check_vendorid()

    # Set the SIP-TQt platform, version and feature flags.
    set_sip_flags()

    # Tell the user what's been found.
    inform_user()

    # Generate the code.

    extra_include_dirs = []
    extra_libs = []

    if tqtmod_lib:
        extra_libs.append(tqtmod_lib)

    if opt_libpython:
        extra_libs.append(opt_libpython)

    if opt_vendorcheck:
        extra_include_dirs.append(opt_vendincdir)
        extra_lib_dir = opt_vendlibdir
        extra_libs.append("vendorid")
    else:
        extra_lib_dir = None

    pytqt.code(extra_include_dirs, extra_lib_dir, extra_libs)

    # Create the additional Makefiles.
    sip_tqt_config.inform("Creating top level Makefile...")

    sip_tqt_config.ParentMakefile(
        configuration=sipcfg,
        subdirs=pytqt_modules + pytqt.tools(),
        installs=(pytqt.module_installs(), pytqt.module_dir())
    ).generate()

    # Install module initialization script.
    create_config("__init__.py", os.path.join(src_dir, "module-init.py"), macros)

    # Install the configuration module.
    create_config("pytqtconfig.py", os.path.join(src_dir, "pytqtconfig.py.in"), macros)


###############################################################################
# The script starts here.
###############################################################################

if __name__ == "__main__":
    try:
        main(sys.argv)
    except SystemExit:
        raise
    except:
        sys.stderr.write(
"""An internal error occured.  Please report all the output from the program,
including the following traceback, to support@riverbankcomputing.co.uk.
""")
        raise
