/***************************************************************************
 *   Copyright (C) 2004-2005 by David Saxton                               *
 *   david@bluehaze.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "circuitdocument.h"
#include "ecsubcircuit.h"
#include "itemdocumentdata.h"
#include "itemlibrary.h"
#include "itemselector.h"
#include "subcircuits.h"

#include <tdeapplication.h>
#include <tdeconfig.h>
#include <kdebug.h>
#include <kiconloader.h>
#include <kstandarddirs.h>
#include <tqfile.h>
#include <tqtextstream.h>

Subcircuits::Subcircuits()
	: TQObject()
{
	connect( ComponentSelector::self(), TQT_SIGNAL(itemRemoved(const TQString& )), this, TQT_SLOT(slotItemRemoved(const TQString& )) );
}


Subcircuits::~Subcircuits()
{
}


void Subcircuits::initECSubcircuit( int subcircuitId, ECSubcircuit *ecSubcircuit )
{
	const TQString fileName = genFileName(subcircuitId);
	if ( !TQFile::exists(fileName) )
	{
		kdDebug() << "Subcircuits::createSubcircuit: Subcircuit \""<<fileName<<"\" was not found."<<endl;
		return;
	}
	
	SubcircuitData subcircuit;
	if (!subcircuit.loadData( genFileName(subcircuitId) ) )
		return;
	
	subcircuit.initECSubcircuit(ecSubcircuit);
}


ECSubcircuit* Subcircuits::createSubcircuit( int id, CircuitDocument *circuitDocument, bool newItem, const char *newId )
{
	//I pass finishCreation = false here because the subcircuit was getting
	//finished twice, causing a segfault in CircuitDocument::assignCircuits()
	//--electronerd
	ECSubcircuit *ecSubcircuit = static_cast<ECSubcircuit*>(itemLibrary()->createItem( "ec/subcircuit", circuitDocument, newItem, newId, false ));
	ecSubcircuit->property("id")->setValue(id);
	return ecSubcircuit;
}


void Subcircuits::loadSubcircuits()
{
	TDEConfig *config = kapp->config();
	config->setGroup("Subcircuits");
	
	TQValueList<int> idList = config->readIntListEntry("Ids");
	const TQValueList<int>::iterator idListEnd = idList.end();
	for ( TQValueList<int>::iterator it = idList.begin(); it != idListEnd; ++it )
	{
		TQFile file( genFileName(*it) );
		if ( file.open(IO_ReadOnly) == false )
		{
			// File has mysteriously disappeared....
			*it = -1;
		}
		else
		{
			config->setGroup("Subcircuit_"+TQString::number(*it));
			updateComponentSelector( *it, config->readEntry("Name") );
		}
		file.close();
	}
	idList.remove(-1);
	
	// Update the config file if any ids have been removed
	config->setGroup("Subcircuits");
	config->writeEntry( "Ids", idList );
}


TQString Subcircuits::genFileName( const int nextId )
{
	return locateLocal( "appdata", "subcircuit_"+TQString::number(nextId)+".circuit" );
}


void Subcircuits::updateComponentSelector( int id, const TQString &name )
{
	if ( name.isEmpty() )
		return;
	
	ComponentSelector::self()->addItem( name, "sc/"+TQString::number(id), "Subcircuits", TDEGlobal::iconLoader()->loadIcon( "ktechlab_circuit", TDEIcon::Small ), true );
}


void Subcircuits::addSubcircuit( const TQString &name, const TQString &subcircuitXml )
{
	TDEConfig *config = kapp->config();
	config->setGroup("Subcircuits");
	
	int nextId = config->readNumEntry( "NextId", 0 );
	
	while ( TQFile::exists( genFileName(nextId) ) ) {
		nextId++;
	}
	
	const int id = nextId;
	
	const TQString fileName = genFileName(id);
	TQFile file(fileName);
	
	if ( file.open(IO_WriteOnly) == false )
	{
		kdError() << "Subcircuits::addSubcircuit: couldn't open subcircuit save file: "<<fileName<<endl;
		return;
	}
	
	TQTextStream stream(&file);
	stream << subcircuitXml;
	file.close();
	
	TQValueList<int> idList = config->readIntListEntry("Ids");
	idList += id;
	config->writeEntry( "Ids", idList );
	config->writeEntry( "NextId", ++nextId );
	
	config->setGroup("Subcircuit_"+TQString::number(id));
	config->writeEntry( "Name", name );
	
	// It's important that we write the configuration *now*, lest the subcircuits be lost
	config->sync();
	
	updateComponentSelector( id, name );
}


void Subcircuits::slotItemRemoved( const TQString &id )
{
	if ( !id.startsWith("sc/") ) {
		return;
	}
	
	TQString temp = id;
	temp.remove("sc/");
	const int id_num = temp.toInt();
	const TQString fileName = genFileName(id_num);
	TQFile file(fileName);
	file.remove();
	
	TDEConfig *config = kapp->config();
	config->setGroup("Subcircuits");
	TQValueList<int> idList = config->readIntListEntry("Ids");
	idList.remove(id_num);
	config->writeEntry( "Ids", idList );
}


#include "subcircuits.moc"


