/*
	mmainwindow.h - The main window
	Copyright (C) 2003  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __MMAINWINDOW_H__
#define __MMAINWINDOW_H__

#include "actions.h"
#include "kshutdowniface.h"

#include <tqdatetime.h>

#include <tdemainwindow.h>

class AppObserver;

class TQDateTimeEdit;
class TQLabel;

class TDEGlobalAccel;
class KPushButton;

#define ks_main MMainWindow::getInstance()

/** @short The main window. */
class MMainWindow: public TDEMainWindow, virtual public KShutdownIface
{
	Q_OBJECT
public:
	/**
	 * Timeout options.
	 */
	enum DelayType {
		DelayType_Now = 0, /**< Immediate action (no delay). */
		DelayType_TimeFromNow = 1, /**< Time from now in HH:MM format. */
		DelayType_DateTime = 2, /**< Absolute date and time. */
		DelayType_OnAppExit = 3 /**< When desired application exit. */
	};

	/** Constructor. */
	MMainWindow();

	/** Destructor. */
	virtual ~MMainWindow();

	/** Sets current action to @p action. */
	void setAction(const Action::Type action);

	/** Sets delay type to @p value. */
	void setDelayType(const DelayType value);

	/** Sets editor time to @p time. */
	void setTime(const TQTime &time);

	/** Returns the main window widget. Accessed via the @c ks_main macro. */
	inline static MMainWindow *getInstance() { return _instance; }

	/** Sets editor date to @p date. */
	void setDate(const TQDate &date);

	/**
	 * Activates the selected action.
	 * @param confirmation @c false - no confirmation
	 */
	void start(const bool confirmation = true);

	// actions
	inline TDEAction *cancelAction() const { return _cancelAction; }
	inline TDEAction *checkSystemConfigurationAction() const { return _checkSystemConfigurationAction; }
	inline TDEAction *configureKShutDownAction() const { return _configureKShutDownAction; }
	inline TDEAction *lockScreenAction() const { return _lockScreenAction; }
	inline TDEAction *logoutAction() const { return _logoutAction; }
	inline TDEAction *rebootAction() const { return _rebootAction; }
	inline TDEAction *shutDownAction() const { return _shutDownAction; }
	inline TDEAction *statisticsAction() const { return _statisticsAction; }

	// DCOP:
	// general:
	virtual void cancel();
	virtual bool shutDown();
	virtual bool shutdown();
	virtual bool reboot();
	virtual bool lockScreen();
	virtual bool logout();
	// misc:
	virtual void configure();
	virtual TQString getStatusInfo();
	virtual void makeInvisible();
	virtual void makeVisible();
	virtual void setTestMode(bool yes);
private:
	bool _showMinimizeInfo;
	// actions
	TDEAction
		*_cancelAction,
		*_checkSystemConfigurationAction,
		*_configureKShutDownAction,
		*_lockScreenAction,
		*_logoutAction,
		*_rebootAction,
		*_shutDownAction,
		*_startAction,
		*_statisticsAction;
	TDEActionCollection *ac_shutDown;
	TDEGlobalAccel *_globalAccel;

	int _oldSec;
	AppObserver *_appObserver;
	TQDateTimeEdit *_dateTimeEdit;
	KPushButton *b_startStop;

	// points to the main window widget (this)
	static MMainWindow *_instance;
	TQComboBox
		*cb_actions,
		*cb_delayTypes;
	TQDateTime
		dt_end, // end date/time
		dt_start; // start date/time
	TQLabel *l_statusLine;
	TQTimer *_timer; // timeout timer
	DelayType _delayType;

	TQString getTimeInfo();

	void initMainMenu();

	void initActions();
	TQWidget *initWidgets();

	/**
	 * Sets window and system tray captions.
	 * @param remainingTime A remaining time
	 * @param selectedTime A selected time
	 */
	void setCaptions(const TQString &remainingTime = TQString::null, const TQString &selectedTime = TQString::null);

	void setWidgetsEnabled(const bool yes);

	/**
	 * Updates time infos (e.g. window captions).
	 * Displays time to the final shut down, and the selected action name.
	 */
	void updateTimeInfo(const int secs);
private slots:
	/**
	 * This signal is emitted when the application is about to quit, or when the
	 * user shuts down the entire desktop session.
	 */
	void slotAboutToQuit();

	/**
	 * Checks the time, and after timeout executes active action.
	 */
	void slotCheckTime();

	/**
	 * Called when a new item has been selected in the combo box.
	 * @param index A new item index
	 */
	void slotDelayTypeChange(int index);

	void slotConfigureGlobalShortcuts();
	void slotConfigureKShutDown();
	void slotConfigureNotifications();
	void slotConfigureShortcuts();

	void slotActionChange(int index);
	void slotCheckSystemConfig();
	void slotStats();

	/**
	 * Updates "Selected time:" info.
	 * @see calcSelectedTime
	 */
	void slotUpdateTimeInfo();
public slots:
	/** Cancels an active action. */
	void slotCancel();
	/** Activates the selected action. */
	void slotStart();
	/** Quits the application. */
	void slotQuit();
protected:
	/** Closes or minizes the main window. */
	virtual bool queryClose();
	/**
	 * Invoked at application exit.
	 * @return @c true
	 */
	virtual bool queryExit();
};

#endif // __MMAINWINDOW_H__
