/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#ifndef KFTPFILEDIRTREEVIEW_H
#define KFTPFILEDIRTREEVIEW_H

#include <tdelistview.h>
#include <tdelocale.h>
#include <kurl.h>

#include <tqpixmap.h>
#include <tqdict.h>

namespace KFTPWidgets {

namespace Browser {

class TreeViewItem;

/**
 * This class represents a tree view showing the directory structure. It
 * should usually be used together with a ListView.
 *
 * @author Jernej Kos
 */
class TreeView : public TDEListView
{
TQ_OBJECT
friend class ListView;
friend class TreeViewItem;
public:
    /**
     * Class constructor.
     *
     * @param parent The parent widget
     */
    TreeView(TQWidget *parent);
    
    /**
     * Class destructor.
     */
    ~TreeView();

    /**
     * Open an URL.
     *
     * @param url The url to open
     */
    int openUrl(const KURL &url, TQListViewItem *parent = 0);
    
    /**
     * Remove the folder's child items.
     *
     * @param url The folder's URL
     */
    void clearFolder(const KURL &url);
    
    /**
     * Remove the folder and it's child items.
     *
     * @param url The folder's URL
     */
    void removeFolder(const KURL &url);
    
    /**
     * Create a new folder identified by it's URL and any folders missing
     * in the hierarchy.
     *
     * @param url The folder's URL
     * @param icon The icon to use
     */
    void createFolder(const KURL &url, TQPixmap icon);
    
    /**
     * Clear all items in the view and create a new root item.
     *
     * @param url The root's new URL
     */
    void resetView(const KURL &url);
    
    /**
     * End item update and clear all non-dirty items.
     *
     * @param url The url to purge
     */
    void endUpdate(const KURL &url);
private:
    TQDict<TreeViewItem> m_treeItems;
    bool m_noItemOpen;
    
    /* Drag & drop support */
    TQListViewItem *m_dropItem;
    TQDragObject *m_dragObject;
    
    TQDragObject *dragObject();
    void startDrag();
    bool acceptDrag(TQDropEvent *e);
    void contentsDragEnterEvent(TQDragEnterEvent *e);
    void contentsDragMoveEvent(TQDragMoveEvent *e);
    void contentsDragLeaveEvent(TQDragLeaveEvent*);
    void contentsDropEvent(TQDropEvent *e);
protected:
    TQListViewItem *findItem(TQListViewItem *parent, const TQString &name);
private slots:
    void slotClicked(TQListViewItem *item);
    void slotDoubleClicked(TQListViewItem *item);
signals:
    void pathChanged(const KURL &url);
};

/**
 * This class represents an item in the tree view.
 *
 * @author Jernej Kos
 */
class TreeViewItem : public TQListViewItem
{
friend class TreeView;
public:
    /**
     * Class constructor.
     *
     * @param parent The parent tree view
     * @param url This item's URL
     */
    TreeViewItem(TreeView *parent, const KURL &url);
    
    /**
     * Class constructor.
     *
     * @param tree The tree view instance
     * @param parent The parent list view item
     * @param url This item's URL
     */
    TreeViewItem(TreeView *tree, TQListViewItem *parent, const KURL &url);
    
    /**
     * Class destructor.
     */
    ~TreeViewItem();
protected:
    /**
     * @overload
     * Reimplemented from TQListViewItem to provide proper comparation.
     */
    int compare(TQListViewItem *i, int col, bool) const;
private:
    TreeView *m_tree;
    KURL m_url;
    bool m_dirty;
};

}

}

#endif
