/***************************************************************************
*   Copyright (C) 2004-2009 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <tqprocess.h>
#include <tqwaitcondition.h>
#include <tqapplication.h>
#include <tqbuffer.h>

#include "fileexporterbibtex.h"
#include "settings.h"
#include "fileexporterexternal.h"

namespace BibTeX
{

    FileExporterExternal::FileExporterExternal( Exporter exporter, File::FileFormat fileformat )
            : FileExporter(), writeTo( NULL ), m_exporter( exporter ), m_fileformat( fileformat )
    {
        wc = new TQWaitCondition();
    }

    FileExporterExternal::~FileExporterExternal()
    {
        delete wc;
    }

    bool FileExporterExternal::save( TQIODevice* iodevice, const File* bibtexfile, TQStringList *errorLog )
    {
        m_mutex.lock();
        TQBuffer buffer;
        buffer.open( IO_WriteOnly );
        FileExporter * bibtexExporter = new FileExporterBibTeX();
        bool result = bibtexExporter->save( &buffer, bibtexfile, errorLog );
        buffer.close();
        delete bibtexExporter;

        if ( result )
            result = generateOutput( buffer, iodevice );

        m_mutex.unlock();
        return result;
    }

    bool FileExporterExternal::save( TQIODevice* iodevice, const Element* element, TQStringList *errorLog )
    {
        m_mutex.lock();
        TQBuffer buffer;
        buffer.open( IO_WriteOnly );
        FileExporter * bibtexExporter = new FileExporterBibTeX();
        bool result = bibtexExporter->save( &buffer, element, errorLog );
        buffer.close();
        delete bibtexExporter;

        if ( result )
            result = generateOutput( buffer, iodevice );

        m_mutex.unlock();
        return result;
    }

    bool FileExporterExternal::generateOutput( TQBuffer &input, TQIODevice *output )
    {
        bool result = FALSE;

        TQString commandLine;
        switch ( m_fileformat )
        {
        case File::formatHTML:
            switch ( m_exporter )
            {
            case exporterBibTeX2HTML:
                commandLine = "bibtex2html -s plain -a";
                break;
            case exporterBib2XHTML:
                commandLine = "bib2xhtml -s plain -u";
                break;
            case exporterBibConv:
                commandLine = "bibconv -informat=bibtex -outformat=html";
                break;
            default:
                return FALSE;
            }
            break;
        default:
            return FALSE;
        }
        TQStringList args = TQStringList::split( ' ', commandLine );

        writeTo = new TQTextStream( output );
        writeTo->setEncoding( TQTextStream::UnicodeUTF8 );

        TQApplication::setOverrideCursor( TQt::waitCursor );

        process = new TQProcess( args );
        connect( process, TQ_SIGNAL( processExited() ), this, TQ_SLOT( slotProcessExited() ) );
        connect( process, TQ_SIGNAL( readyReadStdout() ), this, TQ_SLOT( slotReadProcessOutput() ) );
        connect( process, TQ_SIGNAL( readyReadStderr() ), this, TQ_SLOT( slotReadProcessOutput() ) );
        connect( process, TQ_SIGNAL( wroteToStdin() ), this, TQ_SLOT( slotWroteToStdin() ) );

        if ( process->start() )
        {
            while ( !process->isRunning() )
            {
                wc->wait( 250 );
                tqApp->processEvents();
            }
            tqApp->processEvents();
            input.open( IO_ReadOnly );
            TQByteArray buf = input.buffer();
            //             tqDebug( "buf.size=%i", buf.size() );
            process->writeToStdin( buf );
            input.close();
            tqApp->processEvents();
            while ( process->isRunning() )
            {
                wc->wait( 250 );
                tqApp->processEvents();
            }

            result = process->normalExit();
        }

        disconnect( process, TQ_SIGNAL( wroteToStdin() ), this, TQ_SLOT( slotWroteToStdin() ) );
        disconnect( process, TQ_SIGNAL( readyReadStdout() ), this, TQ_SLOT( slotReadProcessOutput() ) );
        disconnect( process, TQ_SIGNAL( readyReadStderr() ), this, TQ_SLOT( slotReadProcessOutput() ) );
        disconnect( process, TQ_SIGNAL( processExited() ), this, TQ_SLOT( slotProcessExited() ) );
        delete( process );
        process = NULL;
        delete writeTo;
        writeTo = NULL;

        TQApplication::restoreOverrideCursor();
        return result;
    }

    void FileExporterExternal::slotProcessExited()
    {
        wc->wakeAll();
    }

    void FileExporterExternal::slotReadProcessOutput()
    {
        if ( writeTo != NULL )
            while ( process->canReadLineStdout() )
            {
                TQString line = process->readLineStdout();
                //                 tqDebug( "line=%s", line.latin1() );
                ( *writeTo ) << line.latin1() << endl;
            }
        //         while ( process->canReadLineStderr() )
        //         {
        //             TQString line = process->readLineStderr();
        //             tqDebug( "stderr=%s", line.latin1() );
        //         }
    }

    void FileExporterExternal::slotWroteToStdin()
    {
        process->closeStdin();
    }
}

#include "fileexporterexternal.moc"
