/*
Gwenview - A simple image viewer for TDE
Copyright 2000-2004 Aurlien Gteau

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

*/
#include <config.h>
#ifdef GV_HAVE_KIPI

// TQt
#include <tqdir.h>
#include <tqregexp.h>
#include <tqtimer.h>

// KDE
#include <kdebug.h>
#include <tdefileitem.h>
#include <tdelocale.h>
#include <kurl.h>

// KIPI
#include <libkipi/imagecollectionshared.h>
#include <libkipi/imageinfoshared.h>

// Local
#include "gvcore/archive.h"
#include "gvcore/cache.h"
#include "gvcore/fileviewbase.h"
#include "gvcore/fileviewcontroller.h"
#include "imageutils/jpegcontent.h"
#include "kipiinterface.moc"
namespace Gwenview {

#undef ENABLE_LOG
#undef LOG
//#define ENABLE_LOG
#ifdef ENABLE_LOG
#define LOG(x) kdDebug() << k_funcinfo << x << endl
#else
#define LOG(x) ;
#endif

class ImageCollection : public KIPI::ImageCollectionShared {
public:
	ImageCollection(KURL dirURL, const TQString& name, const KURL::List& images)
	: KIPI::ImageCollectionShared()
    , mDirURL(dirURL)
    , mName(name)
    , mImages(images) {}

	TQString name()           { return mName; }
	TQString comment()        { return TQString(); }
	KURL::List images()      { return mImages; }
	KURL uploadRoot()        { return KURL("/"); }
	KURL uploadPath()        { return mDirURL; }
	TQString uploadRootName() { return "/"; }
	bool isDirectory()       { return true; }

private:
    KURL mDirURL;
	TQString mName;
	KURL::List mImages;
};



class ImageInfo : public KIPI::ImageInfoShared {
	static const TQRegExp sExtensionRE;
public:
	ImageInfo(KIPI::Interface* interface, const KURL& url) : KIPI::ImageInfoShared(interface, url) {}

	TQString title() {
		TQString txt=_url.fileName();
		txt.replace("_", " ");
		txt.replace(sExtensionRE, "");
		return txt;
	}

	TQString description() {
		if (!_url.isLocalFile()) return TQString();

		ImageUtils::JPEGContent content;
		bool ok=content.load(_url.path());
		if (!ok) return TQString();

		return content.comment();
	}

	void setDescription(const TQString&) {}

	TQStringVariantMap attributes() {
		return TQStringVariantMap();
	}

	void clearAttributes() {}

	void addAttributes(const TQStringVariantMap&) {}
};

const TQRegExp ImageInfo::sExtensionRE("\\.[a-z0-9]+$", false /*caseSensitive*/);


struct KIPIInterfacePrivate {
	FileViewController* mFileView;
};


KIPIInterface::KIPIInterface( TQWidget* parent, FileViewController* fileView)
:KIPI::Interface(TQT_TQOBJECT(parent), "Gwenview kipi interface") {
	d=new KIPIInterfacePrivate;
	d->mFileView=fileView;

	connect(d->mFileView, TQT_SIGNAL(selectionChanged()),
		this, TQT_SLOT(slotSelectionChanged()) );

	connect(d->mFileView, TQT_SIGNAL(completed()),
		this, TQT_SLOT(slotDirectoryChanged()) );
// delay a bit, so that it's called after loadPlugins()
	TQTimer::singleShot( 0, this, TQT_SLOT( init()));
}


KIPIInterface::~KIPIInterface() {
	delete d;
}


void KIPIInterface::init() {
	slotDirectoryChanged();
	slotSelectionChanged();
}

KIPI::ImageCollection KIPIInterface::currentAlbum() {
	LOG("");
	KURL::List list;
	KFileItemListIterator it( *d->mFileView->currentFileView()->items() );
	for ( ; it.current(); ++it ) {
		KFileItem* item=it.current();
		if (!Archive::fileItemIsDirOrArchive(item)) {
			list.append(it.current()->url());
		}
	}
	KURL url=d->mFileView->dirURL();
	return KIPI::ImageCollection(new ImageCollection(url, url.fileName(), list));
}


KIPI::ImageCollection KIPIInterface::currentSelection() {
	LOG("");
	KURL::List list=d->mFileView->selectedImageURLs();
	KURL url=d->mFileView->dirURL();
	return KIPI::ImageCollection(new ImageCollection(url, i18n("%1 (Selected Images)").arg(url.fileName()), list));
}


TQValueList<KIPI::ImageCollection> KIPIInterface::allAlbums() {
	LOG("");
	TQValueList<KIPI::ImageCollection> list;
	list << currentAlbum() << currentSelection();
	return list;
}


KIPI::ImageInfo KIPIInterface::info(const KURL& url) {
	LOG("");
	return KIPI::ImageInfo( new ImageInfo(this, url) );
}

int KIPIInterface::features() const {
	return KIPI::AcceptNewImages;
}

/**
 * KDirLister will pick up the image if necessary, so no updating is needed
 * here, it is however necessary to discard caches if the plugin preserves timestamp
 */
bool KIPIInterface::addImage(const KURL& url, TQString&) {
	Cache::instance()->invalidate( url );
	return true;
}

void KIPIInterface::delImage(const KURL& url) {
	Cache::instance()->invalidate( url );
}

// TODO currently KDirWatch doesn't have watching of files in a directory
// implemented, so KDirLister will not inform when a file changes
void KIPIInterface::refreshImages( const KURL::List& urls ) {
	for( KURL::List::ConstIterator it = urls.begin();
	     it != urls.end();
	     ++it ) {
		Cache::instance()->invalidate( *it );
	}
	d->mFileView->refreshItems( urls );
}


void KIPIInterface::slotSelectionChanged() {
	emit selectionChanged(d->mFileView->selectionSize() > 0);
}


void KIPIInterface::slotDirectoryChanged() {
	emit currentAlbumChanged(d->mFileView->fileCount() > 0);
}


} // namespace

#endif /* GV_HAVE_KIPI */
