/*********
*
* This file is part of BibleTime's source code, http://www.bibletime.info/.
*
* Copyright 1999-2006 by the BibleTime developers.
* The BibleTime source code is licensed under the GNU General Public License version 2.0.
*
**********/



#include "cswordsetupmodulelistview.h"
#include "btinstallmgr.h"

#include "backend/cswordbackend.h"
#include "backend/cswordmoduleinfo.h"

#include "util/cresmgr.h"
#include "util/ctoolclass.h"

//QT includes
#include <tqtooltip.h>

//KDE includes
#include <tdelocale.h>
#include <kiconloader.h>

using namespace sword;

namespace BookshelfManager {

    /** Tooltip implementation for TQListView widgets.
     * @short Tooltip for InstallationManager listviews
     * @author Joachim Ansorg
     */
    class ToolTip : public TQToolTip {
    public:
        /** Constructor which takes the listview to operate on.
          * @param listview We operate on this widget to request tooltips from it'd child items.
         * @short Constructor.
         */
        ToolTip(CSwordSetupModuleListView* listview)
                : TQToolTip( listview->viewport() ),
        m_parent( listview ) {}

        /** Reimplementation of TQToolTip::maybeTip. It's requested if a new tooltip may be displayed.
         * @short Display a tooltip if we're over an item
         */
        virtual void maybeTip(const TQPoint& pos) {
            TQListViewItem* i = m_parent->itemAt(pos);
            Q_ASSERT(i);

            const TQRect rect = m_parent->itemRect(i);
            if (m_parent->showTooltip(i, pos, 0)) {
                const TQString tipText = m_parent->tooltip(i, 0);
                tip(rect, tipText);
            }
        }

    protected:
        CSwordSetupModuleListView* m_parent;
    };

    /** Listview specially made for the installation manager.
     * @short InstallationManager module listviews
     * @author Martin Gruner
     */
    CSwordSetupModuleListView::CSwordSetupModuleListView(TQWidget *parent, bool is_remote, sword::InstallSource* installSource)
            : TDEListView(parent), m_is_remote( is_remote ) {
        Q_ASSERT(installSource);
        new BookshelfManager::ToolTip(this);
        m_backend = installSource ? BTInstallMgr::Tool::backend(installSource) : CPointers::backend();

        addColumn(i18n("Name"));
        setColumnWidthMode( 0, TQListView::Maximum );
        setColumnWidth( 0, 200 ); //don`t get too broad

        addColumn(i18n("Status"));
        setColumnAlignment(1, TQt::AlignRight);
        addColumn(i18n("Installed version")); //version
        setColumnAlignment(2, TQt::AlignHCenter);

        if (m_is_remote) {
            addColumn(i18n("Remote version")); //version
        } else {
            addColumn(i18n("Location"));
        }
        setColumnAlignment(3, TQt::AlignLeft);

        setAllColumnsShowFocus(true);
        setFullWidth(true);
        setRootIsDecorated(true);
        setResizeMode(TQListView::LastColumn);
        setTooltipColumn(0);

        init();
    }

    CSwordSetupModuleListView::~CSwordSetupModuleListView() {
        if (m_is_remote) {
            delete m_backend;
        }
    }

    void CSwordSetupModuleListView::init() {
        m_categoryBible = new TQCheckListItem(this, i18n("Bibles"), TQCheckListItem::CheckBoxController);
        m_categoryCommentary = new TQCheckListItem(this, i18n("Commentaries"), TQCheckListItem::CheckBoxController);
        m_categoryLexicon = new TQCheckListItem(this, i18n("Lexicons"), TQCheckListItem::CheckBoxController);
        m_categoryBook = new TQCheckListItem(this, i18n("Books"), TQCheckListItem::CheckBoxController);
        m_categoryDevotionals = new TQCheckListItem(this, i18n("Daily Devotionals"), TQCheckListItem::CheckBoxController);
        m_categoryGlossaries = new TQCheckListItem(this, i18n("Glossaries"), TQCheckListItem::CheckBoxController);

        m_categoryBible->setPixmap(0, SmallIcon(CResMgr::mainIndex::closedFolder::icon, 16));
        m_categoryCommentary->setPixmap(0, SmallIcon(CResMgr::mainIndex::closedFolder::icon, 16));
        m_categoryLexicon->setPixmap(0, SmallIcon(CResMgr::mainIndex::closedFolder::icon, 16));
        m_categoryBook->setPixmap(0, SmallIcon(CResMgr::mainIndex::closedFolder::icon, 16));
        m_categoryDevotionals->setPixmap(0, SmallIcon(CResMgr::mainIndex::closedFolder::icon, 16));
        m_categoryGlossaries->setPixmap(0, SmallIcon(CResMgr::mainIndex::closedFolder::icon, 16));

        m_categoryBible->setOpen(true);
        m_categoryCommentary->setOpen(true);
        m_categoryLexicon->setOpen(true);
        m_categoryBook->setOpen(true);
        m_categoryDevotionals->setOpen(true);
        m_categoryGlossaries->setOpen(true);

        //   connect(this, TQT_SIGNAL(executed(TQListViewItem*)), TQT_SLOT(slotItemClicked(TQListViewItem*)));
        connect(this, TQT_SIGNAL(clicked(TQListViewItem*)), TQT_SLOT(slotItemClicked(TQListViewItem*))); //items have to be clicked only once in double click mode
        connect(this, TQT_SIGNAL(spacePressed(TQListViewItem*)), TQT_SLOT(slotItemClicked(TQListViewItem*)));
    }

    void CSwordSetupModuleListView::finish() {
        if (!m_categoryBible->childCount())
            delete m_categoryBible;
        if (!m_categoryCommentary->childCount())
            delete m_categoryCommentary;
        if (!m_categoryBook->childCount())
            delete m_categoryBook;
        if (!m_categoryLexicon->childCount())
            delete m_categoryLexicon;
        if (!m_categoryDevotionals->childCount())
            delete m_categoryDevotionals;
        if (!m_categoryGlossaries->childCount())
            delete m_categoryGlossaries;
    }

    void CSwordSetupModuleListView::clear() {
        TQListView::clear();
        init();
    }

    void CSwordSetupModuleListView::addModule(CSwordModuleInfo* module, TQString localVersion) {

        TQListViewItem* parent = 0;
        switch ( module->type() ) {
        case CSwordModuleInfo::Bible:
            parent = m_categoryBible;
            break;
        case CSwordModuleInfo::Commentary:
            parent = m_categoryCommentary;
            break;
        case CSwordModuleInfo::Lexicon:
            parent = m_categoryLexicon;
            break;
        case CSwordModuleInfo::GenericBook:
            parent = m_categoryBook;
            break;
        default:
            parent = 0;
            break;
        }

        //handling for special module types
        if ((parent == m_categoryLexicon) && (module->category() == CSwordModuleInfo::Glossary)) {
            parent = m_categoryGlossaries;
        }
        if ((parent == m_categoryLexicon) && (module->category() == CSwordModuleInfo::DailyDevotional)) {
            parent = m_categoryDevotionals;
        }

        //now we know the category, find the right language group in that category
        const CLanguageMgr::Language* const lang = module->language();
        TQString langName = lang->translatedName();
        if (!lang->isValid()) {
            langName = TQString(module->module()->Lang());
        }

        TQListViewItem * langFolder = 0;
        if (parent) {
            langFolder = parent->firstChild();

            while( langFolder ) { //try to find language folder if it exsists
                if (langFolder->text(0) == langName) { //found right folder
                    break;
                }

                langFolder = langFolder->nextSibling();
            }
        }

        if (!langFolder) { //not yet there
            langFolder = new TQCheckListItem(parent, langName, TQCheckListItem::CheckBoxController);

            langFolder->setPixmap(0, SmallIcon(CResMgr::mainIndex::closedFolder::icon, 16));
            langFolder->setOpen(false);
        }

        Q_ASSERT(langFolder);

        TQListViewItem* newItem = 0;
        if (langFolder) {
            newItem = new TQCheckListItem(langFolder, module->name(), TQCheckListItem::CheckBox);
        } else { //shouldn't happen
            newItem = new TQCheckListItem(this, module->name(), TQCheckListItem::CheckBox);
        }

        newItem->setPixmap(0, CToolClass::getIconForModule(module));
        if (m_is_remote) {
            newItem->setText(1, localVersion.isEmpty() ? i18n("New") : i18n("Updated"));
        } else {
            newItem->setText(1, i18n("Installed") );
        }

        newItem->setText(2, localVersion);
        if (m_is_remote) {
            newItem->setText(3, module->config(CSwordModuleInfo::ModuleVersion));
        } else {
            newItem->setText(3, module->config(CSwordModuleInfo::AbsoluteDataPath));
        }
    }

    TQStringList CSwordSetupModuleListView::selectedModules() {
        TQStringList moduleList;

        TQListViewItemIterator list_it( this );
        while ( list_it.current() ) {
            TQCheckListItem* i = dynamic_cast<TQCheckListItem*>( list_it.current() );
            if (i && i->isOn() && i->type() == TQCheckListItem::CheckBox ) {
                moduleList << i->text(0);
            }
            ++list_it;
        }

        return moduleList;
    }

    void CSwordSetupModuleListView::slotItemClicked(TQListViewItem*) {
        emit selectedModulesChanged();
    }

    bool CSwordSetupModuleListView::showTooltip(TQListViewItem* i, const TQPoint&, int) const {
        TQCheckListItem* checkItem = dynamic_cast<TQCheckListItem*>( i );
        Q_ASSERT(checkItem);

        return (checkItem && (checkItem->type() == TQCheckListItem::CheckBox));
    }

    TQString CSwordSetupModuleListView::tooltip(TQListViewItem* i, int /*column*/) const {
        TQString ret;
        TQCheckListItem* checkItem = dynamic_cast<TQCheckListItem*>( i );

        if (checkItem && (checkItem->type() == TQCheckListItem::CheckBox)) {
            const TQString moduleName = checkItem->text(0);
            CSwordModuleInfo* module = m_backend->findModuleByName(moduleName);

            ret = CToolClass::moduleToolTip(module);
        }

        return ret;
    }

} //NAMESPACE

#include "cswordsetupmodulelistview.moc"
